//                                               -*- C++ -*-
/**
 *  @file  Cache.cxx
 *  @brief Cache holds the already computed points to speed up calculations
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: Cache.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include "Cache.hxx"
#include "PersistentObjectFactory.hxx"
#include <iterator>
#include <cstdlib>
#include "Log.hxx"
#include "ResourceMap.hxx"


namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      using Common::Log;

      std::ostream & operator << ( std::ostream & os, const Cache::PairType & val)
      {
	os << val.first << "->" << val.second.first << "/" << val.second.second;
	return os;
      }

      CLASSNAMEINIT(Cache);

      static Common::Factory<Cache> RegisteredFactory("Cache");

      /* Default constructor */
      Cache::Cache()
	: Common::PersistentObject(),
	  enabled_(true),
	  maxSize_(std::atol( Common::ResourceMap::GetInstance().get("cache-max-size").c_str() )),
	  hits_(0),
	  points_()
      {
	// Nothing to do
      }

      /* Constructor with upper bound size */
      Cache::Cache(const UnsignedLong maxSize)
	: Common::PersistentObject(),
	  enabled_(true),
	  maxSize_(maxSize),
	  hits_(0),
	  points_()
      {
	// Nothing to do
      }

      /* Virtual constructor */
      Cache * Cache::clone() const
      {
	return new Cache(*this);
      }


      /* String converter */
      String Cache::str() const
      {
	OSS oss;
	oss << "class=" << Cache::GetClassName()
	    << " enabled=" << enabled_
	    << " name=" << getName()
	    << " maxSize=" << maxSize_
	    << " size=" << getSize()
	    << " hits=" << getHits()
	    << " points={" ;
	copy( points_.begin(), points_.end(), OSS_iterator<PairType>( oss, ", " ) );
	oss << "}" ;

	return oss;	
      }


      /* Enable or disable the cache */
      void Cache::enable() const
      {
	enabled_ = true;
      }

      void Cache::disable() const
      {
	enabled_ = false;
      }


      UnsignedLong Cache::getMaxSize() const
      {
	return maxSize_;
      }


      UnsignedLong Cache::getSize() const
      {
	return points_.size();
      }

      Bool Cache::isEnabled() const
      {
	return enabled_;
      }


      UnsignedLong Cache::getHits() const
      {
	return hits_;
      }


      /* Query the cache for the key presence */
      Bool Cache::hasKey(const NumericalPoint & key) const
      {
	if (! enabled_) return false;
	Bool found = ( points_.find( key.getCollection() ) != points_.end() );
	return found;
      }
      
      /* Retrieve the value from the cache with the key */
      const NumericalPoint Cache::find(const NumericalPoint & key) const
      {
	if (enabled_) {
	  std::map< KeyType, ValueType >::iterator it = points_.find( key.getCollection() );
	  Bool found = ( it != points_.end() );
	  if (found) {
	    ++(*it).second.second; // increment age
	    ++hits_;
	    Log::Info(OSS() << "Cache hit !");
	    return NumericalPoint( (*it).second.first );
	    
	  } else
	    return NumericalPoint();
	} else
	  return NumericalPoint();
      }
      
      /* Add a pair (key,value) to the cache. This may wipe out some older pair if maxSize is reached */
      void Cache::add(const NumericalPoint & key,
		      const NumericalPoint & value)
      {
	if (enabled_) insert( key.getCollection(), ValueType( value.getCollection(), 0 ) );
      }

      /* Insert a (key,value) pair in the cache */
      void Cache::insert( const KeyType & key,
			  const ValueType & value )
      {
	if (points_.size() == maxSize_) {
	  std::map< KeyType, ValueType >::iterator it = min_element( points_.begin(), points_.end(), OrderAccordingToAges() );
	  if (it != points_.end() ) points_.erase( it );
	}
	points_[key] = value;
      }


      /* Merge the contents of two caches */
      Cache & Cache::merge (const Cache & other)
      {
	if (enabled_) {
	  for_each( other.points_.begin(), other.points_.end(), addFunctor( this ) );
	}
	return *this;
      }

      /* Assignment operator */
      Cache & Cache::operator = (const Cache & other)
      {
	if (this != &other) {
	  Common::PersistentObject::operator=(other);
	  const_cast<UnsignedLong&>(maxSize_)        = other.maxSize_;
	  points_                                    = other.points_;
	  enabled_                                   = other.enabled_;
	  hits_                                      = other.hits_;
	}

	return *this;
      }


      /* Method save() stores the object through the StorageManager */
      void Cache::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
	adv.writeAttribute(StorageManager::SizeAttribute, points_.size() );
	adv.writeAttribute(StorageManager::MaxSizeAttribute,  maxSize_);

	const UnsignedLong size(points_.size());
	PersistentCollection< KeyType >      keyColl(size);
	keyColl.setName("keyColl");
	PersistentCollection< KeyType >      valueColl(size);
	valueColl.setName("valueColl");
	PersistentCollection< UnsignedLong > ageColl(size);
	ageColl.setName("ageColl");
	std::map< KeyType, ValueType >::const_iterator it = points_.begin();
	std::map< KeyType, ValueType >::const_iterator last = points_.end();
	for( UnsignedLong i = 0 ; it != last ; ++it, ++i) {
	  keyColl[i]   = (*it).first;
	  valueColl[i] = (*it).second.first;
	  ageColl[i]   = (*it).second.second;
	}
	adv.writeValue(keyColl, StorageManager::MemberNameAttribute, "keyColl");
	adv.writeValue(valueColl, StorageManager::MemberNameAttribute, "valueColl");
	adv.writeValue(ageColl, StorageManager::MemberNameAttribute, "ageColl");
      }

      /* Method load() reloads the object from the StorageManager */
      void Cache::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
	UnsignedLong size;
	adv.readAttribute(StorageManager::SizeAttribute, size );
	adv.readAttribute(StorageManager::MaxSizeAttribute, const_cast<UnsignedLong&>(maxSize_) );

	PersistentCollection< KeyType >      keyColl(size);
	PersistentCollection< KeyType >      valueColl(size);
	PersistentCollection< UnsignedLong > ageColl(size);
	adv.readValue(keyColl, StorageManager::MemberNameAttribute, "keyColl");
	adv.readValue(valueColl, StorageManager::MemberNameAttribute, "valueColl");
	adv.readValue(ageColl, StorageManager::MemberNameAttribute, "ageColl");
	points_.erase( points_.begin(), points_.end() );
	for( UnsignedLong i = 0; i < size; ++i) points_[ keyColl[i] ] = ValueType( valueColl[i], ageColl[i] );
      }

      /* These methods are implemented here for the needs of Cache */
      /* We should be careful because they may interfere with other definitions placed elsewhere */
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<UnsignedLong>);
      static Common::Factory<PersistentCollection<UnsignedLong> > RegisteredFactory_alt1("PersistentCollection<UnsignedLong>");
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<PersistentCollection<NumericalScalar> >);
      static Common::Factory<PersistentCollection<PersistentCollection<NumericalScalar> > > RegisteredFactory_alt2("PersistentCollection<PersistentCollection<NumericalScalar> >");


    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
