//                                               -*- C++ -*-
/**
 *  @file  Cache.hxx
 *  @brief Cache holds the already computed points to speed up calculations
 *
 *  (C) Copyright 2005-2008 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: Cache.hxx 995 2008-10-31 10:52:04Z dutka $
 */
#ifndef OPENTURNS_CACHE_HXX
#define OPENTURNS_CACHE_HXX

#include <map>
#include <sstream>
#include "PersistentObject.hxx"
#include "StorageManager.hxx"
#include "Collection.hxx"
#include "NumericalPoint.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {
      /**
       * @class Cache
       *
       * Cache holds the already computed points to speed up calculations
       * Like any cache system, this objet saves time by keeping some costly data
       * and provides access to them through a key. 
       * Here the key is the input NumericalPoint of a NumericalMathFunction and the
       * value is the output NumericalPoint that was computed.
       * When used with ComputedNumericalMathFunctionImplementation objects the Cache
       * may greatly speed up the computations of Analytical algorithms (like FORM
       * or SORM). Its advantage is more doubtful with other NumericalMathFunctionImplementations
       * or Simulation algorithms.
       *
       * The Cache may be set with a maximum size to avoid too much memory consumption especially
       * when running Simulation algorithms (Monte-Carlo or LHS). This maximum size is
       * an upper bound for the number of data kept by the Cache. When this upper bound is 
       * reached, the next data insertion into the Cache flushes the least recently used
       * data before the insertion.
       */
      
      class Cache
	: public Common::PersistentObject
      {
	CLASSNAME;
	friend class Common::Factory<Cache>;

      public:

	typedef PersistentCollection< NumericalScalar>                            KeyType;
	typedef std::pair< PersistentCollection< NumericalScalar>, UnsignedLong > ValueType;
	typedef std::pair< KeyType, ValueType >                                   PairType;
	typedef Common::StorageManager                                            StorageManager;


	/** Default constructor */
	Cache();

	/** Constructor with upper bound size */
	Cache(const UnsignedLong maxSize);

	/** Virtual constructor */
	virtual Cache * clone() const;

	/** String converter */
	virtual String str() const;

#ifndef SWIG
	/** Assignment operator */
	Cache & operator = (const Cache & other);
#endif

	/** Merge the contents of two caches */
	Cache & merge (const Cache & other);

	/** Returns the number of successful hits in the cache */
	UnsignedLong getHits() const;

	/** Query the cache for the key presence */
	Bool hasKey(const NumericalPoint & key) const;

	/** Retrieve the value from the cache with the key */
	const NumericalPoint find(const NumericalPoint & key) const;

	/** Add a pair (key,value) to the cache. This may wipe out some older pair if maxSize is reached */
	void add(const NumericalPoint & key,
		 const NumericalPoint & value);

 	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);

	/** Accessors */
	UnsignedLong getSize() const;
	UnsignedLong getMaxSize() const;
	Bool isEnabled() const;

	/** Enable or disable the cache */
	void enable() const;
	void disable() const;

      private:

	/* Used in sort algorithm to find the Least Recently Used item.
	 * This structure implements the BynaryPredicate concept of the STL.
	 */
	struct OrderAccordingToAges {
	  bool operator() (const PairType & a,
			   const PairType & b)
	  {
	    return a.second.second <= b.second.second ;
	  }
	};

	/* Used to insert elements into the cache */
	struct addFunctor : public std::unary_function< Cache, void >
	{
	  Cache * p_cache_;
	  addFunctor( Cache * p_cache ) : p_cache_(p_cache) {}
	  void operator() ( const std::map< Cache::KeyType, Cache::ValueType >::value_type & val ) { p_cache_->insert( val.first, val.second ); }
	};

      
	/* Insert a (key,value) pair in the cache */
	void insert( const KeyType & key, const ValueType & value );

      protected:

	/** True if cache is enabled */
	mutable Bool enabled_;

	/** Upper bound for the cache size */
	const UnsignedLong maxSize_;

	/** Number of hits */
	mutable UnsignedLong hits_;

	/** The map of elements */
	mutable std::map< KeyType, ValueType > points_;

      }; /* class Cache */




    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_CACHE_HXX */
