//                                               -*- C++ -*-
/**
 *  @file  Interval.hxx
 *  @brief Interval is defined as the cartesian product of n 1D intervalls ]low_1, up_1]x...x]low_n,up_n]
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: Interval.hxx 995 2008-10-31 10:52:04Z dutka $
 */
#ifndef OPENTURNS_INTERVAL_HXX
#define OPENTURNS_INTERVAL_HXX

#include "PersistentObject.hxx"
#include "Exception.hxx"
#include "Collection.hxx"
#include "PersistentCollection.hxx"
#include "NumericalPoint.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Type {

      /**
       * @class Interval
       *
       * A class that holds a collection of interval
       */
      class Interval
	: public Common::PersistentObject
      {
	CLASSNAME;

      public:
	typedef Type::Collection<UnsignedLong>           BoolCollection;
	typedef Type::PersistentCollection<UnsignedLong> BoolPersistentCollection;
	typedef Common::InvalidArgumentException         InvalidArgumentException;
	typedef Common::PersistentObject                 PersistentObject;
        typedef Base::Common::StorageManager             StorageManager;

	/** Default constructor */
	explicit Interval(const UnsignedLong dimension = 1);

	/** Parameters constructor, simplified for 1D case */
	Interval(const NumericalScalar lowerBound,
		 const NumericalScalar upperBound);

	/** Parameters constructor */
	Interval(const NumericalPoint & lowerBound,
		 const NumericalPoint & upperBound)
	  throw(InvalidArgumentException);


	/** Parameters constructor */
	Interval(const NumericalPoint & lowerBound,
		 const NumericalPoint & upperBound,
		 const BoolCollection & finiteLowerBound,
		 const BoolCollection & finiteUpperBound)
	  throw(InvalidArgumentException);

	/** Virtual constructor method */
	Interval * clone() const;
	
	/** Check if the interval is empty, i.e. if we have lowerBound >= upperBound for at least one component */
	Bool isEmpty() const;

	/** Check if the given point is inside of the closed interval */
	Bool isInside(const NumericalPoint & point) const;

	/** Returns the interval equals to the intersection between the interval and another one */
	Interval intersect(const Interval & other) const;

	/** Returns the interval equals to the union between the interval and another one */
	Interval join(const Interval & other) const;

        /** Addition operator */
        Interval operator +(const Interval & rhs) const
          throw (InvalidArgumentException);

        /** Substraction operator */
        Interval operator -(const Interval & rhs) const
          throw (InvalidArgumentException);

        /** In-place addition operator */
        Interval & operator +=(const Interval & other) throw (InvalidArgumentException);

        /** In-place substraction operator */
        Interval & operator -=(const Interval & other) throw (InvalidArgumentException);

        /** Product operator */
        Interval operator *(const NumericalScalar scalar) const;

        /**  In-place product operator */
        Interval & operator *=(const NumericalScalar scalar);


	/** Dimension accessor */
	UnsignedLong getDimension() const;

	/** Lower bound accessor */
	NumericalPoint getLowerBound() const;
	void setLowerBound(const NumericalPoint & lowerBound);

	/** Upper bound accessor */
	NumericalPoint getUpperBound() const;
	void setUpperBound(const NumericalPoint & upperBound);

	/** Lower bound flag accessor */
	BoolCollection getFiniteLowerBound() const;
	void setFiniteLowerBound(const BoolCollection & finiteLowerBound);

	/** Upper bound flag accessor */
	BoolCollection getFiniteUpperBound() const;
	void setFiniteUpperBound(const BoolCollection & finiteUpperBound);

	/** String converter */
	String str() const;

        /** Method save() stores the object through the StorageManager */
        virtual void save(const StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        virtual void load(const StorageManager::Advocate & adv);

      private:
	// Dimension of the interval
	UnsignedLong dimension_;

	// An n-D intervall is defined as the cartesian product of n 1D intervalls ]low_1, up_1]x...x]low_n,up_n]
	NumericalPoint lowerBound_;
	NumericalPoint upperBound_;

	// Flags that tell if the bounds are finite or not.
	// The bound is finite if the flag is true, else it is equal to -inf if the associated value is < 0, +inf else
	BoolPersistentCollection finiteLowerBound_;
	BoolPersistentCollection finiteUpperBound_;

      }; /* class Interval */

    } /* Namespace Type */
  } /* Namespace Base */
} /* Namespace OpenTURNS */

#endif /* OPENTURNS_INTERVAL_HXX */
