//                                               -*- C++ -*-
/**
 *  @file  PersistentCollection.hxx
 *  @brief PersistentCollection defines top-most collection strategies
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: PersistentCollection.hxx 995 2008-10-31 10:52:04Z dutka $
 */
#ifndef OPENTURNS_PERSISTENTCOLLECTION_HXX
#define OPENTURNS_PERSISTENTCOLLECTION_HXX

#include "PersistentObject.hxx"
#include "StorageManager.hxx"
#include "Collection.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {
      /**
       * @class PersistentCollection
       *
       * PersistentCollection defines top-most collection strategies
       */
      
      /* Forward declaration */
      template <class T> class PersistentCollection;


      /** Stream operator */
      template <class T> inline
      std::ostream & operator << (std::ostream & os, const PersistentCollection<T> & collection)
      {
	return os << collection.str();
      }


      /** Entity trait */
      template <class T>
      struct EntityTrait
      {
	inline static Common::StorageManager::EntityName GetEntity() { return Common::StorageManager::ObjectEntity; };
      }; /* end struct EntityTrait */

      /** Entity trait specializations */
      template <> inline
      Common::StorageManager::EntityName EntityTrait<NumericalScalar>::GetEntity()
      {
	return Common::StorageManager::NumericalScalarEntity;
      }

      template <> inline
      Common::StorageManager::EntityName EntityTrait<UnsignedLong>::GetEntity()
      {
	return Common::StorageManager::UnsignedLongEntity;
      }

      template <> inline
      Common::StorageManager::EntityName EntityTrait<String>::GetEntity()
      {
	return Common::StorageManager::StringEntity;
      }

      template <> inline
      Common::StorageManager::EntityName EntityTrait<Bool>::GetEntity()
      {
	return Common::StorageManager::BoolEntity;
      }






      template <class T>
      class PersistentCollection
	: public Common::PersistentObject,
	  public Collection<T>
      {
	CLASSNAME;

      public:

	typedef Collection<T>                                 InternalType;
	typedef typename InternalType::ElementType            ElementType;
	typedef typename InternalType::ValueType              ValueType;
	typedef typename InternalType::iterator               iterator;
	typedef typename InternalType::const_iterator         const_iterator;
	typedef typename InternalType::reverse_iterator       reverse_iterator;
	typedef typename InternalType::const_reverse_iterator const_reverse_iterator;

	typedef Common::StorageManager                        StorageManager;


	/** Default constructor */
	PersistentCollection()
	  : Common::PersistentObject(),
	    Collection<T>()
	{
	  // Nothing to do
	}

	/** Constructor from a collection */
	PersistentCollection(const Collection<T> & collection)
	  : Common::PersistentObject(),
	    Collection<T>(collection)
	{
	  // Nothing to do
	}

	/** Constructor that pre-allocate size elements */
	PersistentCollection(const UnsignedLong size)
	  : Common::PersistentObject(),
	    Collection<T>(size)
	{
	  // Nothing to do
	}

	/** Constructor that pre-allocate size elements with value */
	PersistentCollection(const UnsignedLong size,
			     const T & value)
	  : Common::PersistentObject(),
	    Collection<T>(size, value)
	{
	  // Nothing to do
	}

	/** Constructor from a range of elements */
	template <typename InputIterator>
	PersistentCollection(InputIterator first,
			     InputIterator last)
	  : Common::PersistentObject(),
	    Collection<T>(first, last)
	{
	  // Nothing to do
	}


	/** Virtual constructor */
	virtual PersistentCollection * clone() const
	{
	  return new PersistentCollection(*this);
	}

	/** String converter */
	inline
	String str() const
	{
	  return Collection<T>::str();
	}

	/** Method save() stores the object through the StorageManager */
	inline
	void save(const StorageManager::Advocate & adv) const
	{
	  PersistentObject::save(adv);
	  adv.writeAttribute(StorageManager::SizeAttribute, Collection<T>::coll__.size());
	  for(UnsignedLong i=0; i<Collection<T>::coll__.size(); ++i) adv.writeValue(i, Collection<T>::coll__[i]);
	}

	/** Method load() reloads the object from the StorageManager */
	inline
	void load(const StorageManager::Advocate & adv)
	{
	  PersistentObject::load(adv);

	  UnsignedLong size;
	  adv.readAttribute(StorageManager::SizeAttribute, size);
	  Collection<T>::coll__.resize(size);

	  StorageManager::List objList = adv.getList(EntityTrait<T>::GetEntity());
	  ElementType value;
	  UnsignedLong index = 0;
	  for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	    if (objList.readValue(index, value)) Collection<T>::coll__[index] = value;
	  }
	}


      }; /* class PersistentCollection */


      /** Comparison operator */
      template <class T> inline
      Bool operator == (const PersistentCollection<T> & lhs,
			const PersistentCollection<T> & rhs)
      {
	return static_cast<const Collection<T>& >(lhs) == static_cast<const Collection<T>& >(rhs);
      }


      /** Ordering operator */
      template <class T> inline
      Bool operator < (const PersistentCollection<T> & lhs,
		       const PersistentCollection<T> & rhs)
      {
	return static_cast<const Collection<T>& >(lhs) < static_cast<const Collection<T>& >(rhs);
      }


    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_PERSISTENTCOLLECTION_HXX */
