//                                               -*- C++ -*-
/**
 *  @file  Analytical.cxx
 *  @brief Analytical implements an algorithm to find the design point
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-22 11:34:11 +0200 (lun 22 sep 2008) $
 *  Id:      $Id: Analytical.cxx 941 2008-09-22 09:34:11Z dutka $
 */
#include <cmath>

#include "Analytical.hxx"
#include "Distribution.hxx"
#include "ComparisonOperatorImplementation.hxx"

namespace OpenTURNS
{

  namespace Uncertainty

  {

    namespace Algorithm
    {

      CLASSNAMEINIT(Analytical);

      /*
       * @brief  Standard constructor: the class is defined by an optimisation algorithm, a failure event and a physical starting point
       */
      Analytical::Analytical(const NearestPointAlgorithm & nearestPointAlgorithm,
			     const Event & event,
			     const NumericalPoint & physicalStartingPoint)
	: Base::Common::PersistentObject(),
	  Base::Common::Threadable(),
	  nearestPointAlgorithm_(nearestPointAlgorithm),
	  event_(event),
	  physicalStartingPoint_(physicalStartingPoint),
	  result_(event_.getImplementation()->getAntecedent()->getDistribution().getIsoProbabilisticTransformation().operator()(physicalStartingPoint_), event, true)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      Analytical * Analytical::clone() const
      {
	return new Analytical(*this);
      }

      /* Destructor */
      Analytical::~Analytical()
      {
	// Nothing to do
      }

      /* Physical starting point accessor */
      Analytical::NumericalPoint Analytical::getPhysicalStartingPoint() const
      {
	return physicalStartingPoint_;
      }

      /* Physical starting point accessor */
      void Analytical::setPhysicalStartingPoint(const NumericalPoint & physicalStartingPoint)
      {
	physicalStartingPoint_ = physicalStartingPoint;
      }

      /* Event accessor */
      Analytical::Event Analytical::getEvent() const
      {
	return event_;
      }

      /* Event accessor */
      void Analytical::setEvent(const Event & event)
      {
	event_ =  event;
      }

      /* NearestPointAlgorithm accessor */
      Analytical::NearestPointAlgorithm Analytical::getNearestPointAlgorithm() const
      {
	return nearestPointAlgorithm_;
      }

      /* NearestPointAlgorithm accessor */
      void Analytical::setNearestPointAlgorithm(const NearestPointAlgorithm & nearestPointAlgorithm)
      {
	nearestPointAlgorithm_ = nearestPointAlgorithm;
      }

      /* String converter */
      String Analytical::str() const
      {
	OSS oss;
	oss << "class=" << Analytical::GetClassName()
	    << " nearestPointAlgorithm=" << nearestPointAlgorithm_.str()
	    << " event=" << event_
	    << " physicalstartingPoint=" << physicalStartingPoint_;
	return oss;
      }

      /* Performs the actual computation. */
      void Analytical::run()
      {
	/* Build a standard event from the event */
	StandardEvent standardEvent(event_);

	/* set the level function of the algorithm */
 	nearestPointAlgorithm_.setLevelFunction(standardEvent.getImplementation()->getFunction());
  
	/* set the level value of the algorithm in the standard space  */
	nearestPointAlgorithm_.setLevelValue(event_.getThreshold());

	/* set the starting point of the algorithm in the standard space  */
	nearestPointAlgorithm_.setStartingPoint(event_.getImplementation()->getAntecedent()->getDistribution().getIsoProbabilisticTransformation().operator()(physicalStartingPoint_));

	/* solve the nearest point problem */
	nearestPointAlgorithm_.run();

	/* set standard space design point in Result */
	NumericalPoint standardSpaceDesignPoint(nearestPointAlgorithm_.getResult().getMinimizer());
	standardSpaceDesignPoint.setName("Standard Space Design Point");
	result_.setStandardSpaceDesignPoint(standardSpaceDesignPoint);

	/* set isStandardPointOriginInFailureSpace in Result */
	NumericalPoint origin(standardSpaceDesignPoint.getDimension(), 0.0);
	NumericalPoint value(standardEvent.getImplementation()->getFunction().operator()(origin));

	result_.setIsStandardPointOriginInFailureSpace(event_.getOperator().compare(value[0], event_.getThreshold()));
      } /* Analytical::run() */

      /* Result accessor */
      AnalyticalResult Analytical::getAnalyticalResult() const
      {
	return result_;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

