//                                               -*- C++ -*-
/**
 *  @file  AnalyticalResult.cxx
 *  @brief Result computes the results of an analytical Algorithm
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: AnalyticalResult.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <iomanip>

#include "Analytical.hxx"
#include "Distribution.hxx"
#include "Pie.hxx"
#include "NumericalSample.hxx"
#include "NumericalMathFunction.hxx"
#include "BarPlot.hxx"
#include "Description.hxx"
#include "CompositeRandomVector.hxx"
#include "ConstantRandomVector.hxx"
#include "RandomVector.hxx"
#include "Less.hxx"
#include "PersistentObjectFactory.hxx"
#include "ComparisonOperatorImplementation.hxx"


namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {
      typedef Base::Type::Description                             Description;
      typedef Base::Stat::NumericalSample                         NumericalSample;

      typedef Base::Graph::Pie                                    Pie;
      typedef Base::Graph::BarPlot                                BarPlot;
      typedef Base::Graph::DrawableImplementation                 DrawableImplementation;
      typedef DrawableImplementation::Description                 Description;
      typedef Base::Func::NumericalMathFunction                   NumericalMathFunction;
      typedef NumericalMathFunction::Matrix                       Matrix;
      typedef Base::Common::Less                                  Less;
      typedef Model::RandomVector                                 RandomVector;
      typedef Model::ConstantRandomVector           ConstantRandomVector;
      typedef Model::CompositeRandomVector          CompositeRandomVector;
      typedef Model::Distribution                                 Distribution;
      typedef Distribution::IsoProbabilisticTransformation        IsoProbabilisticTransformation;
      typedef Distribution::InverseIsoProbabilisticTransformation InverseIsoProbabilisticTransformation;
      typedef Base::Type::PersistentCollection<Base::Type::NumericalPointWithDescription> PersistentSensitivity;

      const NumericalScalar AnalyticalResult::DefaultWidth;

      CLASSNAMEINIT(AnalyticalResult);

      static Base::Common::Factory<AnalyticalResult> RegisteredFactory("AnalyticalResult");

      /*
       * @brief  Standard constructor
       */
      AnalyticalResult::AnalyticalResult(const NumericalPoint & standardSpaceDesignPoint,
					 const Event & limitStateVariable,
					 const Bool isStandardPointOriginInFailureSpace,
					 const String & name):
	PersistentObject(name),
        physicalSpaceDesignPoint_(standardSpaceDesignPoint.getDimension()),
	limitStateVariable_(limitStateVariable),
	isStandardPointOriginInFailureSpace_(isStandardPointOriginInFailureSpace),
	hasoferReliabilityIndex_(0.0),
	importanceFactors_(standardSpaceDesignPoint.getDimension()),
	hasoferReliabilityIndexSensitivity_(0),
	isAlreadyComputedImportanceFactors_(false),
	isAlreadyComputedHasoferReliabilityIndexSensitivity_(false)
      {
	/* we use the attribute accessor in order to compute automatically all the other attributes */
	setStandardSpaceDesignPoint(standardSpaceDesignPoint);
      }

      /* Default constructor, needed by SWIG */
      AnalyticalResult::AnalyticalResult():
	PersistentObject(),
	standardSpaceDesignPoint_(0),
	physicalSpaceDesignPoint_(0),
	// Fake event based on a fake 1D composite random vector, which requires a fake 1D NumericalMathFunction
	limitStateVariable_(RandomVector(new CompositeRandomVector(NumericalMathFunction(Description(0),Description(1),Description(1)),
										 new ConstantRandomVector(NumericalPoint(0)))),
			    Less(), 0.0),
	isStandardPointOriginInFailureSpace_(false),
	hasoferReliabilityIndex_(0.0),
	importanceFactors_(0),
	hasoferReliabilityIndexSensitivity_(0),
	isAlreadyComputedImportanceFactors_(false),
	isAlreadyComputedHasoferReliabilityIndexSensitivity_(false)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      AnalyticalResult * AnalyticalResult::clone() const
      {
	return new AnalyticalResult(*this);
      }

      /* StandardSpaceDesignPoint accessor */
      AnalyticalResult::NumericalPoint AnalyticalResult::getStandardSpaceDesignPoint() const
      {
	return standardSpaceDesignPoint_;
      }


      /* StandardSpaceDesignPoint accessor */
      void AnalyticalResult::setStandardSpaceDesignPoint(const NumericalPoint & standardSpaceDesignPoint)
      {
	standardSpaceDesignPoint_ = standardSpaceDesignPoint;
	computePhysicalSpaceDesignPoint();
	computeHasoferReliabilityIndex();
      }

      /* PhysicalSpaceDesignPoint evaluation */
      void AnalyticalResult::computePhysicalSpaceDesignPoint()
      {
	/* Compute the physical design point */
	physicalSpaceDesignPoint_ = limitStateVariable_.getImplementation()->getAntecedent()->getDistribution().getInverseIsoProbabilisticTransformation().operator()(standardSpaceDesignPoint_);
	/* we give to the physical space design point the description of the input random vector */
	//physicalSpaceDesignPoint_.setDescription(limitStateVariable_.getImplementation()->getAntecedent()->getDescription());

	/* we give to the physical space design point the name Physical Design Point */
	physicalSpaceDesignPoint_.setName("Physical Space Design Point");
      }

      /* PhysicalSpaceDesignPoint accessor */
      void AnalyticalResult::setPhysicalSpaceDesignPoint(const NumericalPoint & physicalSpaceDesignPoint)
      {
	physicalSpaceDesignPoint_ = physicalSpaceDesignPoint;
      }

      /* PhysicalSpaceDesignPoint accessor */
      AnalyticalResult::NumericalPoint AnalyticalResult::getPhysicalSpaceDesignPoint() const
      {
	return physicalSpaceDesignPoint_;
      }

      /* LimitStateVariable accessor */
      AnalyticalResult::Event AnalyticalResult::getLimitStateVariable() const
      {
	return limitStateVariable_;
      }

      /* IsStandardPointOriginInFailureSpace  accessor */
      Bool AnalyticalResult::getIsStandardPointOriginInFailureSpace() const
      {
	return isStandardPointOriginInFailureSpace_;
      }

      /* IsStandardPointOriginInFailureSpace  accessor */
      void AnalyticalResult::setIsStandardPointOriginInFailureSpace(const Bool isStandardPointOriginInFailureSpace)
      {
	isStandardPointOriginInFailureSpace_ = isStandardPointOriginInFailureSpace;
      }

      /* ImportanceFactors evaluation */
      void AnalyticalResult::computeImportanceFactors()
      {
	const UnsignedLong dimension(standardSpaceDesignPoint_.getDimension());
	importanceFactors_ = NumericalPoint(dimension, -1.0);
	/* First, check that the importance factors are well-defined */
	if (standardSpaceDesignPoint_.norm() > 0.0)
	  {
	    /* get the input distribution */
	    const Distribution inputDistribution(limitStateVariable_.getImplementation()->getAntecedent()->getDistribution());
	    /* get the input standard distribution */
	    const Distribution standardDistribution(inputDistribution.getStandardDistribution());
	    /* get the marginal 1D from in the standard space where all marginals are identical */
	    const Distribution standardMarginalDistribution(standardDistribution.getMarginal(0));

	    /* evaluate the corresponding vector in the importance factors space */
	    /* if Generalised Nataf Transformation : Z = E1^(-1)o F_{Xi}(Xi*) */
	    /* if Rosenblatt  Transformation : Z = Phi^(-1)o F_{Xi}(Xi*) */
	    // for each marginals */
	    for (UnsignedLong marginalIndex = 0; marginalIndex < dimension; ++marginalIndex)
	      {
		const NumericalScalar y(inputDistribution.getMarginal(marginalIndex).computeCDF(NumericalPoint(1, physicalSpaceDesignPoint_[marginalIndex])));
		importanceFactors_[marginalIndex] = standardMarginalDistribution.computeQuantile(y)[0];
	      }
	    /* compute  square norm of importanceFactors vector */
	    const NumericalScalar inorm2(1.0 / importanceFactors_.norm2());
	    /* compute final importance factors */
	    for (UnsignedLong marginalIndex = 0; marginalIndex < dimension; ++marginalIndex)
	      {
		importanceFactors_[marginalIndex] *= importanceFactors_[marginalIndex] * inorm2;
	      }
	  }
	/* we give a name to the importance factors vector */
	importanceFactors_.setName("Importance Factors");

	/* we give to the importance factors vector the description of the input random vector */
	importanceFactors_.setDescription(limitStateVariable_.getImplementation()->getAntecedent()->getDescription());

	isAlreadyComputedImportanceFactors_ = true;
      }

      /* ImportanceFactors accessor */
      AnalyticalResult::NumericalPoint AnalyticalResult::getImportanceFactors()
      {
	if (!isAlreadyComputedImportanceFactors_)
	  {
	    computeImportanceFactors();
	  }
	return importanceFactors_;
      }

      /* ImportanceFactors graph */
      AnalyticalResult::Graph AnalyticalResult::drawImportanceFactors()
      {
	// To ensure that the importance factors are up to date
	if (!isAlreadyComputedImportanceFactors_)
	  {
	    computeImportanceFactors();
	  }
	const UnsignedLong dimension(importanceFactors_.getDimension());
	NumericalSample data(dimension, 1);

	/* build data for the pie */
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    data[i]=NumericalPoint(1, importanceFactors_[i]);
	  }
	/* we build the pie */
	Pie importanceFactorsPie(data);

	/* build labels and colors for the pie */
	Description palette(dimension);
	Description labels(dimension);
	const Description colors(DrawableImplementation::GetValidColors());
	Description::const_iterator colorsIterator;
	colorsIterator = colors.begin();
	Description description(importanceFactors_.getDescription());
	// If no description has been given for the input distribution components, give standard ones
	if (description.getSize() != dimension)
	  {
	    description = Description(dimension);
	    for (UnsignedLong i = 0; i < dimension; ++i)
	      {
		OSS oss;
		oss << "Component " << i;
		description[i] = oss;
	      }
	  }
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    OSS oss;
	    oss << description[i] << " : " << std::fixed;
	    oss.setPrecision(1);
	    oss << 100.0 * data[i][0] << "%";
	    labels[i] = oss;
	    palette[i] = (*colorsIterator);
	    ++colorsIterator;
	    if (colorsIterator == colors.end())
	      {
		colorsIterator == colors.begin();
	      }
	  }
	/* we complete the pie */
	importanceFactorsPie.setLabels(labels);
	importanceFactorsPie.setPalette(palette);

	/* we build the graph with a title */
	OSS oss;
	oss << "Importance Factors from Design Point - " << limitStateVariable_.getName();
	Graph importanceFactorsGraph(oss, "", "", false,"");

	/* we embed the pie into the graph */
	importanceFactorsGraph.addDrawable(importanceFactorsPie);
	return importanceFactorsGraph;
      }


      /* ImportanceFactors accessor */
      void AnalyticalResult::setImportanceFactors(const NumericalPoint & importanceFactors)
      {
	importanceFactors_ = importanceFactors;
	isAlreadyComputedImportanceFactors_ = true;
      }


      /* HasoferReliabilityIndex evaluation */
      void AnalyticalResult::computeHasoferReliabilityIndex()
      {
	/* evaluate the HasoferReliabilityIndex */
	if (standardSpaceDesignPoint_.getDimension() > 0) hasoferReliabilityIndex_ = standardSpaceDesignPoint_.norm();
      }

      /*  HasoferReliabilityIndex accessor */
      NumericalScalar AnalyticalResult::getHasoferReliabilityIndex() const
      {
	return hasoferReliabilityIndex_;
      }


      /* HasoferReliabilityIndex accessor */
      void AnalyticalResult::setHasoferReliabilityIndex(const NumericalScalar & hasoferReliabilityIndex)
      {
	hasoferReliabilityIndex_ = hasoferReliabilityIndex;
      }

      /* HasoferReliabilityIndex evaluation */
      void AnalyticalResult::computeHasoferReliabilityIndexSensitivity()
      {
	/* get Set1 : parameters of the physical distribution */
	const Distribution physicalDistribution(limitStateVariable_.getImplementation()->getAntecedent()->getDistribution());
	const NumericalPointWithDescriptionCollection set1(physicalDistribution.getParametersCollection());
	/* get Set2 : parameters of the physical model */
	const NumericalMathFunction physicalModel(limitStateVariable_.getImplementation()->getFunction());
	const NumericalPointWithDescription set2(physicalModel.getParameters());
	const Bool isSet2Empty(set2.getDimension() == 0);

	/* get SetIso : parameters included in the isoprobabilistic transformation which is a subset of Set1 */
	const InverseIsoProbabilisticTransformation inverseIsoProbabilisticTransformation(physicalDistribution.getInverseIsoProbabilisticTransformation());
	const NumericalPointWithDescription setIso(inverseIsoProbabilisticTransformation.getParameters());
	/* scaling factor between sensitivity and gradients (ref doc : factor = -lambda/beta) */
   	/* gradient of the standard limite state function */
	const Matrix physicalGradientMatrix(physicalModel.gradient(physicalSpaceDesignPoint_));
	const NumericalPoint standardFunctionGradient(inverseIsoProbabilisticTransformation.gradient(standardSpaceDesignPoint_) * (physicalGradientMatrix * NumericalPoint(1, 1.0)));
	const NumericalScalar gradientToSensitivity(-(limitStateVariable_.getOperator().compare(1.0, 0.0) ? 1.0 : -1.0) / standardFunctionGradient.norm());
	/* evaluate the gradients of the physical model with respect to Set2 (ref doc : K2) */
	NumericalPoint physicalGradient;
	if (!isSet2Empty)
	  {
	    physicalGradient = physicalModel.parametersGradient(physicalSpaceDesignPoint_) * NumericalPoint(1, 1.0);
	  }
	/* evaluate the gradients of the isoprobabilistic transformation with respect to SetIso (ref doc : KS) */
	NumericalPoint isoProbabilisticGradient;
	if (setIso.getDimension() > 0)
	  {
	    isoProbabilisticGradient = inverseIsoProbabilisticTransformation.parametersGradient(standardSpaceDesignPoint_) * (physicalGradientMatrix * NumericalPoint(1, 1.0));
	  }
	/* associate to each element of Set1 the gradient value */

	/* hasoferReliabilityIndexSensitivity is the collection Set1 + one other collection wich is Set2 */
	const UnsignedLong set1Size(set1.getSize());
	const UnsignedLong size(set1Size + (isSet2Empty ? 0 : 1));
	hasoferReliabilityIndexSensitivity_ = Sensitivity(size);

	for (UnsignedLong sensitivityIndex = 0; sensitivityIndex < set1Size; ++sensitivityIndex)
	  {
	    const NumericalPointWithDescription currentParameters(set1[sensitivityIndex]);
	    const UnsignedLong currentDimension(currentParameters.getDimension());
	    NumericalPointWithDescription currentSensitivity(currentDimension);
	    const Description currentDescription(currentParameters.getDescription());
	    // the currentSensitivity gets the description and the name from set1
	    currentSensitivity.setDescription(currentDescription);
	    const String currentName(currentParameters.getName());
	    currentSensitivity.setName(currentName);
	    const Description isoDescription(setIso.getDescription());
	    for (UnsignedLong currentIndex = 0; currentIndex < currentDimension; ++currentIndex)
	      {
		const UnsignedLong position(computePosition(currentName, currentDescription[currentIndex], isoDescription));
		/* if currentParameters[currentIndex] is into setIso, then we get the sensitivity value in the corresponding isoProbabilisticGradient */
		if (position < setIso.getDimension())
		  {
		    currentSensitivity[currentIndex] = gradientToSensitivity * isoProbabilisticGradient[position];
		  }
		/* else the  sensitivity value is null */
	      } // currentIndex
	    hasoferReliabilityIndexSensitivity_[sensitivityIndex] = currentSensitivity;
	  } // sensitivityIndex

	/* if set2 is not empty : convert the matrix isoProbabilisticGradient, which has n rows and 1 column, into a NumericalPoint */
	if (size > set1Size)
	  {
	    hasoferReliabilityIndexSensitivity_[set1Size] = gradientToSensitivity * physicalGradient;
	    hasoferReliabilityIndexSensitivity_[set1Size].setName(physicalGradient.getName());
	  }
        isAlreadyComputedHasoferReliabilityIndexSensitivity_ = true;
      } // end computeHasoferReliabilityIndexSensitivity()


      /* Returns the position of the given (value, name) into the NumericalPoint or the dimension of the NumericalPoint if failed */
      UnsignedLong AnalyticalResult::computePosition(const String & marginalName, const String & marginalParameterName, const Description & parameterSetNames) const
      {
	const UnsignedLong dimension(parameterSetNames.getSize());
	const String fullName(OSS() << marginalName << "_" << marginalParameterName);
	for (UnsignedLong index = 0; index < dimension; ++index)
	  {
	    if ( parameterSetNames[index] == fullName ) return index;
	  }
	return dimension;
      }



      /* HasoferReliabilityIndexSensitivity Graph */
      AnalyticalResult::GraphCollection AnalyticalResult::drawHasoferReliabilityIndexSensitivity(NumericalScalar width)
      {
	GraphCollection hasoferReliabilityIndexSensitivityGraphCollection(0);
	// To ensure that the hasoferReliabilityIndexSensitivity_ are up to date
	if (! isAlreadyComputedHasoferReliabilityIndexSensitivity_)
	  {
	    computeHasoferReliabilityIndexSensitivity();
	  }
	const UnsignedLong dimension(standardSpaceDesignPoint_.getDimension());
	const UnsignedLong size(hasoferReliabilityIndexSensitivity_.getSize());
	// The first graph shows the sensitivities with respect to the marginal parameters if there exist
	// in the cas where the distribution or some marginals are defined by user, it may not have parameters : we create a empty sensitivity graph
	Sensitivity marginalSensitivity(dimension);
	for(UnsignedLong i = 0; i < dimension; ++i)
	  {
	    marginalSensitivity[i] = hasoferReliabilityIndexSensitivity_[i];
	  }
        Graph hasoferReliabilityIndexSensitivityGraphMarginal(drawSensitivity(marginalSensitivity, width));
	OSS oss1;
	oss1 << "Hasofer Reliability Index Sensitivities - Marginal parameters - " << limitStateVariable_.getName();
	hasoferReliabilityIndexSensitivityGraphMarginal.setTitle(oss1);
	hasoferReliabilityIndexSensitivityGraphCollection.add(hasoferReliabilityIndexSensitivityGraphMarginal);
	// The second graph shows the sensitivities with respect to the other parameters if there exist
	if (size > dimension)
	  {
	    Sensitivity otherSensitivity(size - dimension);
	    for(UnsignedLong i = dimension; i < size; ++i)
	      {
		otherSensitivity[i - dimension] = hasoferReliabilityIndexSensitivity_[i];
	      }
	    Graph hasoferReliabilityIndexSensitivityGraphOther(drawSensitivity(otherSensitivity, width));
	    OSS oss2;
	    oss2 << "Hasofer Reliability Index Sensitivities - Other parameters - " << limitStateVariable_.getName();
	    hasoferReliabilityIndexSensitivityGraphOther.setTitle(oss2);
	    hasoferReliabilityIndexSensitivityGraphCollection.add(hasoferReliabilityIndexSensitivityGraphOther);
	  }
	return hasoferReliabilityIndexSensitivityGraphCollection;
      }

      /* Sensitivity Graph */
      AnalyticalResult::Graph AnalyticalResult::drawSensitivity(const Sensitivity & sensitivity, NumericalScalar width) const
      {
	// shift position of the barplots
	NumericalScalar shift(0.0);

	// Create an empty graph
	Graph sensitivityGraph("Sensitivity", "parameters", "sensitivities", true, "topright");

	BarPlot sensitivityBarPlot(NumericalSample(0, 2), shift, "");

	const Description colors(DrawableImplementation::GetValidColors());
	Description::const_iterator colorsIterator;
	colorsIterator = colors.begin();

	// Create the barplots
	const UnsignedLong sensitivitySize(sensitivity.getSize());
	for (UnsignedLong collectionIndex = 0; collectionIndex < sensitivitySize; ++collectionIndex)
	  {
	    NumericalSample data(sensitivity[collectionIndex].getDimension(), 2);
	    const UnsignedLong dataSize(data.getSize());
	    for (UnsignedLong sensitivityIndex = 0; sensitivityIndex < dataSize; ++sensitivityIndex)
	      {
		data[sensitivityIndex][0] = width;
		data[sensitivityIndex][1] = sensitivity[collectionIndex][sensitivityIndex];
	      }
	    // Add the barplot to the graph
	    OSS oss;
	    oss << sensitivity[collectionIndex].getName() << " " << sensitivity[collectionIndex].getDescription();
	    sensitivityGraph.addDrawable(BarPlot(data, shift, (*colorsIterator), "solid", "solid", oss));
	    ++colorsIterator;
	    if (colorsIterator == colors.end())
	      {
		colorsIterator == colors.begin();
	      }
	    shift += width * (1 + dataSize);
	  }
	return sensitivityGraph;
      } //drawSensitivity


      /*  HasoferReliabilityIndexSensitivity accessor */
      AnalyticalResult::Sensitivity AnalyticalResult::getHasoferReliabilityIndexSensitivity()
      {
	if (! isAlreadyComputedHasoferReliabilityIndexSensitivity_)
	  {
	    computeHasoferReliabilityIndexSensitivity();
	  }
	return hasoferReliabilityIndexSensitivity_;
      }

      /* String converter */
      String AnalyticalResult::str() const
      {
	OSS oss;
	oss << "class=" << AnalyticalResult::GetClassName()
	    << " standardSpaceDesignPoint=" << standardSpaceDesignPoint_
	    << " physicalSpaceDesignPoint=" << physicalSpaceDesignPoint_
	    << " limitStateVariable=" << limitStateVariable_
	    << " isStandardPointOriginInFailureSpace=" << isStandardPointOriginInFailureSpace_
	    << " hasoferReliabilityIndex=" << hasoferReliabilityIndex_
	    << " importanceFactors=" << importanceFactors_
	    << " hasoferReliabilityIndexSensitivity=" << hasoferReliabilityIndexSensitivity_;
	return oss;
      }

      /* Method save() stores the object through the StorageManager */
      void AnalyticalResult::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
	adv.writeValue(standardSpaceDesignPoint_, StorageManager::MemberNameAttribute, "standardSpaceDesignPoint_");
	adv.writeValue(physicalSpaceDesignPoint_, StorageManager::MemberNameAttribute, "physicalSpaceDesignPoint_");
	adv.writeValue(limitStateVariable_, StorageManager::MemberNameAttribute, "limitStateVariable_");
	adv.writeValue("isStandardPointOriginInFailureSpace_", isStandardPointOriginInFailureSpace_);
 	adv.writeValue("hasoferReliabilityIndex_", hasoferReliabilityIndex_);
	adv.writeValue(importanceFactors_, StorageManager::MemberNameAttribute, "importanceFactors_");
	PersistentSensitivity sensitivity(hasoferReliabilityIndexSensitivity_);
	adv.writeValue(sensitivity, StorageManager::MemberNameAttribute, "hasoferReliabilityIndexSensitivity_");
 	adv.writeValue("isAlreadyComputedImportanceFactors_", isAlreadyComputedImportanceFactors_);
 	adv.writeValue("isAlreadyComputedHasoferReliabilityIndexSensitivity_", isAlreadyComputedHasoferReliabilityIndexSensitivity_);
      }

      /* Method load() reloads the object from the StorageManager */
      void AnalyticalResult::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
 	adv.readValue(standardSpaceDesignPoint_, StorageManager::MemberNameAttribute, "standardSpaceDesignPoint_");
 	adv.readValue(physicalSpaceDesignPoint_, StorageManager::MemberNameAttribute, "physicalSpaceDesignPoint_");
 	adv.readValue(limitStateVariable_, StorageManager::MemberNameAttribute, "limitStateVariable_");
	adv.readValue(importanceFactors_, StorageManager::MemberNameAttribute, "importanceFactors_");
	PersistentSensitivity sensitivity;
	adv.readValue(sensitivity, StorageManager::MemberNameAttribute, "hasoferReliabilityIndexSensitivity_");
	hasoferReliabilityIndexSensitivity_ = sensitivity;
	String name;
	NumericalScalar scalarValue;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, scalarValue)) {
	    if (name == "hasoferReliabilityIndex_") hasoferReliabilityIndex_ = scalarValue;
	  }
	}
	Bool boolValue;
	objList = adv.getList(StorageManager::BoolEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, boolValue)) {
	    if (name == "isStandardPointOriginInFailureSpace_") isStandardPointOriginInFailureSpace_ = boolValue;
	    if (name == "isAlreadyComputedImportanceFactors_") isAlreadyComputedImportanceFactors_ = boolValue;
	    if (name == "isAlreadyComputedHasoferReliabilityIndexSensitivity_") isAlreadyComputedHasoferReliabilityIndexSensitivity_ = boolValue;
	  }
	}
      }

    } /* namespace  Algorithm*/
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
