//                                               -*- C++ -*-
/**
 *  @file  AnalyticalResult.hxx
 *  @brief AnalyticalResult implements an algorithm to find the design point
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-07-03 08:31:34 +0200 (jeu, 03 jui 2008) $
 *  Id:      $Id: AnalyticalResult.hxx 867 2008-07-03 06:31:34Z lebrun $
 */
#ifndef OPENTURNS_ANALYTICALRESULT_HXX
#define OPENTURNS_ANALYTICALRESULT_HXX

#include "PersistentObject.hxx"
#include "NumericalPoint.hxx"
#include "NumericalPointWithDescription.hxx"
#include "Collection.hxx"
#include "Event.hxx"
#include "Graph.hxx"
#include "Description.hxx"

namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {

      /**
       * @class AnalyticalResult
       * AnalyticalResult stores the Analytical result 
       */
      class AnalyticalResult
	: public Base::Common::PersistentObject
      {
	CLASSNAME;
      public:

	static const NumericalScalar DefaultWidth = 1.0;

	typedef Base::Graph::Graph                                    Graph;
	typedef Graph::GraphCollection                                GraphCollection;
	typedef Base::Common::StorageManager                          StorageManager;
	typedef Base::Type::Description                               Description;
	typedef Base::Type::NumericalPoint                            NumericalPoint;
	typedef Base::Type::Collection<NumericalPoint>                NumericalPointCollection;
	typedef Base::Type::NumericalPointWithDescription             NumericalPointWithDescription;
	typedef Base::Type::Collection<NumericalPointWithDescription> NumericalPointWithDescriptionCollection;
	typedef NumericalPointWithDescriptionCollection               Sensitivity;
	typedef Model::Event                                          Event;

	/** Standard constructor */
	AnalyticalResult(const NumericalPoint & standardSpaceDesignPoint,
			 const Event & limitStateVariable,
			 const Bool isStandardPointOriginInFailureSpace,
			 const String & name = OT::DefaultName);

	/* Default constructor (required by SWIG :-<) */
	explicit AnalyticalResult();

	/** Virtual constructor */
	virtual AnalyticalResult * clone() const;

	/** StandardSpaceDesignPoint accessor */
	NumericalPoint getStandardSpaceDesignPoint() const;

	/** PhysicalSpaceDesignPoint accessor */
	void setStandardSpaceDesignPoint(const NumericalPoint & standardSpaceDesignPoint);

	/** PhysicalSpaceDesignPoint accessor */
	NumericalPoint getPhysicalSpaceDesignPoint() const;

	/** LimitStateVariable accessor */
	Event getLimitStateVariable() const;

	/** IsStandardPointOriginInFailureSpace  accessor */
	Bool getIsStandardPointOriginInFailureSpace() const;

	/** IsStandardPointOriginInFailureSpace  accessor */
	void setIsStandardPointOriginInFailureSpace(const Bool isStandardPointOriginInFailureSpace);

	/** ImportanceFactors accessor */
	NumericalPoint getImportanceFactors();

	/** ImportanceFactors graph */
	Graph drawImportanceFactors();

	/**  HasoferReliabilityIndex accessor */
	NumericalScalar getHasoferReliabilityIndex() const;

	/**  HasoferReliabilityIndexSensitivity accessor */
	Sensitivity getHasoferReliabilityIndexSensitivity();

	/** HasoferReliabilityIndexSensitivitygraph */
	GraphCollection drawHasoferReliabilityIndexSensitivity(NumericalScalar width = DefaultWidth);

	/** String converter */
	String str() const;

	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);

      protected:


	/** Sensitivitygraph */
	Graph drawSensitivity(const Sensitivity & sensitivity, NumericalScalar width) const;

      private:

	/** PhysicalSpaceDesignPoint evaluation */
	void computePhysicalSpaceDesignPoint();

	/** HasoferReliabilityIndex evaluation */
	void computeHasoferReliabilityIndex();

	/** ImportanceFactors evaluation */
	void computeImportanceFactors();

	/** HasoferReliabilityIndexSensitivity  evaluation */
	void computeHasoferReliabilityIndexSensitivity();

	/** PhysicalSpaceDesignPoint accessor */
	void setPhysicalSpaceDesignPoint(const NumericalPoint & physicalSpaceDesignPoint);

	/** HasoferReliabilityIndex accessor */
	void setHasoferReliabilityIndex(const NumericalScalar & hasoferReliabilityIndex);

	/** ImportanceFactors accessor */
	void setImportanceFactors(const NumericalPoint & importanceFactors);

	/** Returns the position of the given (value, name) into the NumericalPoint or the dimension of the NumericalPoint if failed */
	UnsignedLong computePosition(const String & marginalName,
				     const String & marginalParameterName,
				     const Description & parameterSetNames) const;

	NumericalPoint    standardSpaceDesignPoint_;
	NumericalPoint    physicalSpaceDesignPoint_;
	Event             limitStateVariable_;
	Bool isStandardPointOriginInFailureSpace_;
	NumericalScalar hasoferReliabilityIndex_;
	NumericalPointWithDescription importanceFactors_;
	Sensitivity hasoferReliabilityIndexSensitivity_;
	Bool isAlreadyComputedImportanceFactors_;
	Bool isAlreadyComputedHasoferReliabilityIndexSensitivity_;

      }; // class AnalyticalResult
    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_ANALYTICALRESULT_HXX */
