//                                               -*- C++ -*-
/**
 *  @file  SORMResult.cxx
 *  @brief SORMResult implements the results obtained from the Second Order Reliability Method
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: SORMResult.cxx 929 2008-09-13 20:37:56Z dutka $
 */
#include <cmath>
#include <complex>

#include "SORM.hxx"
#include "StandardEvent.hxx"
#include "NumericalMathFunction.hxx"
#include "SymmetricTensor.hxx"
#include "IdentityMatrix.hxx"
#include "Matrix.hxx"
#include "Log.hxx"
#include "Normal.hxx"
#include "StandardEvent.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      typedef Base::Type::SymmetricTensor       SymmetricTensor;
      typedef Base::Type::IdentityMatrix        IdentityMatrix;
      typedef Base::Type::Matrix                Matrix;
      typedef Base::Func::NumericalMathFunction NumericalMathFunction;
      typedef Base::Common::Log                 Log;
      typedef Distribution::Normal              Normal;
      typedef Model::StandardEvent              StandardEvent;

      CLASSNAMEINIT(SORMResult);

      static Base::Common::Factory<SORMResult> RegisteredFactory("SORMResult");

      /*
       * @brief  Standard constructor: the class is defined by an optimisation algorithm, a failure event and a physical starting point
       */
      SORMResult::SORMResult(const NumericalPoint & standardSpaceDesignPoint,
			     const Event & limitStateVariable,
			     const Bool isStandardPointOriginInFailureSpace,
			     const String & name):
	AnalyticalResult(standardSpaceDesignPoint, limitStateVariable, isStandardPointOriginInFailureSpace, name),
	standardDistribution_(limitStateVariable.getImplementation()->getAntecedent().getImplementation()->getDistribution().getStandardDistribution()),
	standardMarginal_(standardDistribution_.getMarginal(0))
      {
	/* get the physical Limite State Function */
	
      	NumericalMathFunction limitStateFunction(StandardEvent(limitStateVariable).getImplementation()->getFunction());
	/* compute its gradient */
	Matrix gradient(limitStateFunction.gradient(getStandardSpaceDesignPoint()));
	/* Get the first column */
	gradientLimitStateFunction_ = gradient * NumericalPoint(1, 1.0);
	/* compute its hessian */
	SymmetricTensor hessian(limitStateFunction.hessian(getStandardSpaceDesignPoint()));
	/* Get the first sheet */
	hessianLimitStateFunction_ = SquareMatrix(hessian.getNbRows());
	for (UnsignedLong i = 0; i < hessian.getNbRows(); i++)
	  {
	    for (UnsignedLong j = 0; j < hessian.getNbColumns(); j++)
	      {
		hessianLimitStateFunction_(i, j) = hessian(i, j, 0);
	      }
	  }
	/* compute its curvatures and the several formulas for the SORM probability */
	computeCurvatures();
	eventProbabilityBreitung_ = -1.0;
	try{
	  computeEventProbabilityBreitung();
	}
	catch (NotDefinedException & ex) {
	  Log::Info(ex.str());
	}


	eventProbabilityHohenBichler_ = -1.0;
	try{
	  computeEventProbabilityHohenBichler();
	}
	catch (NotDefinedException & ex) {
	  Log::Info(ex.str());
	}


	eventProbabilityTvedt_ = -1.0;
	try{
	  computeEventProbabilityTvedt();
	}
	catch (NotDefinedException & ex) {
	  Log::Info(ex.str());
	}


	try{
	  computeGeneralisedReliabilityIndex();
	}
	catch (NotDefinedException & ex) {
	  Log::Info(ex.str());
	}
      } // end SORMResult::Result

      /* Default constructor */
      SORMResult::SORMResult():
	AnalyticalResult(),
	hessianLimitStateFunction_(),
	gradientLimitStateFunction_(),
	sortedCurvatures_(),
	eventProbabilityBreitung_(),
	eventProbabilityHohenBichler_(),
	eventProbabilityTvedt_(),
	generalisedReliabilityIndexBreitung_(),
	generalisedReliabilityIndexHohenBichler_(),
	generalisedReliabilityIndexTvedt_(),
	standardDistribution_(Normal(1)),
	standardMarginal_(Normal(1))
      {
	// Nothing to do
      }

      /* Virtual constructor */
      SORMResult * SORMResult::clone() const
      {
	return new SORMResult(*this);
      }

      /* The function that actually evaluates the event probability with SORM Breitung approximation */
      void SORMResult::computeEventProbabilityBreitung()
	throw(NotDefinedException)
      {
	NumericalScalar beta(getHasoferReliabilityIndex());
	NumericalPoint minusBeta(1, -beta);
	NumericalScalar standardCDFBeta(standardMarginal_.computeCDF(minusBeta));

	/* test if all curvatures verifie 1 + beta * curvature  > 0 */
	/* curvatures are sorted with increasing order and stocked in the attribute SortedCurvatures */
	if (1.0 + beta * sortedCurvatures_[0] < 0) 
	  {
	    throw NotDefinedException(HERE) << "Error: impossible to calculate Breitung SORM probability, one of the curvatures is < -1/beta";
	  }

	/* PBreitung = E(-beta)/Prod(sqrt(1+beta*curvature[i])) */
	eventProbabilityBreitung_ = standardCDFBeta;
	for (UnsignedLong index = 0 ; index < sortedCurvatures_.getDimension(); index ++)
	  {
	    eventProbabilityBreitung_ /= sqrt(1.0 + beta*sortedCurvatures_[index]);
	  }
	/* if the Standard Point Origin is in the failure space, take the complementry probability */
	if (getIsStandardPointOriginInFailureSpace())
	  {
	    eventProbabilityBreitung_ = 1.0 - eventProbabilityBreitung_;
	  }
      } // end SORMResult::computeEventProbabilityBreitung

	/* EventProbabilityBreitung accessor */
      NumericalScalar SORMResult::getEventProbabilityBreitung() const
      { 
	return eventProbabilityBreitung_;
      }

      /* The function that actually evaluates the curvatures of the standard limite state function at the standard design point */
      void SORMResult::computeCurvatures()
      {
	/* see Mefisto v3.2 documentation */
	/* we calculate the main curvatures */
	NumericalScalar inverseGradientNorm(1.0 / gradientLimitStateFunction_.norm());
	NumericalPoint unitGradientLimitStateFunction(inverseGradientNorm * gradientLimitStateFunction_);
	UnsignedLong dimension(getStandardSpaceDesignPoint().getDimension());
	SquareMatrix kroneckerUnitGradientLimitStateFunction(dimension);
	for (UnsignedLong i = 0; i < dimension; i++)
	  {
	    for (UnsignedLong j = 0; j < dimension; j++)
	      {
		kroneckerUnitGradientLimitStateFunction(i, j) = unitGradientLimitStateFunction[i] * unitGradientLimitStateFunction[j];
	      }
	  } 
	/* W = (uGrad.uGrad^t -Id) * Hess(g) */
	SquareMatrix Id = IdentityMatrix(dimension);
	SquareMatrix W((kroneckerUnitGradientLimitStateFunction - Id) * hessianLimitStateFunction_);

	/* The curvatures are proportional to the eigenvalues of W
	   If the normal of the boundary of the failure domain points to the origin at the design point, then we change
           the sign of the curvatures. It insure that a convexe failure domain will have positive curvatures */
	sortedCurvatures_ = (NumericalPoint::dot(gradientLimitStateFunction_, getStandardSpaceDesignPoint()) > 0.0 ? 1.0 : -1.0) * inverseGradientNorm * (W.computeEigenValues());
	/* we sort the curvatures with increasing order */
	std::sort(sortedCurvatures_.begin(), sortedCurvatures_.end());
      } // end SORMResult::computeCurvatures

      /* SortedCurvatures accessor */
      SORM::NumericalPoint SORMResult::getSortedCurvatures() const
      {
	return sortedCurvatures_;
      }

      /* The function that actually evaluates the event probability with SORM HohenBichler approximation */
      void SORMResult::computeEventProbabilityHohenBichler()
	throw(NotDefinedException)
      {    
	/* this formula is valid only in the gaussian case */
	/* DEBUG : PUT AN EXCEPTION HERE TO TEST THE GAUSSIAN CASE */
	NumericalPoint minusBeta(1, -getHasoferReliabilityIndex());
	NumericalScalar standardPDFBeta(standardMarginal_.computePDF(minusBeta));
	NumericalScalar standardCDFBeta(standardMarginal_.computeCDF(minusBeta));

	NumericalScalar rho(standardPDFBeta / standardCDFBeta);

	/* test if all curvatures verifie 1 + rho * curvature  > 0 */
	/* curvatures are sorted with increasing order and stocked in the attribute SortedCurvatures */
	if (1.0 + rho * sortedCurvatures_[0] < 0) 
	  {
	    throw NotDefinedException(HERE) << "Error: impossible to calculate HohenBichler SORM probability, one of the curvatures is < -1/rho";
	  }

	/* Phohenbichler = Phi(-beta)/Prod(sqrt(1+rho*curvature[i])) */
	eventProbabilityHohenBichler_ = standardCDFBeta;
	for (UnsignedLong index = 0 ; index < sortedCurvatures_.getDimension(); index ++)
	  {
	    eventProbabilityHohenBichler_ /= sqrt(1.0 + rho*sortedCurvatures_[index]);
	  }

	/* if the Standard Point Origin is in the failure space, take the complementry probability */
	if (getIsStandardPointOriginInFailureSpace()) 
	  {
	    eventProbabilityHohenBichler_ = 1.0 - eventProbabilityHohenBichler_;
	  }
      } // SORMResult::computeEventProbabilityHohenBichler

	/* EventProbability HohenBichleraccessor */
      NumericalScalar SORMResult::getEventProbabilityHohenBichler() const
      { 
	return eventProbabilityHohenBichler_;
      }

      /* The function that actually evaluates the event probability with SORM Tvedtapproximation */
      void SORMResult::computeEventProbabilityTvedt()
	throw(NotDefinedException)
      {    
	/* this formula is valid only in the gaussian case */
	/* PUT AN EXCEPTION HERE TO TEST THE GAUSSIAN CASE */

	NumericalScalar beta(getHasoferReliabilityIndex());
	NumericalPoint minusBeta(1, -beta);

	/* test if all curvatures verifie 1 + (beta+1) * curvature  > 0 */
	/* curvatures are sorted with increasing order and stocked in the attribute SortedCurvatures */
	if (1.0 + (1 + beta) * sortedCurvatures_[0] < 0) 
	  {
	    throw NotDefinedException(HERE) << "Error: impossible to calculate Tvedt SORM probability, one of the curvatures is < -1/(1+beta)";
	  }

	NumericalScalar standardPDFBeta(standardMarginal_.computePDF(minusBeta));
	NumericalScalar standardCDFBeta(standardMarginal_.computeCDF(minusBeta));

	/* compute the first term A1 */

	NumericalScalar prod1(1.0);
	UnsignedLong dimension(sortedCurvatures_.getDimension());
	for (UnsignedLong index = 0 ; index < dimension; index ++)
	  {
	    prod1 /= sqrt(1.0 + beta * sortedCurvatures_[index]);
	  }
	NumericalScalar termA1(standardCDFBeta * prod1);

	/* compute the second term A2 */

	NumericalScalar rho(beta * standardCDFBeta - standardPDFBeta);

	NumericalScalar prod2(1.0);
	for (UnsignedLong index = 0; index < dimension; index ++)
	  {
	    prod2 /= sqrt(1.0 + (1.0 + beta) * sortedCurvatures_[index]);
	  }

	NumericalScalar termA2(rho * (prod1 - prod2));

	/* compute the second term A3 */

	std::complex<NumericalScalar> complexProd3(1.0, 0.0);
	std::complex<NumericalScalar> iPlusBeta(beta, 1.0);

	for (UnsignedLong index = 0; index < dimension; index ++)
	  {
	    complexProd3 /= sqrt(1.0 + iPlusBeta * sortedCurvatures_[index]);
	  }
	NumericalScalar termA3((beta + 1.0) * rho * (prod1 - complexProd3.real()));

	/* compute tvedt probability */

	eventProbabilityTvedt_ = termA1 + termA2 + termA3;
	/* if the Standard Point Origin is in the failure space, take the complementry probability */
	if (getIsStandardPointOriginInFailureSpace()) 
	  {
	    eventProbabilityTvedt_ = 1.0 - eventProbabilityTvedt_;
	  }
      } // end SORMResult::computeEventProbabilityTvedt

	/* EventProbability accessor */
      NumericalScalar SORMResult::getEventProbabilityTvedt() const
      { 
	return eventProbabilityTvedt_;
      }

      /* The function that actually evaluates the generalised reliability index with SORM BreitungHB and Tvedt  approximations */
      void SORMResult::computeGeneralisedReliabilityIndex()
      {
	/* evaluate the GeneralisedReliabilityIndex */
	//* GeneralisedReliabilityIndex is defined by : - Inverse standard marginal CDF (eventProbability) in usual case or : + Inverse standard marginal CDF (eventProbability) in other case */
	NumericalScalar sign(1.0);
	if (!getIsStandardPointOriginInFailureSpace()) { // StandardPointOriginInFailureSpace is FALSE : usual case
	  sign = -1.0;
	}

	/* generalised reliability index with SORM Breitung approximation */
	generalisedReliabilityIndexBreitung_ = sign * standardMarginal_.computeQuantile(eventProbabilityBreitung_)[0];

	/* generalised reliability index with SORM HohenBichler approximation */
	generalisedReliabilityIndexHohenBichler_ = 0.0;
	if (eventProbabilityHohenBichler_ != -1.0)
	  {
	    generalisedReliabilityIndexHohenBichler_ = sign * standardMarginal_.computeQuantile(eventProbabilityHohenBichler_)[0];
	  }

	/* generalised reliability index with SORM Tvedt approximation */
	generalisedReliabilityIndexTvedt_ = 0.0;
	if (eventProbabilityTvedt_ != -1.0)
	  {
	    generalisedReliabilityIndexTvedt_ = sign * standardMarginal_.computeQuantile(eventProbabilityTvedt_)[0];
	  }

      } // end SORMResult::computeGeneralisedReliabilityIndex

	/* GeneralisedReliabilityIndexBreitung accessor */
      NumericalScalar SORMResult::getGeneralisedReliabilityIndexBreitung() const
      {
	return generalisedReliabilityIndexBreitung_;
      }

      /* GeneralisedReliabilityIndex accessor */
      NumericalScalar SORMResult::getGeneralisedReliabilityIndexHohenBichler() const
      {
	return generalisedReliabilityIndexHohenBichler_;
      }

      /* GeneralisedReliabilityIndex accessor */
      NumericalScalar SORMResult::getGeneralisedReliabilityIndexTvedt() const
      {
	return generalisedReliabilityIndexTvedt_;
      }

      /* String converter */ 
      String SORMResult::str() const 
      {
	OSS oss;
	oss << "class=" << SORMResult::GetClassName()
	    << " " << AnalyticalResult::str()
	    << " sortedCurvatures=" << sortedCurvatures_
	    << " eventProbabilityBreitung=" << eventProbabilityBreitung_
	    << " eventProbabilityHohenBichler=" << eventProbabilityHohenBichler_
	    << " eventProbabilityTvedt=" << eventProbabilityTvedt_
	    << " generalisedReliabilityIndexBreitung=" << generalisedReliabilityIndexBreitung_
	    << " generalisedReliabilityIndexHohenBichler=" << generalisedReliabilityIndexHohenBichler_
	    << " generalisedReliabilityIndexTvedt=" << generalisedReliabilityIndexTvedt_
	    << " gradientLimitStateFunction_=" << gradientLimitStateFunction_
	    << " hessianLimitStateFunction_=" << hessianLimitStateFunction_;
	return oss;
      }

      /* Method save() stores the object through the StorageManager */
      void SORMResult::save(const StorageManager::Advocate & adv) const
      {
	AnalyticalResult::save(adv);
	adv.writeValue(hessianLimitStateFunction_, StorageManager::MemberNameAttribute, "hessianLimitStateFunction_");
	adv.writeValue(gradientLimitStateFunction_, StorageManager::MemberNameAttribute, "gradientLimitStateFunction_");
	adv.writeValue(sortedCurvatures_, StorageManager::MemberNameAttribute, "sortedCurvatures_");
	adv.writeValue("eventProbabilityBreitung_", eventProbabilityBreitung_);
	adv.writeValue("eventProbabilityHohenBichler_", eventProbabilityHohenBichler_);
	adv.writeValue("eventProbabilityTvedt_", eventProbabilityTvedt_);
	adv.writeValue("generalisedReliabilityIndexBreitung_", generalisedReliabilityIndexBreitung_);
	adv.writeValue("generalisedReliabilityIndexHohenBichler_", generalisedReliabilityIndexHohenBichler_);
	adv.writeValue("generalisedReliabilityIndexTvedt_", generalisedReliabilityIndexTvedt_);
	adv.writeValue(standardDistribution_, StorageManager::MemberNameAttribute, "standardDistribution_");
	adv.writeValue(standardMarginal_, StorageManager::MemberNameAttribute, "standardMarginal_");
      }

      /* Method load() reloads the object from the StorageManager */
      void SORMResult::load(const StorageManager::Advocate & adv)
      {
	AnalyticalResult::load(adv);
	adv.readValue(hessianLimitStateFunction_, StorageManager::MemberNameAttribute, "hessianLimitStateFunction_");
	adv.readValue(gradientLimitStateFunction_, StorageManager::MemberNameAttribute, "gradientLimitStateFunction_");
	adv.readValue(sortedCurvatures_, StorageManager::MemberNameAttribute, "sortedCurvatures_");
	adv.readValue(standardDistribution_, StorageManager::MemberNameAttribute, "standardDistribution_");
	adv.readValue(standardMarginal_, StorageManager::MemberNameAttribute, "standardMarginal_");
	String name;
	NumericalScalar scalarValue;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, scalarValue)) {
	    if (name == "eventProbabilityBreitung_") eventProbabilityBreitung_ = scalarValue;
	    if (name == "eventProbabilityHohenBichler_") eventProbabilityHohenBichler_ = scalarValue;
	    if (name == "eventProbabilityTvedt_") eventProbabilityTvedt_ = scalarValue;
	    if (name == "generalisedReliabilityIndexBreitung_") generalisedReliabilityIndexBreitung_ = scalarValue;
	    if (name == "generalisedReliabilityIndexHohenBichler_") generalisedReliabilityIndexHohenBichler_ = scalarValue;
	    if (name == "generalisedReliabilityIndexTvedt_") generalisedReliabilityIndexTvedt_ = scalarValue;
	  }
	}
      }

    } /* namespace  Algorithm*/
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
