//                                               -*- C++ -*-
/**
 *  @file  Axial.cxx
 *  @brief Implementation of the axial experiment plane
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: Axial.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include "OT.hxx"
#include "Axial.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      CLASSNAMEINIT(Axial);

      /* Default constructor */
      Axial::Axial(const String & name):
	ExperimentImplementation(name)
      {
	// Nothing to do
      }

      /* Constructor with parameters */
      Axial::Axial(const NumericalPoint & center,
		   const NumericalPoint & levels,
		   const String & name):
	ExperimentImplementation(center, levels, name)
      {
	// Nothing to do
      }

      /* Constructor with parameters */
      Axial::Axial(const UnsignedLong dimension,
		   const NumericalPoint & levels,
		   const String & name):
	ExperimentImplementation(NumericalPoint(dimension, 0.0), levels, name)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      Axial * Axial::clone() const
      {
	return new Axial(*this);
      }

      /* Experiment plane generation */
      Axial::NumericalSample Axial::generate()
      {
	NumericalPoint center(getCenter());
	/* Dimension of the realizations */
	UnsignedLong dimension(center.getDimension());
	/* Levels */
	NumericalPoint levels(getLevels());
	/* Number of level to be generated */
	UnsignedLong levelNumber(levels.getDimension());
	/* Size of the sample to be generated: 1 + 2 x number of levels x dimension */
	UnsignedLong size(1 + 2 * levelNumber * dimension);
	NumericalSample axialPlane(size, center);
	axialPlane.setName("Axial plane");
	UnsignedLong index(1);
	/* For each level of the axial plane */
	for(UnsignedLong levelIndex = 0; levelIndex < levelNumber; levelIndex++)
	  {
	    NumericalScalar levelValue(levels[levelIndex]);
	    /* For each vertex of one half of the current level */
	    for(UnsignedLong vertex = 0; vertex < dimension; vertex++)
	      {
		/* Compute the coordinates of the positive vertex */
		axialPlane[index][vertex] += levelValue;
		index++;
		/* Compute the coordinates of the negative vertex */
		axialPlane[index][vertex] -= levelValue;
		index++;
	      } // vertex
	  } // levelIndex
	return axialPlane;
      } // generate()

      /* String converter */
      String Axial::str() const
      {
	OSS oss;
	oss << "class=" << GetClassName()
	    << " name=" << getName()
	    << " center=" << getCenter()
	    << " levels=" << getLevels();
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
