//                                               -*- C++ -*-
/**
 *  @file  Box.cxx
 *  @brief Implementation of the box experiment plane
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: Box.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "OT.hxx"
#include "Box.hxx"
#include "Collection.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      CLASSNAMEINIT(Box);

      typedef Base::Type::Collection<UnsignedLong> UnsignedLongCollection;

      /* Default constructor */
      Box::Box(const String & name):
	ExperimentImplementation(name)
      {
	// Nothing to do
      }

      /* Constructor with parameters */
      Box::Box(const NumericalPoint & center,
	       const NumericalPoint & levels,
	       const String & name)
	throw(InvalidArgumentException):
	ExperimentImplementation(center, levels, name)
      {
	// Check if there is the same number of levels than the dimension of the experiment plane
	if (center.getDimension() != levels.getDimension()) throw InvalidArgumentException(HERE) << "Error: the number of levels must be equal to the dimension of the box experiment plane";
	setLevels(levels);
      }

      /* Constructor with parameters */
      Box::Box(const NumericalPoint & levels,
	       const String & name)
	throw(InvalidArgumentException):
	ExperimentImplementation(NumericalPoint(levels.getDimension(), 0.0), levels, name)
      {
	// Check if there is the same number of levels than the dimension of the experiment plane
	if (levels.getDimension() == 0) throw InvalidArgumentException(HERE) << "Error: the levels dimension must be > 0";
	setLevels(levels);
      }

      /* Virtual constructor */
      Box * Box::clone() const
      {
	return new Box(*this);
      }

      /* Experiment plane generation
         The box [center - 1/2, center + 1/2] is uniformly sampled in each dimension
         levels counts the number of interior points in each dimension */
      Box::NumericalSample Box::generate()
      {
	NumericalPoint center(getCenter());
	/* Dimension of the realizations */
	UnsignedLong dimension(center.getDimension());
	/* Levels */
	NumericalPoint scalarLevels(getLevels());
	UnsignedLongCollection levels(dimension);
	/* Convert NumericalScalar values to UnsignedLong values for the levels */
	for (UnsignedLong i = 0; i < dimension; i++) levels[i] = UnsignedLong(floor(scalarLevels[i])) + 2;
	/* Size of the sample to be generated: levels[0] * ... * levels[dimension-1] */
	UnsignedLong size(levels[0]);
	for (UnsignedLong i = 1; i < dimension; i++) size *= levels[i];
	NumericalSample boxPlane(size, dimension);
	boxPlane.setName("Box plane");
	/* Indices would have stored the indices of the nested loops if we were able to code "dimension" nested loops dynamically */
	UnsignedLongCollection indices(dimension, 0);
	/* The experiment plane will extend from center - 1/2 to center + 1/2. Origin = -center/2 */
	NumericalPoint origin(center - NumericalPoint(dimension, 0.5));
	for (UnsignedLong flatIndex = 0; flatIndex < size; flatIndex++)
	  {
	    NumericalPoint point(origin);
	    for (UnsignedLong i = 0; i < dimension; i++) point[i] += indices[i] / (scalarLevels[i] + 1.0);
	    boxPlane[flatIndex] = point;
	    /* Update the indices */
	    indices[0]++;
	    /* Propagate the remainders */
	    for (UnsignedLong i = 0; i < dimension - 1; i++) indices[i + 1] += (indices[i] == levels[i]);
	    /* Correction of the indices. The last index cannot overflow. */
	    for (UnsignedLong i = 0; i < dimension - 1; i++) indices[i] = indices[i] % levels[i];
	  }
	return boxPlane;
      } // generate()

      /* String converter */
      String Box::str() const
      {
	OSS oss;
	oss << "class=" << GetClassName()
	    << " name=" << getName()
	    << " center=" << getCenter()
	    << " levels=" << getLevels();
	return oss;
      }

      /** Specific levels accessor */
      void Box::setLevels(const NumericalPoint & levels) throw(InvalidArgumentException)
      {
	UnsignedLong dimension(getCenter().getDimension());
	UnsignedLong size(levels.getDimension());
	if (size != dimension) throw InvalidArgumentException(HERE) << "Error: levels dimension must equal center dimension for the Box design of experiment, here levels dimension=" << size << " and center dimension=" << dimension;
	for (UnsignedLong i = 0; i < dimension; i++)
	  {
	    if (levels[i] < 0.0) throw InvalidArgumentException(HERE) << "Error: levels values must be greater or equal to 0 for the Box design of experiment";
	  }
	ExperimentImplementation::setLevels(levels);
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
