//                                               -*- C++ -*-
/**
 *  @file  InverseNatafIndependentCopulaGradient.cxx
 *  @brief Class for the InverseNataf transformation evaluation for independent
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: InverseNatafIndependentCopulaGradient.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "InverseNatafIndependentCopulaGradient.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      /*
       * @class InverseNatafIndependentCopulaGradient
       *
       * This class offers an interface for the InverseNataf function for independent copula
       */

      CLASSNAMEINIT(InverseNatafIndependentCopulaGradient);

      /* Parameter constructor */
      InverseNatafIndependentCopulaGradient::InverseNatafIndependentCopulaGradient(const UnsignedLong dimension):
	NumericalMathGradientImplementation(),
	dimension_(dimension)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      InverseNatafIndependentCopulaGradient * InverseNatafIndependentCopulaGradient::clone() const
      {
	return new InverseNatafIndependentCopulaGradient(*this);
      }

      /* String converter */
      String InverseNatafIndependentCopulaGradient::str() const
      {
	OSS oss;
	oss << "class=" << InverseNatafIndependentCopulaGradient::GetClassName()
            << " dimension=" << dimension_;
                   
	return oss;
      }

      /*
       * Evaluation
       * The inverse Nataf transform S reads:
       * Si(u) = Phi(ui), where Phi is the CDF of the standard normal distribution
       * Jij = dSi/duj = Phi'(ui) if i = j
       *               = 0 else
       */
      InverseNatafIndependentCopulaGradient::Matrix InverseNatafIndependentCopulaGradient::gradient(const NumericalPoint & in) const
	throw(InvalidArgumentException, InternalException)
      {
	Matrix result(dimension_, dimension_);
	for (UnsignedLong i = 0; i < dimension_; ++i)
	  {
	    NumericalScalar x(in[i]);
	    // 0.3989422804014326779399462 = 1/sqrt(2*Pi)
	    result(i, i) = 0.3989422804014326779399462 * exp(-0.5 * x * x);
	  }
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong InverseNatafIndependentCopulaGradient::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return dimension_;
      }

      /* Accessor for output point dimension */
      UnsignedLong InverseNatafIndependentCopulaGradient::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return dimension_;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
