//                                               -*- C++ -*-
/**
 *  @file  StandardSpaceImportanceSampling.cxx
 *  @brief StandardSpaceImportanceSampling is an implementation of the  importance sampling Monte Carlo simulation method in standard space
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-04-15 16:24:44 +0200 (mar, 15 avr 2008) $
 *  Id:      $Id: StandardSpaceImportanceSampling.cxx 789 2008-04-15 14:24:44Z lebrun $
 */
#include "StandardSpaceImportanceSampling.hxx"
#include "NumericalPoint.hxx"
#include "ComparisonOperatorImplementation.hxx"

namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {

      /*
       * @class StandardSpaceImportanceSampling
       */

      CLASSNAMEINIT(StandardSpaceImportanceSampling);

      typedef Base::Type::NumericalPoint NumericalPoint;

      /* Constructor with parameters */
      StandardSpaceImportanceSampling::StandardSpaceImportanceSampling(const AnalyticalResult & analyticalResult) throw(InvalidArgumentException):
	StandardSpaceSimulation(analyticalResult)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      StandardSpaceImportanceSampling * StandardSpaceImportanceSampling::clone() const
      {
	return new StandardSpaceImportanceSampling(*this);
      }

      /* Compute the block sample */
      StandardSpaceImportanceSampling::NumericalSample StandardSpaceImportanceSampling::computeBlockSample()
      {
	const UnsignedLong blockSize(getBlockSize());
	const NumericalPoint standardSpaceDesignPoint(analyticalResult_.getStandardSpaceDesignPoint());
	// Get the threshold and the reliability index
	const NumericalScalar threshold(event_.getThreshold());
	// First, compute a sample of the importance distribution. It is simply
	// the standard distribution translated to the design point
	NumericalSample inputSample(standardDistribution_.getNumericalSample(blockSize));
	inputSample.translate(standardSpaceDesignPoint);
	// Then, evaluate the function on this sample
	NumericalSample blockSample(getEvent().getImplementation()->getFunction()(inputSample));
	// Then, modify in place this sample to take into account the change in the input distribution
	for (UnsignedLong i = 0; i < blockSize; i++)
	  {
	    inputStrategy_.store(inputSample[i]);
	    outputStrategy_.store(blockSample[i]);
	    if (event_.getOperator()(blockSample[i][0], threshold))
	      {
		// If the event occured, the value is p_initial(x[i]) / p_importance(x[i])
		// Having access to p_initial is a long trip...
		blockSample[i][0] = standardDistribution_.computePDF(inputSample[i]) / standardDistribution_.computePDF(inputSample[i] - standardSpaceDesignPoint);
	      }
	    else
	      {
		blockSample[i][0] = 0.0;
	      }
	  }
	return blockSample;
      }

      /* String converter */
      String StandardSpaceImportanceSampling::str() const
      {
	OSS oss;
	oss << "class=" << StandardSpaceImportanceSampling::GetClassName()
	    << " derived from " << StandardSpaceSimulation::str();
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
