//                                               -*- C++ -*-
/**
 *  @file  Beta.cxx
 *  @brief The Beta distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: Beta.cxx 929 2008-09-13 20:37:56Z dutka $
 */
#include <cmath>
#include "Beta.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "DistFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      CLASSNAMEINIT(Beta);

      static Base::Common::Factory<Beta> RegisteredFactory("Beta");

      typedef Base::Stat::RandomGenerator RandomGenerator;

      /* Default constructor */
      Beta::Beta()
	: NonEllipticalDistribution("Beta"),
	  r_(2.0), t_(4.0), a_(-1.0), b_(1.0), normalizationFactor_(0.75)
      {
	setDimension(1);
	computeRange();
      }

      /* Parameters constructor */
      Beta::Beta(const NumericalScalar arg1,
		 const NumericalScalar arg2,
		 const NumericalScalar a,
		 const NumericalScalar b,
		 const ParameterSet set)
	throw (InvalidArgumentException)
	: NonEllipticalDistribution("Beta"),
	  r_(0.), t_(0.), a_(a), b_(b), normalizationFactor_(0.0)
      {
	if (b <= a) throw InvalidArgumentException(HERE) << "Error: the upper bound must be greater than the lower bound, here a=" << a << " and b=" << b;
	switch (set) {
	case RT:
	  setR(arg1);
	  setT(arg2);
	  break;
	  
	case MUSIGMA:
	  setMuSigma(arg1, arg2);
	  break;

	default:
	  throw InvalidArgumentException(HERE) << "Invalid parameter set argument";

	} /* end switch */
	setDimension(1);
	computeRange();
	update();
      }

      /* Comparison operator */
      Bool Beta::operator ==(const Beta & other) const {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	  // TODO: Write Beta::operator ==(...)
	  sameObject = (r_ == other.r_) && (t_ == other.t_) &&
	    (a_ == other.a_) && (b_ == other.b_);
	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String Beta::str() const {
	OSS oss;
	oss << "class=" << Beta::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " r=" << r_
	    << " t=" << t_
	    << " a=" << a_
	    << " b=" << b_;
	return oss;
      }
  
      /* Virtual constructor */
      Beta * Beta::clone() const
      {
	return new Beta(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void Beta::computeRange()
      {
	setRange(Interval(a_, b_));
      }

      /** Update the derivative attributes */
      void Beta::update()
      {
	normalizationFactor_ = pow(b_ - a_, 1.0 - t_) / SpecFunc::Beta(r_, t_ - r_);
      }


      /* Get one realization of the distribution */
      Beta::NumericalPoint Beta::getRealization() const
      {
	return NumericalPoint(1, a_ + (b_ - a_) * DistFunc::rBeta(r_, t_ - r_));
      }

      /* Get the DDF of the distribution */
      Beta::NumericalPoint Beta::computeDDF(const NumericalPoint & point) const
      {
	NumericalScalar x(point[0]);
	if ((x <= a_) || (x > b_)) return NumericalPoint(1, 0.0);
	return NumericalPoint(1, ((r_ - 1.0) / (x - a_) - (t_ - r_ - 1.0) / (b_ - x)) * computePDF(point));
      }


      /* Get the PDF of the distribution */
      NumericalScalar Beta::computePDF(const NumericalPoint & point) const
      {
	NumericalScalar x(point[0]);
	if ((x == b_) && (t_ - r_ == 1.0)) return 1.0;
	if ((x <= a_) || (x >= b_)) return 0.0;
	return normalizationFactor_ * pow(x - a_, r_ - 1.0) * pow(b_ - x, t_ - r_ - 1.0);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Beta::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	NumericalScalar x(point[0]);
	if (x <= a_) return (tail ? 1.0 : 0.0);
	if (x > b_) return (tail ? 0.0 : 1.0);
	return DistFunc::pBeta(r_, t_ - r_, (x - a_) / (b_ - a_), tail);
      }

      /** Get the PDFGradient of the distribution */
      Beta::NumericalPoint Beta::computePDFGradient(const NumericalPoint & point) const
      {
	NumericalPoint pdfGradient(4, 0.0);
	NumericalScalar x(point[0]);
	if ((x <= a_) || (x > b_)) return pdfGradient;
	NumericalScalar pdf(computePDF(point));
	NumericalScalar psiTR(SpecFunc::Psi(t_ - r_));
	NumericalScalar iBA(1.0 / (b_ - a_));
	NumericalScalar BX(b_ - x);
	NumericalScalar iBX(1.0 / BX);
	NumericalScalar XA(x - a_);
	NumericalScalar iXA(1.0 / XA);
	pdfGradient[0] = pdf * (log(XA * iBX) + psiTR - SpecFunc::Psi(r_));
	pdfGradient[1] = pdf * (log(BX * iBA) - psiTR + SpecFunc::Psi(t_));
	pdfGradient[2] = pdf * ((t_ - 1.0) * iBA - (r_ - 1.0) * iXA);
	pdfGradient[3] = pdf * ((t_ - 1.0) * XA * iBA * iBX - r_ * iBX);
	return pdfGradient;
      }

      /** Get the CDFGradient of the distribution */
      Beta::NumericalPoint Beta::computeCDFGradient(const NumericalPoint & point) const
      {
	NumericalPoint cdfGradient(4, 0.0);
	NumericalScalar x(point[0]);
	if ((x <= a_) || (x > b_)) return cdfGradient;
	NumericalScalar cdf(computeCDF(point));
	NumericalScalar iBA(1.0 / (b_ - a_));
	NumericalScalar cdfShift(DistFunc::pBeta(r_ + 1.0, t_ - r_ - 1.0, (x - a_) * iBA));
	NumericalScalar cdfDiff(cdfShift - cdf);
	NumericalScalar factor(r_ * iBA);
	static const NumericalScalar eps(pow(DistFunc::Precision, 1.0 / 3.0));
	static const NumericalScalar i2Eps(0.5 / eps);
	cdfGradient[0] = i2Eps * (DistFunc::pBeta(r_ + eps, t_ - r_ - eps, (x - a_) / (b_ - a_)) - DistFunc::pBeta(r_ - eps, t_ - r_ + eps, (x - a_) / (b_ - a_)));
	cdfGradient[1] = i2Eps * (DistFunc::pBeta(r_, t_ - r_ + eps, (x - a_) / (b_ - a_)) - DistFunc::pBeta(r_, t_ - r_ - eps, (x - a_) / (b_ - a_)));
	cdfGradient[3] = factor * cdfDiff;
	cdfGradient[2] = cdfGradient[3] * (b_ - x) / (x - a_);
	return cdfGradient;
      }

      /* Get the quantile of the distribution */
      Beta::NumericalPoint Beta::computeQuantile(const NumericalScalar prob) const
      {
	if (prob < 0.0 || prob > 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of [0, 1]";
	// Special case for boarding values
	if (prob == 0.0) return getRange().getLowerBound();
	if (prob == 1.0) return getRange().getUpperBound();
	return NumericalPoint(1, a_ + (b_ - a_) * DistFunc::qBeta(r_, t_ - r_, prob));
      }

      /* Get the roughness, i.e. the L2-norm of the PDF */
      NumericalScalar Beta::getRoughness() const
      {
	NumericalScalar den(SpecFunc::Beta(r_, t_ - r_));
	return SpecFunc::Beta(2.0 * r_ - 1.0, 2.0 * (t_ - r_) - 1.0) / (den * den * (b_ - a_));
      }

      /* Get the mean of the distribution */
      Beta::NumericalPoint Beta::getMean() const throw(NotDefinedException)
      {
	return NumericalPoint(1, a_ + (b_ - a_) * r_ / t_);
      }

      /* Get the standard deviation of the distribution */
      Beta::NumericalPoint Beta::getStandardDeviation() const throw(NotDefinedException)
      {
	return NumericalPoint(1, getSigma());
      }

      /* Get the skewness of the distribution */
      Beta::NumericalPoint Beta::getSkewness() const throw(NotDefinedException)
      {
	return NumericalPoint(1, 2.0 * (t_ - 2.0 * r_) / (t_ + 2.0) * sqrt((t_ + 1.0) / (r_ * (t_ - r_))));
      }

      /* Get the kurtosis of the distribution */
      Beta::NumericalPoint Beta::getKurtosis() const throw(NotDefinedException)
      {
	return NumericalPoint(1, 3.0 * (1.0 + t_) * (2.0 * t_ * t_ + r_ * (t_ - 6.0) * (t_ - r_))/ (r_ * (t_ - r_) * (3.0 + t_) * (2.0 + t_)));
      }

      /* Get the covariance of the distribution */
      Beta::CovarianceMatrix Beta::getCovariance() const throw(NotDefinedException)
      {
	CovarianceMatrix covariance(1);
	NumericalScalar eta((b_ - a_) / t_);
	covariance(0, 0) = eta * eta * r_ * (t_ - r_) / (t_ + 1.0);
	return covariance;
      }

      /* Parameters value and description accessor */
      Beta::NumericalPointWithDescriptionCollection Beta::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	NumericalPointWithDescription point(4);
        Description description(point.getDimension());
	point[0] = r_;
	point[1] = t_;
	point[2] = a_;
	point[3] = b_;
	description[0] = "r";
	description[1] = "t";
	description[2] = "a";
	description[3] = "b";
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }




      /* R accessor */
      void Beta::setR(const NumericalScalar r)
	throw(InvalidArgumentException)
      {
	if (r <= 0.) throw InvalidArgumentException(HERE) << "R MUST be positive";
	r_ = r;
	update();
      }

      NumericalScalar Beta::getR() const
      {
	return r_;
      }


      /* T accessor */
      void Beta::setT(const NumericalScalar t)
	throw(InvalidArgumentException)
      {
	if (t <= r_) throw InvalidArgumentException(HERE) << "T MUST be greater than r, here t=" << t << " and r=" << r_;
	t_ = t;
	update();
      }

      NumericalScalar Beta::getT() const
      {
	return t_;
      }


      /* Mu accessor */
      void Beta::setMuSigma(const NumericalScalar mu,
                            const NumericalScalar sigma)
      {
	t_ = (b_ - mu) * (mu - a_) / (sigma * sigma) - 1;
	r_ = t_ * (mu - a_) / (b_ - a_);
	update();
      }

      NumericalScalar Beta::getMu() const
      {
	return a_ + (b_ - a_) * r_ / t_;
      }

      NumericalScalar Beta::getSigma() const
      {
	return (b_ - a_) / t_ * sqrt(r_ * (t_ - r_) / (t_ + 1.0));
      }


      /* A accessor */
      void Beta::setA(const NumericalScalar a)
      {
	a_ = a;
	computeRange();
	update();
      }

      NumericalScalar Beta::getA() const
      {
	return a_;
      }


      /* B accessor */
      void Beta::setB(const NumericalScalar b)
      {
	b_ = b;
	computeRange();
	update();
      }

      NumericalScalar Beta::getB() const
      {
	return b_;
      }

      /* Method save() stores the object through the StorageManager */
      void Beta::save(const StorageManager::Advocate & adv) const
      {
	NonEllipticalDistribution::save(adv);
	adv.writeValue("r_", r_);
	adv.writeValue("t_", t_);
	adv.writeValue("a_", a_);
	adv.writeValue("b_", b_);
	adv.writeValue("normalizationFactor_", normalizationFactor_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Beta::load(const StorageManager::Advocate & adv)
      {
	NonEllipticalDistribution::load(adv);

	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "r_") r_ = value;
	    if (name == "t_") t_ = value;
	    if (name == "a_") a_ = value;
	    if (name == "b_") b_ = value;
	    if (name == "normalizationFactor_") normalizationFactor_ = value;
	  }
	}
	computeRange();
      }
      
    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
