//                                               -*- C++ -*-
/**
 *  @file  LogNormal.cxx
 *  @brief The LogNormal distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: LogNormal.cxx 929 2008-09-13 20:37:56Z dutka $
 */
#include <cmath>
#include "LogNormal.hxx"
#include "DistFunc.hxx"
#include "SpecFunc.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(LogNormal);

      static Base::Common::Factory<LogNormal> RegisteredFactory("LogNormal");

      /* Default constructor */
      LogNormal::LogNormal()
	: NonEllipticalDistribution("LogNormal"),
	  muLog_(0.0),
	  sigmaLog_(1.0),
	  gamma_(0.0),
	  // 1 / SQRT(2Pi)
	  normalizationFactor_(0.39894228040143267794)
      {
	setDimension(1);
	computeRange();
      }

      /* Default constructor */
      LogNormal::LogNormal(const NumericalScalar arg1,
			   const NumericalScalar arg2,
			   const NumericalScalar gamma,
			   const ParameterSet set)
	throw (InvalidArgumentException)
	: NonEllipticalDistribution("LogNormal"),
	  muLog_(0.0),
	  sigmaLog_(0.0),
	  gamma_(gamma),
	  normalizationFactor_(0.0)
      {
	switch (set) {
	case MUSIGMA_LOG:
	  setMuLog(arg1);
	  setSigmaLog(arg2);
	  break;
	  
	case MUSIGMA:
	  setMuSigma(arg1, arg2);
	  break;

	case MU_SIGMAOVERMU:
	  if (arg1 == 0.0) throw InvalidArgumentException(HERE) << "Error: mu cannot be null in the parameter set (mu, sigmaOverMu)";
	  setMuSigma(arg1, arg1 * arg2);
	  break;

	default:
	  throw InvalidArgumentException(HERE) << "Invalid parameter set argument";

	} /* end switch */
	normalizationFactor_ = 1.0 / (sigmaLog_ * sqrt(2.0 * M_PI));
	setDimension(1);
	computeRange();
      }

      /* Comparison operator */
      Bool LogNormal::operator ==(const LogNormal & other) const
      {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	  // TODO: Write LogNormal::operator ==(...)
	  sameObject = true;
	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String LogNormal::str() const
      {
	OSS oss;
	oss << "class=" << LogNormal::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " muLog=" << muLog_
	    << " sigmaLog=" << sigmaLog_
	    << " gamma=" << gamma_;
	return oss;
      }
  
      /* Virtual constructor */
      LogNormal * LogNormal::clone() const
      {
	return new LogNormal(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void LogNormal::computeRange()
      {
        NumericalPoint lowerBound(1, gamma_);
	const NumericalScalar root(sigmaLog_ * sqrt(sigmaLog_ * sigmaLog_ - 2.0 * (SpecFunc::LogMinNumericalScalar + muLog_ + log(sigmaLog_)) + log(2.0 * M_PI)));
        const NumericalPoint upperBound(1, exp(muLog_ - sigmaLog_ * sigmaLog_ + root));
        const Interval::BoolCollection finiteLowerBound(1, true);
        const Interval::BoolCollection finiteUpperBound(1, false);
        setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }

      /* Get one realization of the distribution */
      LogNormal::NumericalPoint LogNormal::getRealization() const
      {
	return NumericalPoint(1, gamma_ + exp(muLog_ + sigmaLog_ * DistFunc::rNormal()));
      }
     


      /* Get the DDF of the distribution */
      LogNormal::NumericalPoint LogNormal::computeDDF(const NumericalPoint & point) const
      {
	NumericalScalar x(point[0] - gamma_);
	// Here we keep the bound within the special case as the distribution is continuous
	if (x <= 0.0) return NumericalPoint(1, 0.0);
	NumericalScalar v(sigmaLog_ * sigmaLog_);
	return NumericalPoint(1, (muLog_ - log(x) - v) / (v *x) * computePDF(point));
      }


      /* Get the PDF of the distribution */
      NumericalScalar LogNormal::computePDF(const NumericalPoint & point) const
      {
	NumericalScalar x(point[0] - gamma_);
	// Here we keep the bound within the special case as the distribution is continuous
	if (x <= 0.0) return 0.0;
	NumericalScalar logX((log(x) - muLog_) / sigmaLog_);
	return normalizationFactor_ * exp(-0.5 * logX * logX) / x;
      }


      /* Get the CDF of the distribution */
      NumericalScalar LogNormal::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	NumericalScalar x(point[0] - gamma_);
	// Here we keep the bound within the special case as the distribution is continuous
	if (x <= 0.0) return (tail ? 1.0 : 0.0);
	NumericalScalar logX((log(x) - muLog_) / sigmaLog_);
	return DistFunc::pNormal(logX, tail);
      }

      /* Get the PDFGradient of the distribution */
      LogNormal::NumericalPoint LogNormal::computePDFGradient(const NumericalPoint & point) const
      {
	NumericalScalar x(point[0] - gamma_);
	NumericalPoint pdfGradient(3, 0.0);
	// Here we keep the bound within the special case as the distribution is continuous
	if (x <= 0.0) return pdfGradient;
	NumericalScalar logX((log(x) - muLog_) / sigmaLog_);
	NumericalScalar pdf(normalizationFactor_ * exp(-0.5 * logX * logX) / x);
	pdfGradient[0] = pdf * logX / sigmaLog_;
	pdfGradient[1] = pdf * (logX - 1.0) * (logX + 1.0) / sigmaLog_;
	pdfGradient[2] = pdf * (1.0 + logX / sigmaLog_) / x;
	return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      LogNormal::NumericalPoint LogNormal::computeCDFGradient(const NumericalPoint & point) const
      {
	NumericalScalar x(point[0] - gamma_);
	NumericalPoint cdfGradient(3, 0.0);
	// Here we keep the bound within the special case as the distribution is continuous
	if (x <= 0.0) return cdfGradient;
	NumericalScalar logX((log(x) - muLog_) / sigmaLog_);
	NumericalScalar pdf(normalizationFactor_ * exp(-0.5 * logX * logX) / x);
	cdfGradient[0] = -x * pdf;
	cdfGradient[1] = -logX * x * pdf;
	cdfGradient[2] = -pdf;
	return cdfGradient;
      }

      /* Get the quantile of the distribution */
      LogNormal::NumericalPoint LogNormal::computeQuantile(const NumericalScalar prob) const
      {
	if (prob <= 0.0 || prob >= 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of ]0, 1[";
	return NumericalPoint(1, gamma_ + exp(muLog_ + sigmaLog_ * DistFunc::qNormal(prob)));
      }

      /* Get the mean of the distribution */
      LogNormal::NumericalPoint LogNormal::getMean() const throw(NotDefinedException)
      {
	return NumericalPoint(1, getMu());
      }

      /* Get the standard deviation of the distribution */
      LogNormal::NumericalPoint LogNormal::getStandardDeviation() const throw(NotDefinedException)
      {
	return NumericalPoint(1, getSigma());
      }

      /* Get the skewness of the distribution */
      LogNormal::NumericalPoint LogNormal::getSkewness() const throw(NotDefinedException)
      {
	NumericalScalar expSigmaLog2(exp(sigmaLog_ * sigmaLog_));
	return NumericalPoint(1, (expSigmaLog2 + 2.0) * sqrt(expSigmaLog2 - 1.0));
      }

      /* Get the kurtosis of the distribution */
      LogNormal::NumericalPoint LogNormal::getKurtosis() const throw(NotDefinedException)
      {
	NumericalScalar expSigmaLog2(exp(sigmaLog_ * sigmaLog_));
	return NumericalPoint(1, -3.0 + expSigmaLog2 * expSigmaLog2 * (3.0 + expSigmaLog2 * (2.0 + expSigmaLog2)));
      }

      /* Get the covariance of the distribution */
      LogNormal::CovarianceMatrix LogNormal::getCovariance() const throw(NotDefinedException)
      {
	CovarianceMatrix covariance(1);
	NumericalScalar expSigmaLog2(exp(sigmaLog_ * sigmaLog_));
	covariance(0, 0) = expSigmaLog2 * exp(2.0 * muLog_) * (expSigmaLog2 - 1.0);
	return covariance;
      }

      /* Parameters value and description accessor */
      LogNormal::NumericalPointWithDescriptionCollection LogNormal::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	NumericalPointWithDescription point(3);
        Description description(point.getDimension());
	point[0] = muLog_;
	point[1] = sigmaLog_;
	point[2] = gamma_;
	description[0] = "muLog";
	description[1] = "sigmaLog";
	description[2] = "gamma";
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }


      /* Interface specific to LogNormal */

      /* MuLog accessor */
      void LogNormal::setMuLog(const NumericalScalar muLog)
      {
	muLog_ = muLog;
	computeRange();
      }

      NumericalScalar LogNormal::getMuLog() const
      {
	return muLog_;
      }


      /* SigmaLog accessor */
      void LogNormal::setSigmaLog(const NumericalScalar sigmaLog)
	throw (InvalidArgumentException)
      {
	if (sigmaLog <= 0.) throw InvalidArgumentException(HERE) << "SigmaLog MUST be positive, here sigmaLog=" << sigmaLog;
	sigmaLog_ = sigmaLog;
	computeRange();
      }

      NumericalScalar LogNormal::getSigmaLog() const
      {
	return sigmaLog_;
      }

      NumericalScalar LogNormal::getMu() const
      {
	return gamma_ + exp(muLog_ + 0.5 * sigmaLog_ * sigmaLog_);
      }


      /* Sigma accessor */
      void LogNormal::setMuSigma(const NumericalScalar mu,
				 const NumericalScalar sigma)
	throw (InvalidArgumentException)
      {
	if (sigma <= 0.0) throw InvalidArgumentException(HERE) << "Error: sigma must be > 0, here sigma=" << sigma;
	if (mu <= gamma_) throw InvalidArgumentException(HERE) << "Error: mu must be greater than gamma, here mu=" << mu << " and gamma=" << gamma;
	NumericalScalar shift(mu - gamma_);
	NumericalScalar shiftSquared(shift * shift);
	NumericalScalar deltaSquareRoot(sqrt(shiftSquared + sigma * sigma));
	muLog_ = log(shiftSquared / deltaSquareRoot);
	sigmaLog_ = sqrt(2 * log(deltaSquareRoot / shift));
	computeRange();
      }

      NumericalScalar LogNormal::getSigma() const
      {
	NumericalScalar expSigmaLog2(exp(sigmaLog_ * sigmaLog_));
	return exp(muLog_) * sqrt(expSigmaLog2 * (expSigmaLog2 - 1.0));
      }

      /* Gamma accessor */
      void LogNormal::setGamma(const NumericalScalar gamma)
      {
	gamma_ = gamma;
	computeRange();
      }

      NumericalScalar LogNormal::getGamma() const
      {
	return gamma_;
      }

      /* SigmaOverMu accessor */
      NumericalScalar LogNormal::getSigmaOverMu() const
	throw(NotDefinedException)
      {
	NumericalScalar mu(getMu());
	if (mu == 0.0) throw NotDefinedException(HERE) << "Error: trying to get sigmaOverMu with mu equals to zero";
	return getSigma() / mu;
      }

      /* Method save() stores the object through the StorageManager */
      void LogNormal::save(const StorageManager::Advocate & adv) const
      {
	NonEllipticalDistribution::save(adv);
	adv.writeValue("muLog_", muLog_);
	adv.writeValue("sigmaLog_", sigmaLog_);
	adv.writeValue("gamma_", gamma_);
	adv.writeValue("normalizationFactor_", normalizationFactor_);
      }

      /* Method load() reloads the object from the StorageManager */
      void LogNormal::load(const StorageManager::Advocate & adv)
      {
	NonEllipticalDistribution::load(adv);

	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "muLog_") muLog_ = value;
	    if (name == "sigmaLog_") sigmaLog_ = value;
	    if (name == "gamma_") gamma_ = value;
	    if (name == "normalizationFactor_") normalizationFactor_ = value;
	  }
	}
	computeRange();
      }
      
    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
