//                                               -*- C++ -*-
/**
 *  @file  Poisson.cxx
 *  @brief The Poisson distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: Poisson.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "Poisson.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "DistFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Poisson);

      static Base::Common::Factory<Poisson> RegisteredFactory("Poisson");

      /* Default constructor */
      Poisson::Poisson()
	: DiscreteDistribution("Poisson"),
	  lambda_(1.0)
      {
	// We set the dimension of the Poisson distribution
	setDimension( 1 );
      }

      /* Parameters constructor */
      Poisson::Poisson(const NumericalScalar lambda)
	throw (InvalidArgumentException)
	: DiscreteDistribution("Poisson"),
	  lambda_(lambda)
      {
	// We set the dimension of the Poisson distribution
	setDimension( 1 );
      }

      /* Comparison operator */
      Bool Poisson::operator ==(const Poisson & other) const
      {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  if ( (lambda_ == other.lambda_) )
	    sameObject = true;

	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String Poisson::str() const
      {
	OSS oss;
	oss << "class=" << Poisson::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " lambda=" << lambda_;
	return oss;
      }
  
      /* Virtual constructor */
      Poisson * Poisson::clone() const
      {
	return new Poisson(*this);
      }

      /* Get one realization of the distribution */
      Poisson::NumericalPoint Poisson::getRealization() const
      {
	return NumericalPoint(1, DistFunc::rPoisson(lambda_));
      }
     

      /* Get the PDF of the distribution */
      NumericalScalar Poisson::computePDF(const NumericalPoint & point) const
      {
	const NumericalScalar k(point[0]);
	if ((k < -DiscreteDistribution::SupportEpsilon) || (fabs(k - round(k)) > DiscreteDistribution::SupportEpsilon)) return 0.0;
	return exp(k * log(lambda_) - lambda_ - SpecFunc::LnGamma(k + 1.0));
      }


      /* Get the CDF of the distribution */
      NumericalScalar Poisson::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	const NumericalScalar k(point[0]);
	if (k < -DiscreteDistribution::SupportEpsilon) return (tail ? 1.0 : 0.0);
	if (tail) return SpecFunc::GammaInc(floor(k) + 1.0, lambda_) / SpecFunc::Gamma(floor(k) + 1.0);
	return 1.0 - SpecFunc::GammaInc(floor(k) + 1.0, lambda_) / SpecFunc::Gamma(floor(k) + 1.0); // Find a better implementation for large values of k
      }

      /* Get the PDF gradient of the distribution */
      Poisson::NumericalPoint Poisson::computePDFGradient(const NumericalPoint & point) const
      {
	const NumericalScalar k(point[0]);
	NumericalPoint pdfGradient(1, 0.0);
	if ((k < -DiscreteDistribution::SupportEpsilon) || (fabs(k - round(k)) > DiscreteDistribution::SupportEpsilon)) return pdfGradient;
	return NumericalPoint(1, (k - lambda_) * exp((k - 1.0) * log(lambda_) - lambda_ - SpecFunc::LnGamma(k + 1.0)));
      }


      /* Get the CDF gradient of the distribution */
      Poisson::NumericalPoint Poisson::computeCDFGradient(const NumericalPoint & point) const
      {
	const NumericalScalar k(point[0]);
	if (k < -DiscreteDistribution::SupportEpsilon) return NumericalPoint(1, 0.0);
	return NumericalPoint(1, -exp(floor(k) * log(lambda_) - lambda_ - SpecFunc::LnGamma(floor(k) + 1.0)));
      }

      /* Get the quantile of the distribution */
      Poisson::NumericalPoint Poisson::computeQuantile(const NumericalScalar prob) const
      {
	if (prob <= 0.0 || prob >= 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of ]0, 1[";
	UnsignedLong quantile((UnsignedLong)(round(DistFunc::qGamma(lambda_, prob))));
	NumericalScalar pdf(computePDF(NumericalPoint(1, quantile)));
	NumericalScalar cdf(computeCDF(NumericalPoint(1, quantile)));
	if (cdf < prob)
	  {
	    while (cdf < prob)
	      {
		quantile++;
		pdf *= lambda_ / quantile;
		cdf += pdf;
	      }
	    return NumericalPoint(1, quantile);
	  }
	else
	  {
	    while (cdf >= prob)
	      {
		cdf -= pdf;
		if (quantile == 0) return NumericalPoint(1, 0.0);
		quantile--;
		pdf *= quantile / lambda_;
	      }
	    return NumericalPoint(1, quantile + 1);
	  }
      }

      /* Get the mean of the distribution */
      Poisson::NumericalPoint Poisson::getMean() const throw(NotDefinedException)
      {
	return NumericalPoint(1, lambda_);
      }

      /* Get the standard deviation of the distribution */
      Poisson::NumericalPoint Poisson::getStandardDeviation() const throw(NotDefinedException)
      {
	return NumericalPoint(1, sqrt(lambda_));
      }

      /* Get the skewness of the distribution */
      Poisson::NumericalPoint Poisson::getSkewness() const throw(NotDefinedException)
      {
	return NumericalPoint(1, 1.0 / sqrt(lambda_));
      }

      /* Get the kurtosis of the distribution */
      Poisson::NumericalPoint Poisson::getKurtosis() const throw(NotDefinedException)
      {
	return NumericalPoint(1, 3.0 + 1.0 / lambda_);
      }

      /* Get the covariance of the distribution */
      Poisson::CovarianceMatrix Poisson::getCovariance() const throw(NotDefinedException)
      {
	CovarianceMatrix covariance(1);
	covariance(0, 0) = lambda_;
	return covariance;
      }

      /* Parameters value and description accessor */
      Poisson::NumericalPointWithDescriptionCollection Poisson::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	NumericalPointWithDescription point(1);
        Description description(point.getDimension());
	point[0] = lambda_;
	description[0] = "lambda";
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }



      /* Lambda accessor */
      void Poisson::setLambda(const NumericalScalar lambda)
	throw(InvalidArgumentException)
      {
	if (lambda <= 0.0) throw InvalidArgumentException(HERE) << "Lambda must be positive, here lambda=" << lambda;
	lambda_ = lambda;
      }

      /* Lambda accessor */
      NumericalScalar Poisson::getLambda() const
      {
	return lambda_;
      }


      /* Method save() stores the object through the StorageManager */
      void Poisson::save(const StorageManager::Advocate & adv) const
      {
	DiscreteDistribution::save(adv);
	adv.writeValue("lambda_", lambda_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Poisson::load(const StorageManager::Advocate & adv)
      {
	DiscreteDistribution::load(adv);

	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "lambda_") lambda_ = value;
	  }
	}
      }
      
    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
