//                                               -*- C++ -*-
/**
 *  @file  TruncatedNormal.cxx
 *  @brief The TruncatedNormal distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: TruncatedNormal.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "TruncatedNormal.hxx"
#include "RandomGenerator.hxx"
#include "DistFunc.hxx"
#include "SpecFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(TruncatedNormal);

      static Base::Common::Factory<TruncatedNormal> RegisteredFactory("TruncatedNormal");

      /* Default onstructor */
      TruncatedNormal::TruncatedNormal()
	: NonEllipticalDistribution("TruncatedNormal"),
	  mu_(0.0),
          sigma_(1.0),
          a_(-1.0),
          b_(1.0),
	  aNorm_(-1.0),
	  bNorm_(1.0),
	  // phi(-1) / (Phi(1) - Phi(-1))
	  phiANorm_(0.3544374526136036),
	  // phi(1) / (Phi(1) - Phi(-1))
	  phiBNorm_(phiANorm_),
	  // Phi(-1) / (Phi(1) - Phi(-1))
	  PhiANorm_(0.2323973867457722),
	  // Phi(1) / (Phi(1) - Phi(-1))
	  PhiBNorm_(1.2323973867457722),
	  // 1 / (Phi(1) - Phi(-1))
          normalizationFactor_(1.464794773491545)
      {
	setDimension(1);
	computeRange();
      }

      /* Default onstructor */
      TruncatedNormal::TruncatedNormal(const NumericalScalar mu,
				       const NumericalScalar sigma,
				       const NumericalScalar a,
				       const NumericalScalar b)
	throw (InvalidArgumentException)
	: NonEllipticalDistribution("TruncatedNormal"),
	  mu_(mu),
          sigma_(0.0),
          a_(a),
          b_(b),
	  aNorm_(0.0),
	  bNorm_(0.0),
	  phiANorm_(0.0),
	  phiBNorm_(0.0),
	  PhiANorm_(0.0),
	  PhiBNorm_(0.0),
          normalizationFactor_(0.0)
      {
	if (sigma <= 0.0) throw InvalidArgumentException(HERE) << "Error: cannot build a TruncatedNormal distribution with sigma <=0. Here, sigma=" << sigma;
	if (a >= b) throw InvalidArgumentException(HERE) << "Error: cannot build a TruncatedNormal distribution with a >= b. Here, a=" << a << " and b=" << b;
	setSigma(sigma);
	setDimension(1);
	const NumericalScalar iSigma(1.0 / sigma_);
	aNorm_ = (a_ - mu_) * iSigma;
	bNorm_ = (b_ - mu_) * iSigma;
	PhiANorm_ = DistFunc::pNormal(aNorm_);
	PhiBNorm_ = DistFunc::pNormal(bNorm_);
	NumericalScalar denominator(PhiBNorm_ - PhiANorm_);
	// If left tail truncature, use tail CDF to compute the normalization factor
	if (aNorm_ > 0.0)
	  {
	    denominator = DistFunc::pNormal(aNorm_, true) - DistFunc::pNormal(bNorm_, true);
	  }
	if (denominator <= 0.0) throw InvalidArgumentException(HERE) << "Error: the truncation interval has a too small measure. Here, measure=" << denominator;
	normalizationFactor_ = 1.0 / denominator;
	phiANorm_ = SpecFunc::ISQRT2PI * exp(-0.5 * aNorm_ * aNorm_);
	phiBNorm_ = SpecFunc::ISQRT2PI * exp(-0.5 * bNorm_ * bNorm_);
	computeRange();
      }



      /* Comparison operator */
      Bool TruncatedNormal::operator ==(const TruncatedNormal & other) const
      {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	  // TODO: Write TruncatedNormal::operator ==(...)
	  sameObject = (mu_ == other.mu_) && (sigma_ == other.sigma_) &&
	    (a_ == other.a_) && (b_ == other.b_);
	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String TruncatedNormal::str() const
      {
	OSS oss;
	oss << "class=" << TruncatedNormal::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " mu=" << mu_
	    << " sigma=" << sigma_
	    << " a=" << a_
	    << " b=" << b_;
	return oss;
      }
  
      /* Virtual constructor */
      TruncatedNormal * TruncatedNormal::clone() const
      {
	return new TruncatedNormal(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void TruncatedNormal::computeRange()
      {
	setRange(Interval(a_, b_));
      }


      /* Get one realization of the distribution */
      TruncatedNormal::NumericalPoint TruncatedNormal::getRealization() const
      {
	/* Find a better method, e.g.
 	   Algorithm from John Geweke, "Efficient Simulation from the Multivariate Normal and Student-t Distributions Subject to Linear Constraints and the Evaluation of Constraint Probabilities", communication at the meeting "Computer Science and Statistics: the Twenty-Third Symposium on the Interface", April 22-24, 1991. */
	// If the truncation is strong, use CDF inversion, else use rejection. The cut-off must balance the cost of the two methods
	if (PhiBNorm_ - PhiANorm_ < 0.25) return computeQuantile(RandomGenerator::Generate());
	NumericalScalar value;
	do
	  {
	    value = DistFunc::rNormal();
	  } while ((value < aNorm_) || (value >= bNorm_));
	return NumericalPoint(1, mu_ + sigma_ * value);
      }


      /* Get the DDF of the distribution */
      TruncatedNormal::NumericalPoint TruncatedNormal::computeDDF(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0]);
	if ((x <= a_) || (x > b_)) return NumericalPoint(1, 0.0);
	const NumericalScalar iSigma(1.0 / sigma_);
	const NumericalScalar xNorm((x - mu_) * iSigma);
	return NumericalPoint(1, -normalizationFactor_ * xNorm * SpecFunc::ISQRT2PI * exp(-0.5 * xNorm * xNorm) * iSigma * iSigma);
      }


      /* Get the PDF of the distribution */
      NumericalScalar TruncatedNormal::computePDF(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0]);
	if ((x <= a_) || (x > b_)) return 0.0;
	const NumericalScalar iSigma(1.0 / sigma_);
	const NumericalScalar xNorm((x - mu_) * iSigma);
	return normalizationFactor_ * exp(-0.5 * xNorm * xNorm) * SpecFunc::ISQRT2PI * iSigma;
      }


      /* Get the CDF of the distribution */
      NumericalScalar TruncatedNormal::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	const NumericalScalar x(point[0]);
	if (x <= a_) return (tail ? 1.0 : 0.0);
	if (x > b_) return (tail ? 0.0 : 1.0);
	// FIXME
	return normalizationFactor_ * (DistFunc::pNormal((x - mu_) / sigma_) - PhiANorm_);
      }

      /* Get the PDFGradient of the distribution */
      TruncatedNormal::NumericalPoint TruncatedNormal::computePDFGradient(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0]);
	NumericalPoint pdfGradient(4, 0.0);
	if ((x <= a_) || (x > b_)) return pdfGradient;
	const NumericalScalar iSigma(1.0 / sigma_);
	const NumericalScalar xNorm((x - mu_) * iSigma);
	const NumericalScalar iDenom(normalizationFactor_ * iSigma);
	const NumericalScalar iDenom2(iDenom * iDenom);
	const NumericalScalar factPhiXNorm(exp(-0.5 * xNorm * xNorm) * SpecFunc::ISQRT2PI * iDenom2);
	pdfGradient[0] = factPhiXNorm * (xNorm * (PhiBNorm_ - PhiANorm_) + phiBNorm_ - phiANorm_);
	pdfGradient[1] = factPhiXNorm * ((xNorm * xNorm - 1.0) * (PhiBNorm_ - PhiANorm_) + bNorm_ * phiBNorm_ - aNorm_ * phiANorm_);
	pdfGradient[2] = factPhiXNorm * phiANorm_;
	pdfGradient[3] = -factPhiXNorm * phiBNorm_;
	return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      TruncatedNormal::NumericalPoint TruncatedNormal::computeCDFGradient(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0]);
	NumericalPoint cdfGradient(4, 0.0);
	if ((x <= a_) || (x > b_)) return cdfGradient;
	const NumericalScalar iSigma(1.0 / sigma_);
	const NumericalScalar xNorm((x - mu_) * iSigma);
	const NumericalScalar iDenom(normalizationFactor_ * normalizationFactor_ * iSigma);
	const NumericalScalar phiXNorm(exp(-0.5 * xNorm * xNorm) * SpecFunc::ISQRT2PI);
	const NumericalScalar PhiXNorm(DistFunc::pNormal(xNorm));
	cdfGradient[0] = (phiANorm_ * PhiBNorm_ - PhiANorm_ * phiBNorm_ + phiXNorm * PhiANorm_ - PhiXNorm * phiANorm_ + phiBNorm_ * PhiXNorm - PhiBNorm_ * phiXNorm) * iDenom;
	cdfGradient[1] = (phiANorm_ * aNorm_ * PhiBNorm_ - PhiANorm_ * phiBNorm_ * bNorm_ + phiXNorm * xNorm * PhiANorm_ - PhiXNorm * phiANorm_ * aNorm_ + phiBNorm_ * bNorm_ * PhiXNorm - PhiBNorm_ * phiXNorm * xNorm) * iDenom;
	cdfGradient[2] = phiANorm_ * (PhiXNorm - PhiBNorm_) * iDenom;
	cdfGradient[3] = phiBNorm_ * (PhiANorm_ - PhiXNorm) * iDenom;
	return cdfGradient;
      }

      /* Get the quantile of the distribution */
      TruncatedNormal::NumericalPoint TruncatedNormal::computeQuantile(const NumericalScalar prob) const
      {
	if (prob < 0.0 || prob > 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of [0, 1]";
        // Special case for boarding values
        if (prob == 0.0) return getRange().getLowerBound();
        if (prob == 1.0) return getRange().getUpperBound();
	return NumericalPoint(1, mu_ + sigma_ * DistFunc::qNormal(PhiANorm_ + prob / normalizationFactor_));
      }

      /* Get the mean of the distribution */
      TruncatedNormal::NumericalPoint TruncatedNormal::getMean() const throw(NotDefinedException)
      {
	return NumericalPoint(1, mu_ - sigma_ * (phiBNorm_ - phiANorm_) * normalizationFactor_);
      }

      /* Get the standard deviation of the distribution */
      TruncatedNormal::NumericalPoint TruncatedNormal::getStandardDeviation() const throw(NotDefinedException)
      {
	const NumericalScalar ratio((phiBNorm_ - phiANorm_) * normalizationFactor_);
	return NumericalPoint(1, sigma_ * sqrt(1.0 - (bNorm_ * phiBNorm_ - aNorm_ * phiANorm_) * normalizationFactor_ - ratio * ratio));
      }

      /* Get the skewness of the distribution */
      TruncatedNormal::NumericalPoint TruncatedNormal::getSkewness() const throw(NotDefinedException)
      {
	const NumericalScalar ratio((phiBNorm_ - phiANorm_) * normalizationFactor_);
	const NumericalScalar ratio2(ratio * ratio);
	const NumericalScalar crossTerm1((bNorm_ * phiBNorm_ - aNorm_ * phiANorm_) * normalizationFactor_);
	const NumericalScalar crossTerm2((bNorm_ * bNorm_ * phiBNorm_ - aNorm_ * aNorm_ * phiANorm_) * normalizationFactor_);
	return NumericalPoint(1, (-2.0 * ratio * ratio2 - 3.0 * ratio * crossTerm1 + ratio - crossTerm2) / pow(1.0 - crossTerm1 - ratio2, 1.5));
      }

      /* Get the kurtosis of the distribution */
      TruncatedNormal::NumericalPoint TruncatedNormal::getKurtosis() const throw(NotDefinedException)
      {
	const NumericalScalar ratio((phiBNorm_ - phiANorm_) * normalizationFactor_);
	const NumericalScalar ratio2(ratio * ratio);
        const NumericalScalar crossTerm1((bNorm_ * phiBNorm_ - aNorm_ * phiANorm_) * normalizationFactor_);
	const NumericalScalar crossTerm2((bNorm_ * bNorm_ * phiBNorm_ - aNorm_ * aNorm_ * phiANorm_) * normalizationFactor_);
	const NumericalScalar crossTerm3((bNorm_ * bNorm_ * bNorm_ * phiBNorm_ - aNorm_ * aNorm_ * aNorm_ * phiANorm_) * normalizationFactor_);
	return NumericalPoint(1, (3.0 - 3.0 * ratio2 * ratio2 - 6.0 * ratio2 * crossTerm1 - 2.0 * ratio * (ratio + 2.0 * crossTerm2) - 3.0 * crossTerm1 - crossTerm3) / pow(1.0 - crossTerm1 - ratio2, 2.0));
      }

      /* Get the covariance of the distribution */
      TruncatedNormal::CovarianceMatrix TruncatedNormal::getCovariance() const throw(NotDefinedException)
      {
	CovarianceMatrix covariance(1);
	const NumericalScalar ratio((phiBNorm_ - phiANorm_) * normalizationFactor_);
	covariance(0, 0) = sigma_ * sigma_ * (1.0 - (bNorm_ * phiBNorm_ - aNorm_ * phiANorm_) * normalizationFactor_ - ratio * ratio);
	return covariance;
      }

      /* Parameters value and description accessor */
      TruncatedNormal::NumericalPointWithDescriptionCollection TruncatedNormal::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	NumericalPointWithDescription point(4);
        Description description(point.getDimension());
	point[0] = mu_;
	point[1] = sigma_;
	point[2] = a_;
	point[3] = b_;
	description[0] = "mu";
	description[1] = "sigma";
	description[2] = "a";
	description[3] = "b";
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }




      /* Mu accessor */
      void TruncatedNormal::setMu(const NumericalScalar mu)
      {
	mu_ = mu;
      }

      NumericalScalar TruncatedNormal::getMu() const
      {
	return mu_;
      }


      /* Sigma accessor */
      void TruncatedNormal::setSigma(const NumericalScalar sigma)
	throw (InvalidArgumentException)
      {
	if (sigma <= 0.) throw InvalidArgumentException(HERE) << "Sigma MUST be positive";
	sigma_ = sigma;
      }

      NumericalScalar TruncatedNormal::getSigma() const
      {
	return sigma_;
      }


      /* A accessor */
      void TruncatedNormal::setA(const NumericalScalar a)
      {
	a_ = a;
	computeRange();
      }

      NumericalScalar TruncatedNormal::getA() const
      {
	return a_;
      }


      /* B accessor */
      void TruncatedNormal::setB(const NumericalScalar b)
      {
	b_ = b;
	computeRange();
      }

      NumericalScalar TruncatedNormal::getB() const
      {
	return b_;
      }

      /* Method save() stores the object through the StorageManager */
      void TruncatedNormal::save(const StorageManager::Advocate & adv) const
      {
	NonEllipticalDistribution::save(adv);
	adv.writeValue("mu_", mu_);
	adv.writeValue("sigma_", sigma_);
	adv.writeValue("a_", a_);
	adv.writeValue("b_", b_);
	adv.writeValue("aNorm_", aNorm_);
	adv.writeValue("bNorm_", bNorm_);
	adv.writeValue("phiANorm_", phiANorm_);
	adv.writeValue("phiBNorm_", phiBNorm_);
	adv.writeValue("PhiANorm_", PhiANorm_);
	adv.writeValue("PhiBNorm_", PhiBNorm_);
	adv.writeValue("normalizationFactor_", normalizationFactor_);
      }

      /* Method load() reloads the object from the StorageManager */
      void TruncatedNormal::load(const StorageManager::Advocate & adv)
      {
	NonEllipticalDistribution::load(adv);

	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "mu_") mu_ = value;
	    if (name == "sigma_") sigma_ = value;
	    if (name == "a_") a_ = value;
	    if (name == "b_") b_ = value;
	    if (name == "aNorm_") aNorm_ = value;
	    if (name == "bNorm_") bNorm_ = value;
	    if (name == "phiANorm_") phiANorm_ = value;
	    if (name == "phiBNorm_") phiBNorm_ = value;
	    if (name == "PhiANorm_") PhiANorm_ = value;
	    if (name == "PhiBNorm_") PhiBNorm_ = value;
	    if (name == "normalizationFactor_") normalizationFactor_ = value;
	  }
	}
	computeRange();
      }
      
      
    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
