//                                               -*- C++ -*-
/**
 *  @file  Uniform.hxx
 *  @brief The Uniform distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: Uniform.hxx 929 2008-09-13 20:37:56Z dutka $
 */
#ifndef OPENTURNS_UNIFORM_HXX
#define OPENTURNS_UNIFORM_HXX

#include "EllipticalDistribution.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Uniform
       *
       * The Uniform distribution.
       */
      class Uniform 
	: public Model::EllipticalDistribution
      {
	CLASSNAME;
      public:

	typedef Model::EllipticalDistribution                     EllipticalDistribution; // required by SWIG
	typedef EllipticalDistribution::NumericalPoint            NumericalPoint;
	typedef EllipticalDistribution::NumericalSample           NumericalSample;
	typedef EllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
	typedef EllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
	typedef EllipticalDistribution::NotDefinedException       NotDefinedException;
	typedef EllipticalDistribution::InvalidArgumentException  InvalidArgumentException;
	typedef EllipticalDistribution::StorageManager            StorageManager;

	/** Default constructor */
        Uniform();

	/** Parameters constructor */
        Uniform(const NumericalScalar a,
		const NumericalScalar b);


	/** Comparison operator */
	Bool operator ==(const Uniform & other) const;

	/** String converter */
	String str() const;



	/* Interface inherited from Distribution */

	/** Virtual constructor */
	Uniform * clone() const;

	/** Get one realization of the distribution */
	NumericalPoint getRealization() const;

	/** Get the DDF of the distribution, i.e. the gradient of its PDF w.r.t. point */
	using EllipticalDistribution::computeDDF;
	NumericalPoint computeDDF(const NumericalPoint & point) const;

	/** Get the PDF of the distribution, i.e. P(point < X < point+dx) = PDF(point)dx + o(dx) */
	using EllipticalDistribution::computePDF;
	NumericalScalar computePDF(const NumericalPoint & point) const;

	/** Get the CDF of the distribution, i.e. P(X <= point) = CDF(point) */
	using EllipticalDistribution::computeCDF;
	NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

	/** Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
	NumericalComplex computeCharacteristicFunction(const NumericalPoint & point) const;

	/** Get the gradient of the PDF w.r.t the parameters of the distribution */
	NumericalPoint computePDFGradient(const NumericalPoint & point) const;

	/** Get the gradient of the CDF w.r.t the parameters of the distribution */
	NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

	/** Get the quantile of the distribution, i.e the value Xp such that P(X <= Xp) = prob */
	NumericalPoint computeQuantile(const NumericalScalar prob) const;
	
	/** Get the roughness, i.e. the L2-norm of the PDF */
	NumericalScalar getRoughness() const;

	/** Get the skewness of the distribution */
	NumericalPoint getSkewness() const throw(NotDefinedException);

	/** Get the kurtosis of the distribution */
	NumericalPoint getKurtosis() const throw(NotDefinedException);

	/** Compute the density generator of the ellipticalal generator, i.e.
	 *  the scalar function phi such that the density of the distribution can
	 *  be written as p(x) = phi(t(x-mu)R^(-1)(x-mu))                      */
	NumericalScalar computeDensityGenerator(const NumericalScalar betaSquare) const;

	/** Compute the derivative of the density generator */
	NumericalScalar computeDensityGeneratorDerivative(const NumericalScalar betaSquare) const;

	/** Compute the seconde derivative of the density generator */
	NumericalScalar computeDensityGeneratorSecondDerivative(const NumericalScalar betaSquare) const;

	/** Parameters value and description accessor */
	NumericalPointWithDescriptionCollection getParametersCollection() const;

	/* Interface specific to Uniform */

	/** A accessor */
	void setA(const NumericalScalar a);
	NumericalScalar getA() const;

	/** B accessor */
	void setB(const NumericalScalar b);
	NumericalScalar getB() const;

	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);


      protected:
    
    
      private:

	/** Quantile computation for dimension=1 */
	NumericalScalar computeScalarQuantile(const NumericalScalar prob, const NumericalScalar initialGuess = 0.0, const NumericalScalar initialStep = 1.0) const;
    
	/** Compute the numerical range of the distribution given the parameters values */
	void computeRange();

	/** The main parameter set of the distribution */
	NumericalScalar a_;
	NumericalScalar b_;

      }; /* class Uniform */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_UNIFORM_HXX */
