//                                               -*- C++ -*-
/**
 *  @file  ContinuousDistribution.cxx
 *  @brief Abstract top-level class for Continuous distributions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: ContinuousDistribution.cxx 929 2008-09-13 20:37:56Z dutka $
 */

#include <cmath>
#include "ContinuousDistribution.hxx"
#include "Collection.hxx"
#include "Distribution.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      CLASSNAMEINIT(ContinuousDistribution);

      /* Default constructor */
      ContinuousDistribution::ContinuousDistribution(const String & name)
	: UsualDistribution(name)
      {
	// Initialize any other class members here
	// At last, allocate memory space if needed, but go to destructor to free it
      }

      /* Destructor */
      ContinuousDistribution::~ContinuousDistribution() {
	// Remember to free any allocated memory space
	// Remember NOT to throw any exception here
      }

      /* Comparison operator */
      Bool ContinuousDistribution::operator ==(const ContinuousDistribution & other) const {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String ContinuousDistribution::str() const {
	OSS oss;
	oss << "class=" << ContinuousDistribution::GetClassName();
	return oss;
      }
  
      /* Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
      NumericalScalar ContinuousDistribution::computeConditionalCDF(const NumericalScalar x, const NumericalPoint & y) const
      {
	const UnsignedLong conditioningDimension(y.getDimension());
	if (conditioningDimension >= getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional CDF with a conditioning point of dimension greater or equal to the distribution dimension.";
	// Special case for no conditioning or independent copula
	if ((conditioningDimension == 0) || (hasIndependentCopula())) return getMarginal(conditioningDimension)->computeCDF(x);
	// General case
	Indices conditioning(conditioningDimension);
	for (UnsignedLong i = 0; i < conditioningDimension; ++i)
	  {
	    conditioning[i] = i;
	  }
	Indices conditioned(conditioning);
	conditioned.add(conditioningDimension);
	Implementation conditioningDistribution(getMarginal(conditioning));
	const NumericalScalar pdfConditioning(conditioningDistribution->computePDF(y));
	if (pdfConditioning <= 0.0) return 0.0;
	// Compute the conditional CDF by numerical integration of the conditional PDF. We use a fixed point Gauss integration.
	const NumericalSample nodesAndWeights(getGaussNodesAndWeights());
	const UnsignedLong numberOfNodes(nodesAndWeights.getDimension());
	NumericalScalar cdfConditioned(0.0);
	Implementation marginal(getMarginal(conditioningDimension));
	const NumericalScalar dispersion(marginal->getDispersionIndicator());
	const NumericalScalar position(marginal->getPositionIndicator());
	const NumericalScalar lowerBound(std::max(std::min(position, x) - 3.0 * dispersion, marginal->getRange().getLowerBound()[0]));
	const NumericalScalar halfLength(0.5 * (std::min(x, marginal->getRange().getUpperBound()[0]) - lowerBound));
	Implementation conditionedDistribution(getMarginal(conditioned));
	cdfEpsilon_ = 0.0;
	// Map [x-8*s,x] into [-1,1]
	for (UnsignedLong i = 0; i < numberOfNodes; ++i)
	  {
	    const NumericalScalar w(nodesAndWeights[1][i]);
	    const NumericalScalar xi(nodesAndWeights[0][i]);
	    NumericalPoint z(y);
	    z.add(lowerBound + (1.0 + xi) * halfLength);
	    const NumericalScalar pdfConditioned(conditionedDistribution->computePDF(z));
	    // Accumulate CDF value
	    cdfConditioned += w * pdfConditioned;
	    // Compute an approximation of the rounding error
	    cdfEpsilon_ += w * conditionedDistribution->getPDFEpsilon();
	  }
	cdfConditioned *= halfLength;
	cdfEpsilon_ = cdfEpsilon_ * halfLength + conditioningDistribution->getPDFEpsilon();
	return std::max(std::min(cdfConditioned / pdfConditioning, 1.0-1e-7), 1e-7);
      }
      
      /* Tell if the distribution is Continuous */
      Bool ContinuousDistribution::isContinuous() const
      {
	return true;
      }
    
    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
