//                                               -*- C++ -*-
/**
 *  @file  EllipticalDistribution.hxx
 *  @brief Abstract top-level class for elliptical distributions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: EllipticalDistribution.hxx 929 2008-09-13 20:37:56Z dutka $
 */
#ifndef OPENTURNS_ELLIPTICALDISTRIBUTIONIMPLEMENTATION_HXX
#define OPENTURNS_ELLIPTICALDISTRIBUTIONIMPLEMENTATION_HXX

#include "ContinuousDistribution.hxx"
#include "CorrelationMatrix.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {


      /**
       * @class EllipticalDistribution
       *
       * A subclass for elliptical usual distributions.
       */
      class EllipticalDistribution 
	: public ContinuousDistribution
      {
	CLASSNAME;

      public:

	// Numerical precision for computing the quantile
	typedef ContinuousDistribution::Implementation                          Implementation;
	typedef ContinuousDistribution::InvalidArgumentException                InvalidArgumentException;
	typedef ContinuousDistribution::InvalidDimensionException               InvalidDimensionException;
	typedef ContinuousDistribution::NotDefinedException                     NotDefinedException;
	typedef ContinuousDistribution::NumericalPoint                          NumericalPoint;
	typedef ContinuousDistribution::NumericalSample                         NumericalSample;
	typedef ContinuousDistribution::CovarianceMatrix                        CovarianceMatrix;
	typedef ContinuousDistribution::Indices                                 Indices;
	typedef ContinuousDistribution::IsoProbabilisticTransformation          IsoProbabilisticTransformation;
	typedef ContinuousDistribution::InverseIsoProbabilisticTransformation   InverseIsoProbabilisticTransformation;
	typedef Base::Stat::CorrelationMatrix                                   CorrelationMatrix;
	typedef Base::Type::SymmetricMatrix                                     SymmetricMatrix;
	typedef Base::Type::SquareMatrix                                        SquareMatrix;
	typedef ContinuousDistribution::NumericalPointWithDescriptionCollection NumericalPointWithDescriptionCollection;
	typedef ContinuousDistribution::StorageManager                          StorageManager;

	/** Default constructor */
	explicit EllipticalDistribution(const NumericalPoint & mean,
					const NumericalPoint & sigma,
					const CorrelationMatrix & R,
					const String & name = OT::DefaultName);

	/** Parameter constructor */
	explicit EllipticalDistribution(const String & name = OT::DefaultName);


	/** Virtual copy constructor */
	virtual EllipticalDistribution * clone() const;

	/** Destructor */
	virtual ~EllipticalDistribution();

	/** Comparison operator */
	Bool operator ==(const EllipticalDistribution & other) const;

	/** String converter */
	String str() const;

	/** Tell if the distribution is elliptical */
	virtual Bool isElliptical() const;

	/** Tell if the distribution has elliptical copula */
	virtual Bool hasEllipticalCopula() const;

	/** Get the DDF of the distribution */
	using ContinuousDistribution::computeDDF;
	virtual NumericalPoint computeDDF(const NumericalPoint & point) const;

	/** Get the PDF of the distribution */
	using ContinuousDistribution::computePDF;
	virtual NumericalScalar computePDF(const NumericalPoint & point) const;

	/** Get the PDF gradient of the distribution */
	virtual NumericalPoint computePDFGradient(const NumericalPoint & point) const;

	/** Compute the density generator of the elliptical distribution, i.e.
	 *  the function phi such that the density of the distribution can
	 *  be written as p(x) = phi(t(x-mu)R^{-1}(x-mu))                      */
	virtual NumericalScalar computeDensityGenerator(const NumericalScalar betaSquare) const;

	/** Compute the derivative of the density generator */
	virtual NumericalScalar computeDensityGeneratorDerivative(const NumericalScalar betaSquare) const;

	/** Compute the second derivative of the density generator */
	virtual NumericalScalar computeDensityGeneratorSecondDerivative(const NumericalScalar betaSquare) const;

	/** Get the quantile of the distribution */
	virtual NumericalPoint computeQuantile(const NumericalScalar prob) const;

	/** Mean point accessor */
	void setMean(const NumericalPoint & mean)
	  throw(InvalidArgumentException);
	/** Mean point accessor */
	NumericalPoint getMean() const throw(NotDefinedException);

	/** Sigma vector accessor */
	void setSigma(const NumericalPoint & sigma)
	  throw(InvalidArgumentException);
	/** Sigma vector accessor */
	NumericalPoint getSigma() const;

	/** Get the standard deviation of the distribution */
	NumericalPoint getStandardDeviation() const throw(NotDefinedException);

	/** Correlation matrix accessor */
	void setCorrelationMatrix(const CorrelationMatrix & R)
	  throw(InvalidArgumentException);
	/** Correlation matrix accessor */
	CorrelationMatrix getCorrelationMatrix() const;

	/** Get the covariance of the distribution */
	CovarianceMatrix getCovariance() const throw(NotDefinedException);

	/** Normalize the given point u_i = (x_i - mu_i) / sigma_i */
	NumericalPoint normalize(const NumericalPoint & x) const;

	/** Denormalize the given point x_i = mu_i + sigma_i * x_i */
	NumericalPoint denormalize(const NumericalPoint & u) const;

	/** Inverse correlation matrix accessor */
	SquareMatrix getInverseCorrelation() const;

	/** Cholesky factor of the correlation matrix accessor */
	SquareMatrix getCholesky() const;

	/** Inverse of the Cholesky factor of the correlation matrix accessor */
	SquareMatrix getInverseCholesky() const;

	/** Get the isoprobabilist transformation */
	IsoProbabilisticTransformation getIsoProbabilisticTransformation() const;

	/** Get the inverse isoprobabilist transformation */
	InverseIsoProbabilisticTransformation getInverseIsoProbabilisticTransformation() const;

	/** Parameters value and description accessor */
	NumericalPointWithDescriptionCollection getParametersCollection() const;

	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);	

      protected:
    
	/** Compute the numerical range of the distribution given the parameters values */
	virtual void computeRange();

 	/** The correlation matrix (Rij) of the distribution */
	mutable CorrelationMatrix R_;

	/** The sigma vector of the distribution */
	mutable NumericalPoint sigma_;

	/** The mean point of the distribution */
	mutable NumericalPoint mean_;

	/** The covariance matrix of the distribution = Diag(sigma_) * R_ * Diag(sigma_) */
	mutable CovarianceMatrix covariance_;

	/** The inverse of the correlation matrix of the distribution */
	SymmetricMatrix inverseR_;

	/** The Cholesky factor of the covariance matrix covariance_ = cholesky_ * cholesky_.transpose() */
	SquareMatrix cholesky_;

	/** The inverse Cholesky factor of the covariance matrix */
	SquareMatrix inverseCholesky_;

	/** The normalization factor of the distribution */
	NumericalScalar normalizationFactor_;

      private:
	/** Compute the value of the auxiliary attributes */
	void update();

      }; /* class EllipticalDistribution */


    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_ELLIPTICALDISTRIBUTIONIMPLEMENTATION_HXX */
