//                                               -*- C++ -*-
/**
 *  @file  poutre.cxx
 *  @brief 
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-22 11:34:11 +0200 (lun 22 sep 2008) $
 *  Id:      $Id: poutre.cxx 941 2008-09-22 09:34:11Z dutka $
 */
#include <iostream>
#include <cmath>
#include <cstdlib>
#include "WrapperCommon.h"
#include <WrapperMacros.h>

/* TODO : Add error treatment when something goes wrong into one wrapper function. We CAN'T use exceptions */

/*
 * Here is some functions absolutely internal to the wrapper. They are not exported or seen 
 * by the platform, so you can do anything you want.
 */

namespace WrapperInternals {

  struct internalState {
    long numberOfCalls;
    char * currentWorkingDirectory;
    struct WrapperInformation info;
    struct WrapperExchangedData * p_exchangedData;
  };

  static inline
  void internalStateIncrement(struct internalState * p_internalState)
  {
    if (p_internalState) p_internalState->numberOfCalls++;
  }

  static inline
  long internalStateGetNumberOfCalls(struct internalState * p_internalState)
  {
    return (p_internalState ? p_internalState->numberOfCalls : -1);
  }

  static inline
  void internalStatePrint(const char * functionName, struct internalState * p_internalState)
  {
    std::cerr << "(" << functionName << ") NumberOfCalls = " << internalStateGetNumberOfCalls(p_internalState) << std::endl;
    std::cerr << "(" << functionName << ") info = {inSize_=" << p_internalState->info.inSize_
	      << ", outSize_=" << p_internalState->info.outSize_
	      << ", currentWorkingDirectory=" << p_internalState->currentWorkingDirectory
	      << "}" << std::endl;
    dbg_printWrapperExchangedData(functionName, p_internalState->p_exchangedData);
  }

  static inline
  void commonInternalStateCreation(void ** p_p_state, const struct WrapperExchangedData * p_exchangedData)
  {
    *p_p_state = new struct WrapperInternals::internalState;

    struct WrapperInternals::internalState * p_internalState = static_cast<struct WrapperInternals::internalState *>(*p_p_state);

    copyWrapperExchangedData(& (p_internalState->p_exchangedData), p_exchangedData);
    p_internalState->numberOfCalls           = 0;
    p_internalState->info.inSize_            = getNumberOfVariables(p_internalState->p_exchangedData, WRAPPER_IN);
    p_internalState->info.outSize_           = getNumberOfVariables(p_internalState->p_exchangedData, WRAPPER_OUT);
    p_internalState->currentWorkingDirectory = getCurrentWorkingDirectory();
  }

  static inline
  void commonInternalStateDestruction(void * p_state)
  {
    struct WrapperInternals::internalState * p_internalState = static_cast<struct WrapperInternals::internalState *>(p_state);
    free(p_internalState->currentWorkingDirectory);
    freeWrapperExchangedData(p_internalState->p_exchangedData);
    delete p_internalState;
  }

  static inline
  void commonGetInfo(void * p_state, struct WrapperInformation * p_info)
  {
    *p_info = static_cast<struct WrapperInternals::internalState *>(p_state)->info;
  }

} /* namespace WrapperInternals */



/*
 *  This is the declaration of function named 'compute_deviation' into the wrapper.
 */
  

extern "C" {


  /*
******************************************************************************************
*                                                                                        *
*                             compute_deviation function                                 *
*                                                                                        *
******************************************************************************************
*/

  /* The name of the wrapper's functions is defined in WRAPPERNAME macro */
#define WRAPPERNAME compute_deviation


  /* The createState function is optional */
  FUNC_CREATESTATE( WRAPPERNAME ,
		    {
		      WrapperInternals::commonInternalStateCreation(p_p_state, p_exchangedData);
		    } )
    
  /* The deleteState function is optional */
    FUNC_DELETESTATE( WRAPPERNAME ,
		      {
			WrapperInternals::commonInternalStateDestruction(p_state);
		      } )





  /* The wrapper information informs the NumericalMathFunction object that loads the wrapper of the
   * signatures of the wrapper functions. In particular, it hold the size of the input NumericalPoint
   * (inSize_) and of the output NumericalPoint (outSize_).
   * Those information are also used by the gradient and hessian functions to set the correct size
   * of the returned matrix and tensor.
   */
    FUNC_INFO( WRAPPERNAME ,
	       {
		 WrapperInternals::commonGetInfo(p_state, p_info);
	       } )


  /* Any function declared into the wrapper MUST declare three actual function prefixed with
   * 'init_', 'exec_' and 'finalize_' folowed by the name of the function, here 'compute_deviation'.
   *
   * The 'init_' function is only called once when the NumericalMathFunction object is created.
   * It allows the wrapper to set some internal state, read some external file, prepare the function
   * to run, etc. It takes only one argument, the internal state as created by the 
   *
   * The 'exec_' function is intended to execute what the wrapper is done for: compute an mathematical
   * function or anything else. It takes the internal state pointer as its first argument, the input
   * NumericalPoint pointer as the second and the output NumericalPoint pointer as the third.
   *
   * The 'finalize_' function is only called once when the NumericalMathFunction object is destroyed.
   * It allows the wrapper to flush anything before unloading.
   */


  /**
   * Initialization function
   * This function is called once just after the wrapper is loaded to initialize
   * it, ie create a temparary subdirectory (remember that the wrapper may be called
   * concurrently), store exchanged data in some internal repository, do some
   * pre-computational operation, etc.
   */
    FUNC_INIT( WRAPPERNAME , {} )




  /**
   * Execution function
   * This function is called by the platform to do the real work of the wrapper. It may be
   * called concurrently, so be aware of not using shared or global data not protected by
   * a critical section.
   * This function has a mathematical meaning. It operates on one vector (aka point) and
   * returns another vector.
   */
    FUNC_EXEC( WRAPPERNAME, 
	       {
		 struct WrapperInternals::internalState * p_internalState = static_cast<struct WrapperInternals::internalState *>(p_state);

		 internalStateIncrement(p_internalState);
		 internalStatePrint("func_exec_compute_deviation", p_internalState);

		 if ( (inPoint->size_  != p_internalState->info.inSize_) ||
		      (outPoint->size_ != p_internalState->info.outSize_) ) return WRAPPER_EXECUTION_ERROR;

		 double & E = inPoint->data_[0];
		 double & F = inPoint->data_[1];
		 double & L = inPoint->data_[2];
		 double & I = inPoint->data_[3];
    
		 double & d = outPoint->data_[0];
      
		 int rc = 0;
		 switch (p_internalState->p_exchangedData->parameters_->mode_) {

		 case WRAPPER_STATICLINK:
		   /* The real computation is here */
		   if ((E == 0.0) || (I == 0.0)) return WRAPPER_EXECUTION_ERROR;
		   d = -( F * L*L*L ) / ( 3 * E * I );
		   dbg_printMessage("func_exec_compute_deviation", "Computation is done (static-link)");
		   break;

		 case WRAPPER_DYNAMICLINK:
		   dbg_printMessage("func_exec_compute_deviation", "Computation is impossible (dynamic-link)");
		   return WRAPPER_USAGE_ERROR;
		   break;

		 case WRAPPER_FORK:
		   {
		     /* We build a temporary directory in which we will work */
		     char * temporaryDirectory = createTemporaryDirectory("openturnsWorkingDirectory", p_internalState->p_exchangedData);

		     /* We create the input files for the external code in the temporary directory */
		     if (createInputFiles(temporaryDirectory, p_internalState->p_exchangedData, inPoint)) return WRAPPER_EXECUTION_ERROR;

		     /* Change to temporary directory */
		     if (changeDirectory(temporaryDirectory)) return WRAPPER_EXECUTION_ERROR;

		     /* Compute */
		     switch (p_internalState->p_exchangedData->parameters_->in_) {

		     case WRAPPER_FILES:
		       rc = system(p_internalState->p_exchangedData->parameters_->command_);
		       dbg_printMessage("func_exec_compute_deviation", "Computation is done (fork with files)");
		       break;
	  
		     case WRAPPER_ARGUMENTS:
		       {
			 char * cmd = makeCommandFromTemplate(p_internalState->p_exchangedData->parameters_->command_, p_internalState->p_exchangedData, inPoint);
			 rc = system(cmd);
			 free(cmd);
		       }
		       dbg_printMessage("func_exec_compute_deviation", "Computation is done (fork with arguments)");
		       break;

		     default:
		       return WRAPPER_USAGE_ERROR;
		     }

		     /* Return to previous directory */
		     if (changeDirectory(p_internalState->currentWorkingDirectory)) return WRAPPER_EXECUTION_ERROR;

		     /* Read the output values */
		     if (readOutputFiles(temporaryDirectory, p_internalState->p_exchangedData, outPoint)) return WRAPPER_EXECUTION_ERROR;

		     /* We kill the temporary directory */
		     deleteTemporaryDirectory(temporaryDirectory, rc);
		   }
		   break;

		 default:
		   return WRAPPER_USAGE_ERROR;
		 }

	       } )


    FUNC_EXEC_SAMPLE( WRAPPERNAME,
		      {
			struct WrapperInternals::internalState * p_internalState = static_cast<struct WrapperInternals::internalState *>(p_state);

			unsigned long inDimension((inSample->data_[0]).size_);
			if (p_internalState->info.inSize_ != inDimension) return WRAPPER_WRONG_ARGUMENT;
			unsigned long outDimension((outSample->data_[0]).size_);
			if (p_internalState->info.outSize_ != outDimension) return WRAPPER_WRONG_ARGUMENT;
			unsigned long size(inSample->size_);

			WrapperErrorCode code(WRAPPER_OK);
			for (unsigned long i = 0; i < size; i++) {
			  code = func_exec_compute_deviation(p_state, &inSample->data_[i],  &outSample->data_[i]);
			  if (code != WRAPPER_OK) return code;
			}
		      } )



  /**
   * Finalization function
   * This function is called once just before the wrapper is unloaded. It is the place to flush
   * any output file or free any allocated memory. When this function returns, the wrapper is supposed
   * to have all its work done, so it is not possible to get anymore information from it after that.
   */
    FUNC_FINALIZE( WRAPPERNAME , {} )





  /*
******************************************************************************************
*                                                                                        *
*                             compute_deviation gradient                                 *
*                                                                                        *
******************************************************************************************
*/


  /* The createState function is optional */
    GRAD_CREATESTATE( WRAPPERNAME ,
		      {
			WrapperInternals::commonInternalStateCreation(p_p_state, p_exchangedData);
		      } )
    
  /* The deleteState function is optional */
    GRAD_DELETESTATE( WRAPPERNAME ,
		      {
			WrapperInternals::commonInternalStateDestruction(p_state);
		      } )





  /* The wrapper information informs the NumericalMathFunction object that loads the wrapper of the
   * signatures of the wrapper functions. In particular, it hold the size of the input NumericalPoint
   * (inSize_) and of the output NumericalPoint (outSize_).
   * Those information are also used by the gradient and hessian functions to set the correct size
   * of the returned matrix and tensor.
   */
    GRAD_INFO( WRAPPERNAME ,
	       {
		 WrapperInternals::commonGetInfo(p_state, p_info);
	       } )


  /* Any function declared into the wrapper MUST declare three actual function prefixed with
   * 'init_', 'exec_' and 'finalize_' folowed by the name of the function, here 'compute_deviation'.
   *
   * The 'init_' function is only called once when the NumericalMathFunction object is created.
   * It allows the wrapper to set some internal state, read some external file, prepare the function
   * to run, etc. It takes only one argument, the internal state as created by the 
   *
   * The 'exec_' function is intended to execute what the wrapper is done for: compute an mathematical
   * function or anything else. It takes the internal state pointer as its first argument, the input
   * NumericalPoint pointer as the second and the output NumericalPoint pointer as the third.
   *
   * The 'finalize_' function is only called once when the NumericalMathFunction object is destroyed.
   * It allows the wrapper to flush anything before unloading.
   */


  /**
   * Initialization function
   * This function is called once just after the wrapper is loaded to initialize
   * it, ie create a temparary subdirectory (remember that the wrapper may be called
   * concurrently), store exchanged data in some internal repository, do some
   * pre-computational operation, etc.
   */
    GRAD_INIT( WRAPPERNAME , {} )




  /**
   * Execution function
   * This function is called by the platform to do the real work of the wrapper. It may be
   * called concurrently, so be aware of not using shared or global data not protected by
   * a critical section.
   * This function has a mathematical meaning. It operates on one vector (aka point) and
   * returns a gradient matrix.
   */
    GRAD_EXEC( WRAPPERNAME, 
	       {
		 struct WrapperInternals::internalState * p_internalState = static_cast<struct WrapperInternals::internalState *>(p_state);

		 internalStateIncrement(p_internalState);

		 if ( (inPoint->size_     != p_internalState->info.inSize_) ||
		      (outMatrix->nbrows_ != p_internalState->info.inSize_) ||
		      (outMatrix->nbcols_ != p_internalState->info.outSize_) )  return WRAPPER_EXECUTION_ERROR;

		 /* The real computation is here */
		 double & E     = inPoint->data_[0];
		 double & F     = inPoint->data_[1];
		 double & L     = inPoint->data_[2];
		 double & I     = inPoint->data_[3];

		 double & df_dE = outMatrix->data_[0];
		 double & df_dF = outMatrix->data_[1];
		 double & df_dL = outMatrix->data_[2];
		 double & df_dI = outMatrix->data_[3];

		 if ((E == 0.0) || (I == 0.0)) return WRAPPER_EXECUTION_ERROR;

		 df_dE =  ( F * L*L*L ) / ( 3 * E*E * I   );
		 df_dF = -(     L*L*L ) / ( 3 * E   * I   );
		 df_dL = -( F * L*L   ) / (     E   * I   );
		 df_dI =  ( F * L*L*L ) / ( 3 * E   * I*I );
	       } )




  /**
   * Finalization function
   * This function is called once just before the wrapper is unloaded. It is the place to flush
   * any output file or free any allocated memory. When this function returns, the wrapper is supposed
   * to have all its work done, so it is not possible to get anymore information from it after that.
   */
    GRAD_FINALIZE( WRAPPERNAME , {} )






  /*
******************************************************************************************
*                                                                                        *
*                             compute_deviation hessian                                  *
*                                                                                        *
******************************************************************************************
*/


  /* The createState function is optional */
    HESS_CREATESTATE( WRAPPERNAME ,
		      {
			WrapperInternals::commonInternalStateCreation(p_p_state, p_exchangedData);
		      } )
    
  /* The deleteState function is optional */
    HESS_DELETESTATE( WRAPPERNAME ,
		      {
			WrapperInternals::commonInternalStateDestruction(p_state);
		      } )





  /* The wrapper information informs the NumericalMathFunction object that loads the wrapper of the
   * signatures of the wrapper functions. In particular, it hold the size of the input NumericalPoint
   * (inSize_) and of the output NumericalPoint (outSize_).
   * Those information are also used by the hessient and hessian functions to set the correct size
   * of the returned matrix and tensor.
   */
    HESS_INFO( WRAPPERNAME ,
	       {
		 WrapperInternals::commonGetInfo(p_state, p_info);
	       } )


  /* Any function declared into the wrapper MUST declare three actual function prefixed with
   * 'init_', 'exec_' and 'finalize_' folowed by the name of the function, here 'compute_deviation'.
   *
   * The 'init_' function is only called once when the NumericalMathFunction object is created.
   * It allows the wrapper to set some internal state, read some external file, prepare the function
   * to run, etc. It takes only one argument, the internal state as created by the 
   *
   * The 'exec_' function is intended to execute what the wrapper is done for: compute an mathematical
   * function or anything else. It takes the internal state pointer as its first argument, the input
   * NumericalPoint pointer as the second and the output NumericalPoint pointer as the third.
   *
   * The 'finalize_' function is only called once when the NumericalMathFunction object is destroyed.
   * It allows the wrapper to flush anything before unloading.
   */


  /**
   * Initialization function
   * This function is called once just after the wrapper is loaded to initialize
   * it, ie create a temparary subdirectory (remember that the wrapper may be called
   * concurrently), store exchanged data in some internal repository, do some
   * pre-computational operation, etc.
   */
    HESS_INIT( WRAPPERNAME , {} )




  /**
   * Execution function
   * This function is called by the platform to do the real work of the wrapper. It may be
   * called concurrently, so be aware of not using shared or global data not protected by
   * a critical section.
   * This function has a mathematical meaning. It operates on one vector (aka point) and
   * returns a hessient matrix.
   */
    HESS_EXEC( WRAPPERNAME, 
	       {
		 struct WrapperInternals::internalState * p_internalState = static_cast<struct WrapperInternals::internalState *>(p_state);

		 internalStateIncrement(p_internalState);


		 if ( (inPoint->size_       != p_internalState->info.inSize_) ||
		      (outTensor->nbrows_   != p_internalState->info.inSize_) ||
		      (outTensor->nbcols_   != p_internalState->info.inSize_) ||
		      (outTensor->nbsheets_ != p_internalState->info.outSize_) ) return WRAPPER_EXECUTION_ERROR;


		 /* The real computation is here */
		 double & E = inPoint->data_[0];
		 double & F = inPoint->data_[1];
		 double & L = inPoint->data_[2];
		 double & I = inPoint->data_[3];

		 if ((E == 0.0) || (I == 0.0)) return WRAPPER_EXECUTION_ERROR;

		 double d2f_dE2  = -( 2 * F * L*L*L ) / ( 3 * E*E*E * I     );
		 double d2f_dEdF =  (         L*L*L ) / ( 3 * E*E   * I     );
		 double d2f_dEdL =  (     F * L*L   ) / (     E*E   * I     );
		 double d2f_dEdI = -(     F * L*L*L ) / ( 3 * E*E   * I*I   );

		 double d2f_dF2  =  0;
		 double d2f_dFdL = -(         L*L   ) / (     E     * I     );
		 double d2f_dFdI =  (         L*L*L ) / ( 3 * E     * I*I   );

		 double d2f_dL2  = -( 2 * F * L     ) / (     E     * I     );
		 double d2f_dLdI =  (     F * L*L   ) / (     E     * I*I   );

		 double d2f_dI2  = -( 2 * F * L*L*L ) / ( 3 * E     * I*I*I );

		 outTensor->data_[ 0] = d2f_dE2;
		 outTensor->data_[ 1] = d2f_dEdF;
		 outTensor->data_[ 2] = d2f_dEdL;
		 outTensor->data_[ 3] = d2f_dEdI;
    
		 outTensor->data_[ 4] = d2f_dEdF;
		 outTensor->data_[ 5] = d2f_dF2;
		 outTensor->data_[ 6] = d2f_dFdL;
		 outTensor->data_[ 7] = d2f_dFdI;
    
		 outTensor->data_[ 8] = d2f_dEdL;
		 outTensor->data_[ 9] = d2f_dFdL;
		 outTensor->data_[10] = d2f_dL2;
		 outTensor->data_[11] = d2f_dLdI;
    
		 outTensor->data_[12] = d2f_dEdI;
		 outTensor->data_[13] = d2f_dFdI;
		 outTensor->data_[14] = d2f_dLdI;
		 outTensor->data_[15] = d2f_dI2;
	       } )




  /**
   * Finalization function
   * This function is called once just before the wrapper is unloaded. It is the place to flush
   * any output file or free any allocated memory. When this function returns, the wrapper is supposed
   * to have all its work done, so it is not possible to get anymore information from it after that.
   */
    HESS_FINALIZE( WRAPPERNAME , {} )





    } /* end extern "C" */
