//                                               -*- C++ -*-
/**
 *  @file  t_Histogram_std.cxx
 *  @brief The test file of class Histogram for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: t_Histogram_std.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include <iostream>
#include <sstream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "Histogram.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "RandomGenerator.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Distribution;


class TestObject : public Histogram
{
public:
  explicit TestObject() : Histogram(1., Histogram::PairCollection(0)) {}
  explicit TestObject(const String & name) : Histogram(1., Histogram::PairCollection(0)) {setName(name);}
  virtual ~TestObject() {}
};


int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  setRandomGenerator();

  try {
    // Test basic functionnalities
    checkClassWithClassName<TestObject>();

    // Test some extra functionnalities
    checkNameFeature<TestObject>();

    // Instanciate one distribution object
    std::cout << "begin histo comp test" << std::endl;
    UnsignedLong collectionSize(4);
    Histogram::PairCollection collection(collectionSize);
    collection[0] = Histogram::Pair(0.5, 1.0);
    collection[1] = Histogram::Pair(1.5, 0.7);
    collection[2] = Histogram::Pair(3.5, 1.2);
    collection[3] = Histogram::Pair(2.5, 0.9);
    std::cout << "collection=" << collection << std::endl;
    Histogram distribution(-1.5, collection);
    std::cout << "Distribution " << distribution << std::endl;

    // Is this distribution elliptical ?
    std::cout << "Elliptical = " << (distribution.isElliptical() ? "true" : "false") << std::endl;

    // Is this distribution continuous ?
    std::cout << "Continuous = " << (distribution.isContinuous() ? "true" : "false") << std::endl;

    // Test for realization of distribution
    NumericalPoint oneRealization = distribution.getRealization();
    std::cout << "oneRealization=" << oneRealization << std::endl;

    // Test for sampling
    UnsignedLong size = 10000;
    NumericalSample oneSample = distribution.getNumericalSample( size );
    std::cout << "oneSample first=" << oneSample[0] << " last=" << oneSample[size - 1] << std::endl;
    std::cout << "mean=" << oneSample.computeMean() << std::endl;
    std::cout << "covariance=" << oneSample.computeCovariance() << std::endl;

    // Define a point
    NumericalPoint point( distribution.getDimension(), 1.0 );
    std::cout << "Point= " << point << std::endl;

    // Show PDF and CDF of point
    NumericalScalar eps(1e-5);
    NumericalPoint DDF = distribution.computeDDF( point );
    std::cout << "ddf     =" << DDF << std::endl;
    std::cout << "ddf (FD)=" << NumericalPoint(1, (distribution.computePDF( point + NumericalPoint(1, eps) ) - distribution.computePDF( point  + NumericalPoint(1, -eps) )) / (2.0 * eps)) << std::endl;
    NumericalScalar PDF = distribution.computePDF( point );
    std::cout << "pdf     =" << PDF << std::endl;
    std::cout << "pdf (FD)=" << (distribution.computeCDF( point + NumericalPoint(1, eps) ) - distribution.computeCDF( point  + NumericalPoint(1, -eps) )) / (2.0 * eps) << std::endl;
    NumericalScalar CDF = distribution.computeCDF( point );
    std::cout << "cdf=" << CDF << std::endl;
    //    NumericalPoint PDFgr = distribution.computePDFGradient( point );
    //    std::cout << "pdf gradient     =" << PDFgr << std::endl;
    //    NumericalPoint PDFgrFD(1 + 2 * collectionSize);
    //    PDFgrFD[0] = (Histogram(distribution.getFirst() + eps, distribution.getPairCollection()).computePDF(point) -
    //                  Histogram(distribution.getFirst() - eps, distribution.getPairCollection()).computePDF(point)) / (2.0 * eps);
    //    for (UnsignedLong i = 0; i < collectionSize; i++)
    //      {
    //	Histogram::PairCollection collectionLeft = distribution.getPairCollection();
    //	Histogram::PairCollection collectionRight = distribution.getPairCollection();
    //	collectionLeft[i].h_  += eps;
    //	collectionRight[i].h_ -= eps;
    //	PDFgrFD[2 * i + 1] = (Histogram(distribution.getFirst(), collectionLeft).computePDF(point) -
    //			      Histogram(distribution.getFirst(), collectionRight).computePDF(point)) / (2.0 * eps);
    //	collectionLeft = distribution.getPairCollection();
    //	collectionRight = collectionLeft;
    //	collectionLeft[i].l_  += eps;
    //	collectionRight[i].l_ -= eps;
    //	PDFgrFD[2 * i + 2] = (Histogram(distribution.getFirst(), collectionLeft).computePDF(point) -
    //			      Histogram(distribution.getFirst(), collectionRight).computePDF(point)) / (2.0 * eps);
    //      }
    //    std::cout << "pdf gradient (FD)=" << PDFgrFD << std::endl;
    //    NumericalPoint CDFgr = distribution.computeCDFGradient( point );
    //    std::cout << "cdf gradient     =" << CDFgr << std::endl;
    //    NumericalPoint CDFgrFD(1 + 2 * collectionSize);
    //    CDFgrFD[0] = (Histogram(distribution.getFirst() + eps, distribution.getPairCollection()).computeCDF(point) -
    //                  Histogram(distribution.getFirst() - eps, distribution.getPairCollection()).computeCDF(point)) / (2.0 * eps);
    //    for (UnsignedLong i = 0; i < collectionSize; i++)
    //      {
    //	Histogram::PairCollection collectionLeft = distribution.getPairCollection();
    //	Histogram::PairCollection collectionRight = distribution.getPairCollection();
    //	collectionLeft[i].h_  += eps;
    //	collectionRight[i].h_ -= eps;
    //	CDFgrFD[2 * i + 1] = (Histogram(distribution.getFirst(), collectionLeft).computeCDF(point) -
    //			      Histogram(distribution.getFirst(), collectionRight).computeCDF(point)) / (2.0 * eps);
    //	collectionLeft = distribution.getPairCollection();
    //	collectionRight = collectionLeft;
    //	collectionLeft[i].l_  += eps;
    //	collectionRight[i].l_ -= eps;
    //	CDFgrFD[2 * i + 2] = (Histogram(distribution.getFirst(), collectionLeft).computeCDF(point) -
    //			      Histogram(distribution.getFirst(), collectionRight).computeCDF(point)) / (2.0 * eps);
    //      }
    //    std::cout << "cdf gradient (FD)=" << CDFgrFD << std::endl;
    NumericalPoint quantile = distribution.computeQuantile( 0.95 );
    std::cout << "quantile=" << quantile << std::endl;
    std::cout << "cdf(quantile)=" << distribution.computeCDF(quantile) << std::endl;
    NumericalPoint mean = distribution.getMean();
    std::cout << "mean=" << mean << std::endl;
    CovarianceMatrix covariance = distribution.getCovariance();
    std::cout << "covariance=" << covariance << std::endl;
    Histogram::NumericalPointWithDescriptionCollection parameters = distribution.getParametersCollection();
    std::cout << "parameters=" << parameters << std::endl;
    UnsignedLong testSize(0);
    for (UnsignedLong i = 0; i < testSize; i++)
      {
	NumericalScalar q(RandomGenerator::Generate());
	if (fabs(q - distribution.computeCDF(distribution.computeQuantile(q))) > eps)
	  {
	    std::cout << "q=" << q << " quantile=" << distribution.computeQuantile(q)[0] << " CDF(quantile)=" << distribution.computeCDF(distribution.computeQuantile(q)) << std::endl;
	  }
      }

    // Specific to this distribution
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
