//                                               -*- C++ -*-
/**
 *  @file  t_IsoProbabilisticTransformation_EllipticalDistribution.cxx
 *  @brief The test file of class IsoProbabilisticTransformation for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-10 16:43:31 +0200 (jeu 10 mai 2007) $
 *  Id:      $Id: t_IsoProbabilisticTransformation_EllipticalDistribution.cxx 434 2007-05-10 14:43:31Z dutka $
 */
#include <iostream>
#include <sstream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "Normal.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "CorrelationMatrix.hxx"
#include "Distribution.hxx"
#include "CenteredFiniteDifferenceGradient.hxx"
#include "CenteredFiniteDifferenceHessian.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Base::Diff;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;

typedef Distribution::IsoProbabilisticTransformation IsoProbabilisticTransformation;
typedef Distribution::InverseIsoProbabilisticTransformation InverseIsoProbabilisticTransformation;

Matrix clean(Matrix in)
{
  UnsignedLong rowDim(in.getNbRows());
  UnsignedLong colDim(in.getNbColumns());
  for(UnsignedLong i = 0; i < rowDim; i++)
    for(UnsignedLong j = 0; j < colDim; j++)
      if (fabs(in(i, j)) < 1.e-6) in(i, j) = 0.0;
  return in;
}

SymmetricTensor clean(SymmetricTensor in)
{
  UnsignedLong rowDim(in.getNbRows());
  UnsignedLong colDim(in.getNbColumns());
  UnsignedLong sheetDim(in.getNbSheets());
  for(UnsignedLong i = 0; i < rowDim; i++)
    for(UnsignedLong j = 0; j < colDim; j++)
      for (UnsignedLong k = 0; k < sheetDim; k++)
        if (fabs(in(i, j, k)) < 1.e-6) in(i, j, k) = 0.0;
  return in;
}

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  setRandomGenerator();

  try {
    // Instanciate one distribution object
    UnsignedLong dim(3);
    NumericalPoint meanPoint(dim, 1.0);
    meanPoint[0] = 0.5;
    meanPoint[1] = -0.5;
    NumericalPoint sigma(dim, 1.0);
    sigma[0] = 2.0;
    sigma[1] = 3.0;
    CorrelationMatrix R(dim);
    for (UnsignedLong i = 1; i < dim; i++)
      {
	R(i, i - 1) = 0.5;
      }

    Normal distribution(meanPoint, sigma, R);

    // Test for sampling
    UnsignedLong size = 10000;
    NumericalSample sample = distribution.getNumericalSample( size );
    std::cout << "sample first=" << sample[0] << " last=" << sample[size - 1] << std::endl;
    std::cout << "sample mean=" << sample.computeMean() << std::endl;
    std::cout << "sample covariance=" << sample.computeCovariance() << std::endl;

    IsoProbabilisticTransformation transform(distribution.getIsoProbabilisticTransformation());
    std::cout << "isoprobabilistic transformation=" << transform << std::endl;
    NumericalSample transformedSample(transform(sample));
    std::cout << "transformed sample first=" << transformedSample[0] << " last=" << transformedSample[size - 1] << std::endl;
    std::cout << "transformed sample mean=" << transformedSample.computeMean() << std::endl;
    std::cout << "transformed sample covariance=" << transformedSample.computeCovariance() << std::endl;

    // Test for evaluation
    InverseIsoProbabilisticTransformation inverseTransform(distribution.getInverseIsoProbabilisticTransformation());
    std::cout << "inverse isoprobabilistic transformation=" << inverseTransform << std::endl;
    NumericalSample transformedBackSample(inverseTransform(transformedSample));
    std::cout << "transformed back sample first=" << transformedBackSample[0] << " last=" << transformedBackSample[size - 1] << std::endl;
    std::cout << "transformed back sample mean=" << transformedBackSample.computeMean() << std::endl;
    std::cout << "transformed back sample covariance=" << transformedBackSample.computeCovariance() << std::endl;
    NumericalPoint point(dim, 1.0);
    std::cout << "point=" << point << std::endl;
    NumericalPoint transformedPoint(transform(point));
    std::cout << "transform value at point        =" << transformedPoint << std::endl;
    std::cout << "transform gradient at point     =" << clean(transform.gradient(point)) << std::endl;
    std::cout << "transform gradient at point (FD)=" << clean(CenteredFiniteDifferenceGradient(1.0e-5, transform.getEvaluationImplementation()).gradient(point)) << std::endl;
    std::cout << "transform hessian at point      =" << clean(transform.hessian(point)) << std::endl;
    std::cout << "transform hessian at point (FD) =" << clean(CenteredFiniteDifferenceHessian(1.0e-4, transform.getEvaluationImplementation()).hessian(point)) << std::endl;
    std::cout << "inverse transform value at transformed point        =" << inverseTransform(transformedPoint) << std::endl;
    std::cout << "inverse transform gradient at transformed point (FD)=" << clean(inverseTransform.gradient(transformedPoint)) << std::endl;
    std::cout << "inverse transform gradient at transformed point     =" << clean(CenteredFiniteDifferenceGradient(1.0e-5, inverseTransform.getEvaluationImplementation()).gradient(transformedPoint)) << std::endl;
    std::cout << "inverse transform hessian at transformed point      =" << clean(inverseTransform.hessian(transformedPoint)) << std::endl;
    std::cout << "inverse transform hessian at transformed point (FD) =" << clean(CenteredFiniteDifferenceHessian(1.0e-4, inverseTransform.getEvaluationImplementation()).hessian(transformedPoint)) << std::endl;

    // Test for parameters
    std::cout << "parameters gradient at point=" << transform.parametersGradient(point) << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
