//                                               -*- C++ -*-
/**
 *  @file  t_MergeRandomAndConstantInput.cxx
 *  @brief The test file of class NumericalMathFunction for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-10 16:43:31 +0200 (jeu 10 mai 2007) $
 *  Id:      $Id: t_MergeRandomAndConstantInput.cxx 434 2007-05-10 14:43:31Z dutka $
 */
#include <iostream>
#include <iomanip>
#include <sstream>
#include <exception>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "NumericalMathFunction.hxx"
#include "IdentityMatrix.hxx"
#include "NumericalPoint.hxx"
#include "Matrix.hxx"
#include "Normal.hxx"
#include "LinearNumericalMathEvaluationImplementation.hxx"
#include "ConstantNumericalMathGradientImplementation.hxx"
#include "ConstantNumericalMathHessianImplementation.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Func;
using namespace OT::Uncertainty::Distribution;


int main(int argc, char *argv[])
{
  TESTPREAMBLE;

  try {

    /** External code. This code has an input vector of dimension 4, namely (p0, p1, p2, p3)'. */
    NumericalMathFunction externalCode("poutre");
    UnsignedLong dim(externalCode.getInputNumericalPointDimension());

    /** The external code will be connected to 2 independent random variables X0 and X1 and one deterministic variable X2 with the following scheme:
	X2->p0
	X0->p1
	X1->p2
	X0->p3
	It means that (p0, p1, p2, p3)' = A.(X0, X1)' + b with:
	A = [0 0] b = [X2]
	[1 0]     [ 0]
	[0 1]     [ 0]
	[1 0]     [ 0]
	Here we build the linear function x -> A.x + b
    */
    UnsignedLong stochasticDimension(2);
    // UnsignedLong deterministicDimension(1);
    Matrix A(dim, stochasticDimension);
    A(1, 0) = 1;
    A(2, 1) = 1;
    A(3, 0) = 1;
    NumericalPoint b(dim, 0);
    NumericalScalar X2(50.0);
    b[0] = X2;

    NumericalMathFunction connect;
    NumericalPoint zero(stochasticDimension, 0);
    /** A LinearNumericalMathFunction will arrive soon... */
    connect.setEvaluationImplementation(new LinearNumericalMathEvaluationImplementation(zero, b, A.transpose()));
    connect.setGradientImplementation(new ConstantNumericalMathGradientImplementation(A.transpose()));
    connect.setHessianImplementation(new ConstantNumericalMathHessianImplementation(SymmetricTensor(stochasticDimension, dim)));
    /** We are now ready to build the resulting code externalCode(connect()) */
    NumericalMathFunction finalCode(externalCode, connect);

    /** Check if it worked */
    NumericalPoint x(connect.getInputNumericalPointDimension());
    x[0] = 5;
    x[1] = 10;
    std::cout << "finalCode(x)=" << finalCode(x) << std::endl;
    NumericalPoint xRef(dim);
    xRef[0] = X2;
    xRef[1] = x[0];
    xRef[2] = x[1];
    xRef[3] = x[0];
    std::cout << "ref=" << externalCode(xRef) << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
