//                                               -*- C++ -*-
/**
 *  @file  t_Weibull_std.cxx
 *  @brief The test file of class Weibull for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: t_Weibull_std.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include <iostream>
#include <sstream>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "Weibull.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Distribution;


class TestObject : public Weibull
{
public:
  explicit TestObject() : Weibull(2.0, 1.5, -0.5) {}
  explicit TestObject(const String & name) : Weibull(2.0, 1.5, -0.5) {setName(name);}
  virtual ~TestObject() {}
};


int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  setRandomGenerator();

  try {
    // Test basic functionnalities
    checkClassWithClassName<TestObject>();

    // Test some extra functionnalities
    checkNameFeature<TestObject>();

    // Instanciate one distribution object
    Weibull distribution(2.0, 1.5, -0.5);
    std::cout << "Distribution " << distribution << std::endl;

    // Is this distribution elliptical ?
    std::cout << "Elliptical = " << (distribution.isElliptical() ? "true" : "false") << std::endl;

    // Is this distribution continuous ?
    std::cout << "Continuous = " << (distribution.isContinuous() ? "true" : "false") << std::endl;

    // Test for realization of distribution
    NumericalPoint oneRealization = distribution.getRealization();
    std::cout << "oneRealization=" << oneRealization << std::endl;

    // Test for sampling
    UnsignedLong size = 10000;
    NumericalSample oneSample = distribution.getNumericalSample( size );
    std::cout << "oneSample first=" << oneSample[0] << " last=" << oneSample[size - 1] << std::endl;
    std::cout << "mean=" << oneSample.computeMean() << std::endl;
    std::cout << "covariance=" << oneSample.computeCovariance() << std::endl;

    // Define a point
    NumericalPoint point( distribution.getDimension(), 1.0 );
    std::cout << "Point= " << point << std::endl;

    // Show PDF and CDF of point
    NumericalScalar eps(1e-5);
    NumericalPoint DDF = distribution.computeDDF( point );
    std::cout << "ddf     =" << DDF << std::endl;
    std::cout << "ddf (FD)=" << NumericalPoint(1, (distribution.computePDF( point + NumericalPoint(1, eps) ) - distribution.computePDF( point  + NumericalPoint(1, -eps) )) / (2.0 * eps)) << std::endl;
    NumericalScalar PDF = distribution.computePDF( point );
    std::cout << "pdf     =" << PDF << std::endl;
    std::cout << "pdf (FD)=" << (distribution.computeCDF( point + NumericalPoint(1, eps) ) - distribution.computeCDF( point  + NumericalPoint(1, -eps) )) / (2.0 * eps) << std::endl;
    NumericalScalar CDF = distribution.computeCDF( point );
    std::cout << "cdf=" << CDF << std::endl;
    NumericalPoint PDFgr = distribution.computePDFGradient( point );
    std::cout << "pdf gradient     =" << PDFgr << std::endl;
    NumericalPoint PDFgrFD(3);
    PDFgrFD[0] = (Weibull(distribution.getAlpha() + eps, distribution.getBeta(), distribution.getGamma()).computePDF(point) -
                  Weibull(distribution.getAlpha() - eps, distribution.getBeta(), distribution.getGamma()).computePDF(point)) / (2.0 * eps);
    PDFgrFD[1] = (Weibull(distribution.getAlpha(), distribution.getBeta() + eps, distribution.getGamma()).computePDF(point) -
                  Weibull(distribution.getAlpha(), distribution.getBeta() - eps, distribution.getGamma()).computePDF(point)) / (2.0 * eps);
    PDFgrFD[2] = (Weibull(distribution.getAlpha(), distribution.getBeta(), distribution.getGamma() + eps).computePDF(point) -
                  Weibull(distribution.getAlpha(), distribution.getBeta(), distribution.getGamma() - eps).computePDF(point)) / (2.0 * eps);
    std::cout << "pdf gradient (FD)=" << PDFgrFD << std::endl;
    NumericalPoint CDFgr = distribution.computeCDFGradient( point );
    std::cout << "cdf gradient     =" << CDFgr << std::endl;
    NumericalPoint CDFgrFD(3);
    CDFgrFD[0] = (Weibull(distribution.getAlpha() + eps, distribution.getBeta(), distribution.getGamma()).computeCDF(point) -
                  Weibull(distribution.getAlpha() - eps, distribution.getBeta(), distribution.getGamma()).computeCDF(point)) / (2.0 * eps);
    CDFgrFD[1] = (Weibull(distribution.getAlpha(), distribution.getBeta() + eps, distribution.getGamma()).computeCDF(point) -
                  Weibull(distribution.getAlpha(), distribution.getBeta() - eps, distribution.getGamma()).computeCDF(point)) / (2.0 * eps);
    CDFgrFD[2] = (Weibull(distribution.getAlpha(), distribution.getBeta(), distribution.getGamma() + eps).computeCDF(point) -
                  Weibull(distribution.getAlpha(), distribution.getBeta(), distribution.getGamma() - eps).computeCDF(point)) / (2.0 * eps);
    std::cout << "cdf gradient (FD)=" << CDFgrFD << std::endl;
    NumericalPoint quantile = distribution.computeQuantile( 0.95 );
    std::cout << "quantile=" << quantile << std::endl;
    std::cout << "cdf(quantile)=" << distribution.computeCDF(quantile) << std::endl;
    NumericalPoint mean = distribution.getMean();
    std::cout << "mean=" << mean << std::endl;
    NumericalPoint standardDeviation = distribution.getStandardDeviation();
    std::cout << "standard deviation=" << standardDeviation << std::endl;
    NumericalPoint skewness = distribution.getSkewness();
    std::cout << "skewness=" << skewness << std::endl;
    NumericalPoint kurtosis = distribution.getKurtosis();
    std::cout << "kurtosis=" << kurtosis << std::endl;
    CovarianceMatrix covariance = distribution.getCovariance();
    std::cout << "covariance=" << covariance << std::endl;
    Weibull::NumericalPointWithDescriptionCollection parameters = distribution.getParametersCollection();
    std::cout << "parameters=" << parameters << std::endl;

    // Specific to this distribution
    NumericalScalar mu(distribution.getMu());
    std::cout << "mu=" << mu << std::endl;
    NumericalScalar sigma(distribution.getSigma());
    std::cout << "sigma=" << sigma << std::endl;
    Weibull newDistribution(mu, sigma, distribution.getGamma(),Weibull::MUSIGMA);
    std::cout << "alpha from (mu, sigma)=" << newDistribution.getAlpha() << std::endl;
    std::cout << "beta from (mu, sigma)=" << newDistribution.getBeta() << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
