//                                               -*- C++ -*-
/**
 *  @file  ExceptionDefs.hxx
 *  @brief Exception defines top-most exception strategies
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: ExceptionDefs.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_EXCEPTIONDEFS_HXX
#define OPENTURNS_EXCEPTIONDEFS_HXX

#include <exception>
#include "OTprivate.hxx"
#include "OSS.hxx"

namespace OpenTURNS
{


#ifndef SWIG
      /**
       * A help class that saves the position where it was instanciated in file.
       * This class works in conjonction with the preprocessor (cpp). See HERE macro
       */
      class PointInSourceFile
      {
      public:
	PointInSourceFile(const std::string & file, int line
#ifdef __GNUC__
			  , const std::string & function
#endif
			  )
	  : file_(file), line_(line)
#ifdef __GNUC__
	  , function_(function)
#endif
	{}
	~PointInSourceFile() {}
    
	std::string getFile() const { return file_; }
	int getLine() const { return line_; }
#ifdef __GNUC__
	std::string getFunction() const { return function_; }
#endif

	std::string str() const
	{
	  std::ostringstream oss;
	  oss << file_ << ":" << line_
#ifdef __GNUC__
	      << " (in " << function_ << ")"
#endif
	    ;
	  return oss.str();
	}
    
      private:
	std::string file_;
	int line_;
#ifdef __GNUC__
	std::string function_;
#endif
      }; /* class PointInSourceFile */


      /**
       * The macro HERE creates a PointInSourceFile object that saves the position
       * in the source file.
       */
#ifdef __GNUC__
#define HERE OpenTURNS::PointInSourceFile(__FILE__,__LINE__,__FUNCTION__)
#else
#define HERE OpenTURNS::PointInSourceFile(__FILE__,__LINE__)
#endif

#endif /* SWIG */




  namespace Base
  {

    namespace Common
    {

      /**
       * @class Exception
       *
       * Exception defines top-most exception strategies
       */

      class Exception : 
	public std::exception
      {

      public:

	/** Default constructor */
	Exception(const OpenTURNS::PointInSourceFile & point);

	/** Copy constractor */
	Exception(const Exception & other);

	/** Destructor */
	virtual ~Exception() throw();

	/** String converter */
	String __repr__() const throw();

	/** Point accessor */
	const char * where() const throw();

	/** Reason accessor */
	const char * what() const throw();

	/** Class name accessor */
	const char * type() const throw();

	template <class T> Exception & operator << (T obj)
	{
	  reason_ << obj;
	  return *this;
	}
	

      protected:

	/** Inheritance constructor */
	Exception(const PointInSourceFile & point, const char * type);
	Exception(const PointInSourceFile & point, int id);

      private:
	
	/** The point in source file where the exception was raised */
	const PointInSourceFile point_;

	/** Reason describes what was the exception */
	OSS reason_;

	/** Exception class name */
	const String className_;

      }; /* class Exception */

      /**
       * @fn std::ostream & operator <<(std::ostream & os, const Exception & obj)
       * @brief Output stream converter
       * @param os A STL output stream exception
       * @param obj The exception read by \em os
       * @return A reference to \em os
       *
       * Operator << converts the Exception object to an output stream
       * so it is easy to show the reason of the exception.
       */
      std::ostream & operator <<(std::ostream & os, const Exception & obj);
#ifndef SWIG
      OStream & operator <<(OStream & OS, const Exception & obj);
#endif





      /**
       *
       * This section defines all exceptions that should be used.
       * Top-most Exception class is NOT intended to be used, though it is
       * not defined abstract.
       *
       */

      template <int I, class BASE = Exception>
      class UserException : public BASE
      {
      public:

	UserException(const OpenTURNS::PointInSourceFile & point)
	  : BASE(point,I) {}
	~UserException() throw () {}

	template <class T> UserException & operator << (T obj)
	{
	  BASE::operator<<( obj );
	  return *this;
	}

      protected:

	UserException(const PointInSourceFile & point, const char * type)
	  : BASE(point, type) {}
	UserException(const PointInSourceFile & point, int id)
	  : BASE(point, id) {}

      }; /* end class UserException */


      





    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_EXCEPTIONDEFS_HXX */
