//                                               -*- C++ -*-
/**
 *  @file  OSS.hxx
 *  @brief The class OSS streams out objects
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: OSS.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_OSS_HXX
#define OPENTURNS_OSS_HXX

#include <sstream>
#include <iterator>
#include "OStream.hxx"

namespace OpenTURNS
{

  /**
   * Struct Bulk protects the underlying object output from being
   * modified by OSSFormater
   */
  template <class T>
  struct Bulk
  {
    T data_;
    Bulk(T data) : data_(data) {}
    operator T() const { return data_; }
  };

  /**
   * Struct OSSFormater is a helper class for OSS that adapt the
   * format of output according to type T
   */

  /* Any type */
  template <class T>
  struct OSSFormater
  {
    inline
    static
    void apply(std::ostringstream & oss, T obj, int precision)
    {
      oss << obj;
    }

    inline
    static
    void apply(Base::Common::OStream & OS, T obj, int precision)
    {
      OS << obj;
    }
  }; /* struct OSSFormater */

  /* double */
  template <>
  struct OSSFormater<double>
  {
    inline
    static
    void apply(std::ostringstream & oss, double d, int precision)
    {
      int oldPrecision = oss.precision(precision);
      oss << d;
      oss.precision(oldPrecision);
    }

    inline
    static
    void apply(Base::Common::OStream & OS, double d, int precision)
    {
      int oldPrecision = OS.getStream().precision(precision);
      OS.getStream() << d;
      OS.getStream().precision(oldPrecision);
    }
  };

  /* float */
  template <>
  struct OSSFormater<float>
  {
    inline
    static
    void apply(std::ostringstream & oss, float f, int precision)
    {
      int oldPrecision = oss.precision(precision);
      oss << f;
      oss.precision(oldPrecision);
    }

    inline
    static
    void apply(Base::Common::OStream & OS, float f, int precision)
    {
      int oldPrecision = OS.getStream().precision(precision);
      OS.getStream() << f;
      OS.getStream().precision(oldPrecision);
    }
  };






  /**
   * Class OSS is useful when streaming data through a function
   * that expect a string as parameter
   */
  class OSS
  {
  private:
    std::ostringstream oss_;
    mutable int precision_;
    mutable bool full_;

  public:
    explicit OSS(bool full = true);

    template <class T>
    inline
    OSS & operator << (T obj)
    {
      if (full_) {
	Base::Common::OStream OS(oss_);
	OSSFormater<T>::apply(OS, obj, precision_);
      }
      else OSSFormater<T>::apply(oss_, obj, precision_);
      return *this;
    }

    inline
    OSS & setPrecision(int precision)
    {
      precision_ = precision;
      oss_.precision(precision_);
      return *this;
    }
    
    inline
    int getPrecision() const
    {
      precision_ = oss_.precision();
      return precision_;
    }

    operator std::string() const;

    void clear();

  }; /* class OSS */


  template <typename _Tp, typename _CharT = char,
	    typename _Traits = std::char_traits<_CharT> >
  class OSS_iterator
    : public std::iterator<std::output_iterator_tag, void, void, void, void>
  {
  public:
    //@{
    /// Public typedef
    typedef _CharT                         char_type;
    typedef _Traits                        traits_type;
    typedef OSS                            ostream_type;
    //@}
    
  private:
    ostream_type*     _M_stream;
    const _CharT*     _M_string;
    mutable bool      _M_first;
    
  public:
    /// Construct from an ostream.
    OSS_iterator(ostream_type& __s) : _M_stream(&__s), _M_string(0), _M_first(true) {}
    
    /**
     *  Construct from an ostream.
     *
     *  The delimiter string @a c is written to the stream after every Tp
     *  written to the stream.  The delimiter is not copied, and thus must
     *  not be destroyed while this iterator is in use.
     *
     *  @param  s  Underlying ostream to write to.
     *  @param  c  CharT delimiter string to insert.
     */
    OSS_iterator(ostream_type& __s, const _CharT* __c)
      : _M_stream(&__s), _M_string(__c), _M_first(true)  { }

    /// Copy constructor.
    OSS_iterator(const OSS_iterator& __obj)
      : _M_stream(__obj._M_stream), _M_string(__obj._M_string), _M_first(__obj._M_first)  { }

    /// Writes @a value to underlying ostream using operator<<.  If
    /// constructed with delimiter string, writes delimiter to ostream.
    OSS_iterator&
    operator=(const _Tp& __value)
    {
#ifdef __GNUC__
#if ( GCC_VERSION >= 30400 )
      __glibcxx_requires_cond(_M_stream != 0,
			      _M_message(__gnu_debug::__msg_output_ostream)
			      ._M_iterator(*this));
#endif
#endif
      if (_M_string && !_M_first) *_M_stream << _M_string;
      *_M_stream << __value;
      _M_first = false;
      return *this;
    }

    OSS_iterator&
    operator*()
    { return *this; }

    OSS_iterator&
    operator++()
    { return *this; }

    OSS_iterator&
    operator++(int)
    { return *this; }

  };


} /* namespace OpenTURNS */

#endif /* OPENTURNS_OSS_HXX */
