//                                               -*- C++ -*-
/**
 *  @file  Path.cxx
 *  @brief This class provides all the treatments for wrapper file manipulation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-06-15 16:14:58 +0200 (lun. 15 juin 2009) $
 *  Id:      $Id: Path.cxx 1272 2009-06-15 14:14:58Z dutka $
 */
#include <vector>                 // for std::vector
#include <string>                 // for std::string
#include <cstdlib>                // for getenv
#include <cstring>                // for strcpy
#ifdef WIN32
#include <fstream>                // for ofstream
#endif
#include <sys/types.h>            // for stat
#include <sys/stat.h>             // for stat
#include <unistd.h>               // for stat
#include <OTconfig.hxx>
#include "OSS.hxx"
#include "Path.hxx"
#include "ResourceMap.hxx"
#include "Log.hxx"

#ifndef SYSCONFIG_PATH
#error "SYSCONFIG_PATH is NOT defined. Check configuration."
#endif

#ifndef DATA_PATH
#error "DATA_PATH is NOT defined. Check configuration."
#endif

#ifndef OPENTURNS_HOME_ENV_VAR
#error "OPENTURNS_HOME_ENV_VAR is NOT defined. Check configuration."
#endif

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {


      /* The environment variable name */
      const String Path::OpenturnsWrapperPathVariableName_ = "OPENTURNS_WRAPPER_PATH";

      /* The environment variable name */
      const String Path::OpenturnsConfigPathVariableName_ = "OPENTURNS_CONFIG_PATH";

      /* The HOME subdirectory path */
      const String Path::HomeWrapperSubdirectory_ = "/openturns/wrappers";

      /* The HOME subdirectory path */
      const String Path::HomeConfigSubdirectory_ = "/openturns/etc";

      /* The 'wrapper' subdirectory path */
      const String Path::PrefixWrapperSubdirectory_ = "/openturns/wrappers";

      /* The 'openturns' configuration subdirectory path */
      const String Path::PrefixConfigSubdirectory_ = "/openturns";


      /*
       * Default constructor
       */
      Path::Path()
      {
	// Nothing to do
      }

      /* Destructor */
      Path::~Path()
      {
	// Nothing to do
      }

      /**
       * Analyse the process environment and return the directory where
       * installed wrappers and DTD are.
       *
       * The search rule is: check the following paths, in that order:
       *    + if the env var OPENTURNS_HOME exists, return ${OPENTURNS_HOME}/share/openturns/wrappers
       *      if it exists and is a directory,
       *    + otherwise return the installation path ${prefix}/share/openturns/wrappers, where 'prefix'
       *      is the installation path of the platform as provided at configuration time.
       */
      FileName Path::GetStandardWrapperDirectory()
      {
	FileName directory;
	bool dirExists = false;
	const char * otHome = getenv(OPENTURNS_HOME_ENV_VAR);
	if (otHome) {
	  directory = String(otHome) + "/share" + PrefixWrapperSubdirectory_;
	  struct stat status;
	  dirExists = (stat( directory.c_str(), &status ) == 0 && ( status.st_mode & S_IFDIR ));
	}
	if (!dirExists) {
	  directory = String(DATA_PATH) + PrefixWrapperSubdirectory_;
	}
	return directory;
      }


      /**
       * Analyse the process environment
       * and return a list of directories to search in for wrappers.
       *
       * The search rule is :look for the file in the following directories, in that order :
       *    + if the env var OPENTURNS_WRAPPER_PATH exists, in directories listed in
       *      ${OPENTURNS_WRAPPER_PATH} (see openturnsWrapperPathVariableName_)
       *    + in directory ${HOME}/openturns/wrappers (see homeSubdirectory_)
       *    + in the standard wrapper directory, as defined by the method
       *      GetStandardWrapperDirectory().
       */
      Path::DirectoryList Path::GetWrapperDirectoryList()
      {
	// Create an empty directory list
	DirectoryList directoryList;

	// ... search in ${OPENTURNS_WRAPPER_PATH}
	// Because OPENTURNS_WRAPPER_PATH is a path, we have to split it
	char * openturnsWrapperDirectory = getenv(Path::OpenturnsWrapperPathVariableName_.c_str());
	if (openturnsWrapperDirectory) {
	  std::string  pathToSplit = openturnsWrapperDirectory;
	  unsigned int lastColonPosition = 0;
	  unsigned int currentColonPosition = 0;
	  while ( ((currentColonPosition = pathToSplit.find(':', lastColonPosition)) >= 0) &&
		  ( currentColonPosition < pathToSplit.size() ) ) {
	    FileName directory(pathToSplit, lastColonPosition, currentColonPosition - lastColonPosition);
	    if (directory.size() == 0) directory = ".";
	    directoryList.push_back(directory);
	    lastColonPosition = currentColonPosition + 1;
	  } /* end while */
	  FileName directory(pathToSplit, lastColonPosition, pathToSplit.size() - lastColonPosition);
	  if (directory.size() == 0) directory = ".";
	  directoryList.push_back(directory);
	}

	// ... search in ${HOME}/openturns/wrappers
	char * homeDirectory = getenv("HOME");
	if (homeDirectory) {
	  FileName directory = homeDirectory;
	  directory += Path::HomeWrapperSubdirectory_;
	  directoryList.push_back(directory);
	}

	// ... search in standard wrapper directory
	FileName directory = Path::GetStandardWrapperDirectory();
	directoryList.push_back(directory);

	return directoryList;
      } /* end getDirectoryList */

      /**
       * Analyse the process environment
       * and return a list of directories to search in for configuration files.
       *
       * The search rule is :look for the file in the following directories, in that order :
       *    + if the env var OPENTURNS_CONFIG_PATH exists, in directories listed in
       *      ${OPENTURNS_CONFIG_PATH} (see openturnsConfigPathVariableName_)
       *    + in directory ${HOME}/openturns/etc (see homeSubdirectory_)
       *    + in standard config directory (either ${OPENTURNS_HOME}/etc/openturns if OPENTURNS_HOME
       *      is defined and if this path is a directory, or the path ${prefix}/etc/openturns
       *      where 'prefix' is the installation path of the platform as provided at configuration time.
       */
      Path::DirectoryList Path::GetConfigDirectoryList()
      {
	// Create an empty directory list
	DirectoryList directoryList;

	// ... search in ${OPENTURNS_CONFIG_PATH}
	// Because OPENTURNS_CONFIG_PATH is a path, we have to split it
	char * openturnsConfigDirectory = getenv(Path::OpenturnsConfigPathVariableName_.c_str());
	if (openturnsConfigDirectory) {
	  std::string  pathToSplit = openturnsConfigDirectory;
	  unsigned int lastColonPosition = 0;
	  unsigned int currentColonPosition = 0;
	  while ( ((currentColonPosition = pathToSplit.find(':', lastColonPosition)) >= 0) &&
		  ( currentColonPosition < pathToSplit.size() ) ) {
	    FileName directory(pathToSplit, lastColonPosition, currentColonPosition - lastColonPosition);
	    if (directory.size() == 0) directory = ".";
	    directoryList.push_back(directory);
	    lastColonPosition = currentColonPosition + 1;
	  } /* end while */
	  FileName directory(pathToSplit, lastColonPosition, pathToSplit.size() - lastColonPosition);
	  if (directory.size() == 0) directory = ".";
	  directoryList.push_back(directory);
	}

	// ... search in ${HOME}/openturns/etc
	char * homeDirectory = getenv("HOME");
	if (homeDirectory) {
	  FileName directory = homeDirectory;
	  directory += Path::HomeConfigSubdirectory_;
	  directoryList.push_back(directory);
	}

	// ... search in standard config directory
	// (${OPENTURNS_HOME}/etc/openturns or ${prefix}/etc/openturns)
	FileName directory;
	bool dirExists = false;
	const char * otHome = getenv(OPENTURNS_HOME_ENV_VAR);
	if (otHome) {
	  directory = String(otHome) + "/etc" + PrefixConfigSubdirectory_;
	  struct stat status;
	  dirExists = (stat( directory.c_str(), &status ) == 0 && ( status.st_mode & S_IFDIR ));
	}
	if (!dirExists) {
	  directory = String(SYSCONFIG_PATH) + PrefixConfigSubdirectory_;
	}
	directoryList.push_back(directory);

	return directoryList;

      }


      /*
       * Find a file named 'name' located in one of the directories
       * listed in 'dirList'. The function returns the full path (directory/name)
       * of the first match.
       */
      FileName Path::FindFileByNameInDirectoryList(const FileName & name, const DirectoryList & dirList)
	throw(FileNotFoundException)
      {
	if (name.size() == 0) throw FileNotFoundException(HERE) << "Can NOT find a file with no name";

	// If the name starts with a slash then it is an absolute path
	// and it did not need to be searched in the directory list
	// so we return it as is.
	if (name[0] == '/') return name;

	// We create the full path name of the file with each directory
	// of the list in turn, and then we check if this file exists
	FileName fullPathForFile;
	DirectoryList::const_iterator currentDirectory;
	for(currentDirectory  = dirList.begin();
	    currentDirectory != dirList.end();
	    currentDirectory++) {
	  Log::Debug(OSS() << "Searching '" << name << "' in directory : " << *currentDirectory);

	  fullPathForFile = *currentDirectory + "/" + name;
	  struct stat fileStatBuffer;
	  if (!stat(fullPathForFile.c_str(), &fileStatBuffer)) // file is found
	    return fullPathForFile;
	}

	// Hmm ! Seems we have a problem...
	// No file was found is the search path so we throw an exception

	OSS errorMessage;
	errorMessage << "No file named '" << name
		     << "' was found in any of those directories :";
	for(currentDirectory  = dirList.begin();
	    currentDirectory != dirList.end();
	    currentDirectory++) {
	  errorMessage << " " << *currentDirectory;
	}
	throw FileNotFoundException(HERE) << String(errorMessage);

      } /* end findFileByNameInDirectoryList */


      /* Build a temporary file name given a pattern */
      String Path::BuildTemporaryFileName(const String & pattern)
      {
	String fullPattern(ResourceMap::GetInstance().get("temporary-directory") + String("/") + pattern);
	UnsignedLong size(fullPattern.size());
	char temporaryFileName[size + 1];
	strcpy(temporaryFileName, fullPattern.c_str());
	temporaryFileName[size] = 0;
#ifndef WIN32
	int fileDescriptor(mkstemp(temporaryFileName));
	close(fileDescriptor);
#else
	mktemp(temporaryFileName);
	// create the file
	std::ofstream outputFile;
	outputFile.open(temporaryFileName);
	outputFile.close();
#endif
	return temporaryFileName;
      }



    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
