//                                               -*- C++ -*-
/**
 *  @file  CenteredFiniteDifferenceHessian.cxx
 *  @brief Class for the creation of a numerical math hessian implementation
 *         form a numerical math evaluation implementation by using centered
 *         finite difference formula.
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: CenteredFiniteDifferenceHessian.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <cstdlib>

#include "CenteredFiniteDifferenceHessian.hxx"
#include "PersistentObjectFactory.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Diff {

      CLASSNAMEINIT(CenteredFiniteDifferenceHessian);

      typedef Func::NumericalMathEvaluationImplementation NumericalMathEvaluationImplementation;
      typedef Common::ResourceMap                         ResourceMap;

      static Common::Factory<CenteredFiniteDifferenceHessian> RegisteredFactory("CenteredFiniteDifferenceHessian");

      const NumericalScalar CenteredFiniteDifferenceHessian::DefaultEpsilon = 1.0e-4;//strtod(ResourceMap::GetInstance().get("CenteredFiniteDifferenceHessian-DefaultEpsilon").c_str(), NULL);

      /* Default constructor */
      CenteredFiniteDifferenceHessian::CenteredFiniteDifferenceHessian() :
        NumericalMathHessianImplementation()
      {
	// Nothing to do
      }

      /* Parameter constructor */
      CenteredFiniteDifferenceHessian::CenteredFiniteDifferenceHessian(const NumericalPoint & epsilon,
								       const EvaluationImplementation & p_evaluation)
        throw(InvalidDimensionException) :
        NumericalMathHessianImplementation(),
        epsilon_(epsilon),
        p_evaluation_(p_evaluation)
      {
	/* Check if the dimension of the constant term is compatible with the linear and quadratic terms */
        if (epsilon.getDimension() != p_evaluation->getInputNumericalPointDimension()) throw InvalidDimensionException(HERE) << "Epsilon dimension is incompatible with the given evaluation";
	/* Check if any epsilon component is exactly zero */
	for(UnsignedLong i = 0; i < epsilon.getDimension(); i++)
	  {
	    if (epsilon[i] == 0.0) throw InvalidArgumentException(HERE) << "At least one of the components of epsilon is equal to 0.0, namely component " << i;
	  }
      }

      /* Parameter constructor */
      CenteredFiniteDifferenceHessian::CenteredFiniteDifferenceHessian(const NumericalScalar epsilon,
								       const EvaluationImplementation & p_evaluation)
        throw(InvalidDimensionException) :
        NumericalMathHessianImplementation(),
        epsilon_(NumericalPoint(p_evaluation->getInputNumericalPointDimension(), epsilon)),
        p_evaluation_(p_evaluation)
      {
	/* Check if epsilon is exactly zero */
	if (epsilon == 0.0) throw InvalidArgumentException(HERE) << "The given scalar epsilon is equal to 0.0";
      }

      /* Virtual constructor */
      CenteredFiniteDifferenceHessian * CenteredFiniteDifferenceHessian::clone() const
      {
	return new CenteredFiniteDifferenceHessian(*this);
      }

      /* Comparison operator */
      Bool CenteredFiniteDifferenceHessian::operator ==(const CenteredFiniteDifferenceHessian & other) const
      {
	return (epsilon_ == other.getEpsilon());
      }
  
      /* String converter */
      String CenteredFiniteDifferenceHessian::__repr__() const {
	OSS oss;
	oss << "class=" << CenteredFiniteDifferenceHessian::GetClassName()
	    << " name=" << getName()
	    << " epsilon=" << epsilon_.__repr__()
            << " evaluation=" << p_evaluation_->__repr__();
	return oss;
      }
  
      /* Accessor for epsilon */
      CenteredFiniteDifferenceHessian::NumericalPoint CenteredFiniteDifferenceHessian::getEpsilon() const
      {
	return epsilon_;
      }

      /* Accessor for the evaluation */
      CenteredFiniteDifferenceHessian::EvaluationImplementation CenteredFiniteDifferenceHessian::getEvaluation() const
      {
	return p_evaluation_;
      }

      /* Here is the interface that all derived class must implement */
	
      /* Hessian () */
      CenteredFiniteDifferenceHessian::SymmetricTensor CenteredFiniteDifferenceHessian::hessian(const NumericalPoint & in) const
	throw(InvalidArgumentException,InternalException)
      {
	UnsignedLong inputDimension(in.getDimension());
	if (inputDimension != epsilon_.getDimension()) throw InvalidArgumentException(HERE) << "Invalid input dimension";
	/* At which points do we have to compute the evaluation for the centered finite difference. We need 2*dim^2+1 points. */
	Stat::NumericalSample gridPoints(2 * inputDimension * inputDimension + 1, in);
	UnsignedLong index(0);
	for(UnsignedLong i = 1; i < inputDimension; i++)
	  {
	    for(UnsignedLong j = 0; j < i; j++)
	      {
		gridPoints[index][i] += epsilon_[i];
		gridPoints[index][j] += epsilon_[j];
		index++;

		gridPoints[index][i] += epsilon_[i];
		gridPoints[index][j] -= epsilon_[j];
		index++;

		gridPoints[index][i] -= epsilon_[i];
		gridPoints[index][j] -= epsilon_[j];
		index++;

		gridPoints[index][i] -= epsilon_[i];
		gridPoints[index][j] += epsilon_[j];
		index++;
	      }
	  }
	for(UnsignedLong i = 0; i < inputDimension; i++)
	  {
	    /* Special case for the diagonal terms, in order to avoid computing twice
	     *  f(x) = f(x + e_i - e_i) = f(x - e_i + e_i) */
	    gridPoints[index][i] += 2.0 * epsilon_[i];
	    index++;
	    gridPoints[index][i] -= 2.0 * epsilon_[i];
	    index++;
	  }
	/* Evaluate the evaluation */
	Stat::NumericalSample gridValues(p_evaluation_->operator()(gridPoints));
	/* Get the center value */
	NumericalPoint center(gridValues[gridValues.getSize() - 1]);
	/* Compute the hessian */
	UnsignedLong outputDimension(p_evaluation_->getOutputNumericalPointDimension());
	SymmetricTensor result(inputDimension, outputDimension);
	UnsignedLong diagonalOffset(2 * inputDimension * (inputDimension - 1));
	NumericalScalar scale;
	for (UnsignedLong i = 0; i < inputDimension; i++)
	  {
	    // Diagonal term
	    /* result(i, i, k) = (f_k(x + 2 * e_i) - 2 * f_k(x) + f_k(x + 2 * e_i)) / (4 * e_i * e_i) */
	    scale = 1.0 / (4.0 * epsilon_[i] * epsilon_[i]);
	    for (UnsignedLong k = 0; k < outputDimension; k++)
	      {
		result(i, i, k) = scale * (gridValues[diagonalOffset + 2 * i][k] - 2.0 * center[k] + gridValues[diagonalOffset + 2 * i + 1][k]);
	      } // k
	    // Compute only the upper part of each sheet, as the hessian is symmetric
	    UnsignedLong offDiagonalOffset(2 * i * (i - 1));
	    for (UnsignedLong j = 0; j < i; j++)
	      {
		scale = 1.0 / (4.0 * epsilon_[i] * epsilon_[j]);
		for (UnsignedLong k = 0; k < outputDimension; k++)
		  {
		    /* result(i, j, k) = (f_k(x + e_i + e_j) - f_k(x + e_i - e_j) + f_k(x - e_i - e_j) - f_k(x - e_i + e_j)) / (4 * e_i * e_j) ~ d2f_k / dx_idx_j */
		    result(i, j, k) = scale * (gridValues[offDiagonalOffset + 4 * j][k] - gridValues[offDiagonalOffset + 4 * j + 1][k] + gridValues[offDiagonalOffset + 4 * j + 2][k] - gridValues[offDiagonalOffset + 4 * j + 3][k]);
		  } //k
	      } // j
	  } // i
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong CenteredFiniteDifferenceHessian::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return p_evaluation_->getInputNumericalPointDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong CenteredFiniteDifferenceHessian::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return p_evaluation_->getOutputNumericalPointDimension();
      }
      
      /* Method save() stores the object through the StorageManager */
      void CenteredFiniteDifferenceHessian::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathHessianImplementation::save(adv);
	adv.writeValue(epsilon_, StorageManager::MemberNameAttribute, "epsilon_");
	adv.writeValue(*p_evaluation_, StorageManager::MemberNameAttribute, "evaluation_");
      }

      /* Method load() reloads the object from the StorageManager */
      void CenteredFiniteDifferenceHessian::load(const StorageManager::Advocate & adv)
      {
	NumericalMathHessianImplementation::load(adv);
	adv.readValue(epsilon_, StorageManager::MemberNameAttribute, "epsilon_");
	Common::TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluation;
	adv.readValue(evaluation, StorageManager::MemberNameAttribute, "evaluation_");
	p_evaluation_ = evaluation.getImplementation();
      }
      
    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
