//                                               -*- C++ -*-
/**
 *  @file  LinearNumericalMathGradientImplementation.cxx
 *  @brief Class for a quadratic numerical math gradient implementation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: LinearNumericalMathGradientImplementation.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "LinearNumericalMathGradientImplementation.hxx"
#include "PersistentObjectFactory.hxx"


namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(LinearNumericalMathGradientImplementation);

      static Common::Factory<LinearNumericalMathGradientImplementation> RegisteredFactory("LinearNumericalMathGradientImplementation");

      /* Default constructor */
      LinearNumericalMathGradientImplementation::LinearNumericalMathGradientImplementation()
	: NumericalMathGradientImplementation()
      {
	// Nothing to do
      }

      /* Parameter constructor */
      LinearNumericalMathGradientImplementation::LinearNumericalMathGradientImplementation(const NumericalPoint & center,
											   const Matrix & constant,
											   const SymmetricTensor & linear)
        throw(InvalidDimensionException) :
        NumericalMathGradientImplementation(),
	center_(center),
        constant_(constant),
	linear_(linear)
      {
	/* Check if the dimensions of the constant term is compatible with the linear term */
        if ((constant.getNbRows() != linear.getNbRows()) || (constant.getNbColumns() != linear.getNbSheets())) throw InvalidDimensionException(HERE) << "Constant term dimensions are incompatible with the linear term";
	/* Check if the dimensions of the center term is compatible with the linear term */
        if ((center.getDimension() != constant.getNbRows()) || (center.getDimension() != linear.getNbRows())) throw InvalidDimensionException(HERE) << "Center term dimensions are incompatible with the constant term or the linear term";
      }

      /* Virtual constructor */
      LinearNumericalMathGradientImplementation * LinearNumericalMathGradientImplementation::clone() const
      {
	return new LinearNumericalMathGradientImplementation(*this);
      }

      /* Comparison operator */
      Bool LinearNumericalMathGradientImplementation::operator ==(const LinearNumericalMathGradientImplementation & other) const
      {
	return ((linear_ == other.linear_) && (constant_ == other.constant_) && (center_ == other.center_));
      }
  
      /* String converter */
      String LinearNumericalMathGradientImplementation::__repr__() const
      {
	OSS oss;
	oss << "class=" << LinearNumericalMathGradientImplementation::GetClassName()
	    << " name=" << getName()
            << " center=" << center_.__repr__()
            << " constant=" << constant_.__repr__()
            << " linear=" << linear_;
	return oss;
      }
  
      /* Accessor for the center */
      LinearNumericalMathGradientImplementation::NumericalPoint LinearNumericalMathGradientImplementation::getCenter() const
      {
	return center_;
      }

      /* Accessor for the constant term */
      LinearNumericalMathGradientImplementation::Matrix LinearNumericalMathGradientImplementation::getConstant() const
      {
	return constant_;
      }

      /* Accessor for the linear term */
      LinearNumericalMathGradientImplementation::SymmetricTensor LinearNumericalMathGradientImplementation::getLinear() const
      {
	return linear_;
      }

      /* Here is the interface that all derived class must implement */
	
      /* Gradient() */
      LinearNumericalMathGradientImplementation::Matrix LinearNumericalMathGradientImplementation::gradient(const NumericalPoint & in) const
	throw(InvalidArgumentException, InternalException)
      {
	if (in.getDimension() != constant_.getNbRows()) throw InvalidArgumentException(HERE) << "Invalid input dimension";
	Matrix value(constant_);
	// Add the linear term <linear, x>
	for(UnsignedLong i = 0; i < linear_.getNbRows(); i++)
	  {
	    for(UnsignedLong j = 0; j < linear_.getNbColumns(); j++)
	      {
		for(UnsignedLong k = 0; k < linear_.getNbSheets(); k++)
		  {
		    value(i, k) += (in[j] - center_[j]) * linear_(i, j, k);
		  }
	      }
	  }
	callsNumber_++;
	return value;
      }

      /* Accessor for input point dimension */
      UnsignedLong LinearNumericalMathGradientImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return center_.getDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong LinearNumericalMathGradientImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return constant_.getNbColumns();
      }
      
      /* Method save() stores the object through the StorageManager */
      void LinearNumericalMathGradientImplementation::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathGradientImplementation::save(adv);
	adv.writeValue(center_, StorageManager::MemberNameAttribute, "center_");
	adv.writeValue(constant_, StorageManager::MemberNameAttribute, "constant_");
	adv.writeValue(linear_, StorageManager::MemberNameAttribute, "linear_");
      }

      /* Method load() reloads the object from the StorageManager */
      void LinearNumericalMathGradientImplementation::load(const StorageManager::Advocate & adv)
      {
	NumericalMathGradientImplementation::load(adv);
	adv.readValue(center_, StorageManager::MemberNameAttribute, "center_");
	adv.readValue(constant_, StorageManager::MemberNameAttribute, "constant_");
	adv.readValue(linear_, StorageManager::MemberNameAttribute, "linear_");
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
