//                                               -*- C++ -*-
/**
 *  @file  UniVariatePolynomial.cxx
 *  @brief This is a 1D polynomial
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "UniVariatePolynomial.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "Graph.hxx"
#include "Curve.hxx"
#include "NumericalSample.hxx"
#include "SquareMatrix.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      CLASSNAMEINIT(UniVariatePolynomial);

      static Common::Factory<UniVariatePolynomial> RegisteredFactory("UniVariatePolynomial");

      typedef Base::Graph::Graph                Graph;
      typedef Base::Graph::Curve                Curve;
      typedef Base::Stat::NumericalSample       NumericalSample;
      typedef Base::Type::SquareMatrix          SquareMatrix;
      typedef Base::Common::NotDefinedException NotDefinedException;

      /* Default constructor */
      UniVariatePolynomial::UniVariatePolynomial()
	: PersistentObject(),
	  coefficients_(1, 0.0)
      {
	// Nothing to do
      }


      /* Constructor from coefficients */
      UniVariatePolynomial::UniVariatePolynomial(const Coefficients & coeffs)
	: PersistentObject(),
	  coefficients_(coeffs)
      {
	compactCoefficients();
      }



      /* Virtual constructor */
      UniVariatePolynomial * UniVariatePolynomial::clone() const
      {
	return new UniVariatePolynomial(*this);
      }


      /* String converter */
      String UniVariatePolynomial::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " coefficients=" << coefficients_.__repr__();
      }


      /* UniVariatePolynomial are evaluated as functors */
      NumericalScalar UniVariatePolynomial::operator() (const NumericalScalar x) const
      {
	const UnsignedLong size(coefficients_.getSize());
	NumericalScalar y(coefficients_[size - 1]); /* y represents the value of P(x)*/

	for (UnsignedLong i = size - 1; i > 0; --i) 
	  {
	    y = y * x + coefficients_[i - 1];
	  }

	return y;
      }

            
      /* Multiply the polynomial P by a NumericalScalar ex: P*3.*/
      UniVariatePolynomial UniVariatePolynomial::operator* (const NumericalScalar scal) const
      {
	if (scal == 0.0) return UniVariatePolynomial();
	return UniVariatePolynomial(coefficients_ * scal);
      } // end method operator* 

      /* Multiply the polynomial by (x to the power deg) */ 
      UniVariatePolynomial UniVariatePolynomial::incrementDegree(const UnsignedLong deg) const
      {
	// Special case for the null coefficient
	if ((getDegree() == 0) && (coefficients_[0] == 0.0)) return *this;
	const UnsignedLong size(coefficients_.getSize());
	// The coefficients are initialized to 0.0
	Coefficients incrementedCoeffs(size + deg);
	// Just shift the coefficients by deg places
	for (UnsignedLong j = 0; j < size; ++j)	
	  {
	    incrementedCoeffs[j + deg] = coefficients_[j];
	  }
	return UniVariatePolynomial(incrementedCoeffs);	
      }// end incrementDegree

      
      /* Realize the summation of two polynomials of any degree ex:P=P1+P2 */
      UniVariatePolynomial UniVariatePolynomial::operator + (const UniVariatePolynomial & poly1D) const
      {
	Coefficients leftCoefficients(coefficients_);
	Coefficients rightCoefficients(poly1D.getCoefficients());
	const UnsignedLong lhsSize(leftCoefficients.getSize());
	const UnsignedLong rhsSize(rightCoefficients.getSize());
	// If the left hand side has a degree greater than the right hand side, add enough zeros to the coefficients in order to equal the degrees
	if (lhsSize > rhsSize) for (UnsignedLong i = rhsSize; i < lhsSize; ++i) rightCoefficients.add(0.0);
	// Else the right hand side has a degree greater than the left hand side, add enough zeros to the coefficients in order to equal the degrees
	else for (UnsignedLong i = lhsSize; i < rhsSize; ++i) leftCoefficients.add(0.0);
	// Then, we just have to sum-up the degrees
	UniVariatePolynomial sum(leftCoefficients + rightCoefficients);
	sum.compactCoefficients();
	return sum;
      } // end summation of P1 & P2


      /* Realize the substraction of two polynomials of any degree ex:P=P1-P2 */
      UniVariatePolynomial UniVariatePolynomial::operator - (const UniVariatePolynomial & poly1D) const
      {
	return operator + (poly1D * (-1.0));
      } // end substraction of P1 & P2 */

      /* Coefficients accessor */
      void UniVariatePolynomial::setCoefficients(const Coefficients & coeffs)
      {
	coefficients_ = coeffs;
	compactCoefficients();
      }
      
      const UniVariatePolynomial::Coefficients & UniVariatePolynomial::getCoefficients() const
      {
	return coefficients_;
      }


      /* Method to draw the graph of the polynomial between given bounds */
      UniVariatePolynomial::Graph UniVariatePolynomial::draw(const NumericalScalar xMin, const NumericalScalar xMax, const UnsignedLong pointNumber) const
      {
	NumericalSample data(pointNumber, 2);
	for (UnsignedLong i = 0; i < pointNumber; ++i)
	  {
	    const NumericalScalar x(xMin + (xMax - xMin) * static_cast<NumericalScalar>(i) / static_cast<NumericalScalar>(pointNumber - 1.0));
	    data[i][0] = x;
	    data[i][1] = operator()(x);
	  }
	Curve curve(data, "red", "solid", 2, getName());
	Graph graph(getName(), "x", "y", true, "topright");
	graph.addDrawable(curve);
	return graph;
      }


      /* Get the degree of the polynomial */
      UnsignedLong UniVariatePolynomial::getDegree() const
      {
	return coefficients_.getDimension() - 1;
      }


      /* Root of the polynomial of degree n as the eigenvalues of the associated  matrix */
      UniVariatePolynomial::NumericalComplexCollection UniVariatePolynomial::getRoots() const
      {
	const UnsignedLong degree(getDegree());
	if (degree == 0) throw NotDefinedException(HERE) << "Error: cannot compute the roots of a constant polynomial.";
	const NumericalScalar scale(-1.0 / coefficients_[degree]);
	SquareMatrix m(degree);
	m(0, degree - 1) = coefficients_[0] * scale;
	for (UnsignedLong i = 1; i < degree; ++i)
	  {
	    m(i, i - 1) = 1.0;
	    m(i, degree - 1) = coefficients_[i] * scale;
	  }
	return m.computeEigenValues();
      }

      /* remove null leading coefficients. Special case for the null coefficient, which is constant so we don't remove this particular zero. */
      void UniVariatePolynomial::compactCoefficients()
      {
	UnsignedLong degree(coefficients_.getDimension() - 1);
	while ((degree > 0) && (coefficients_[degree] == 0.0))
	  {
	    coefficients_.erase(degree);
	    --degree;
	  }
      }

      /* Method save() stores the object through the StorageManager */
      void UniVariatePolynomial::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
	adv.writeValue(coefficients_, StorageManager::MemberNameAttribute, "coefficients_");
      }

      /* Method load() reloads the object from the StorageManager */
      void UniVariatePolynomial::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
	adv.readValue(coefficients_, StorageManager::MemberNameAttribute, "coefficients_");
      }
      

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
