//                                               -*- C++ -*-
/**
 *  @file  WrapperFile.hxx
 *  @brief This class provides all the treatments for wrapper file manipulation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: WrapperFile.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_WRAPPERFILE_HXX
#define OPENTURNS_WRAPPERFILE_HXX

#include <iostream>              // for std::ostream
#include "OTprivate.hxx"
#include "PersistentObject.hxx"
#include "WrapperData.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {


      /**
       * @class WrapperFile
       *
       * This class provides all the treatments for wrapper file manipulation
       */

      class WrapperFile
	: public Common::PersistentObject
      {
	CLASSNAME;
      public:

	typedef Common::NoWrapperFileFoundException NoWrapperFileFoundException;
	typedef Common::WrapperFileParsingException WrapperFileParsingException;

	/**
	 * Find the path of a wrapper file from its name.
	 */
	static FileName FindWrapperPathByName(const String & name)
	  throw(NoWrapperFileFoundException);

	/**
	 * Build a wrapper from a name
	 */
	static WrapperFile FindWrapperByName(const String & name)
	  throw(NoWrapperFileFoundException);

	/**
	 * Build a wrapper from a stream
	 */
	static WrapperFile BuildWrapperFromStream(const String & xmlStream);

      protected:

	/** The file name extension */
	static const String extension_;


      public:

	/** Default constructor */
	WrapperFile();

	/** Constructor from file */
	explicit WrapperFile(const FileName & pathToFile)
	  throw(WrapperFileParsingException);

	/** Virtual constructor */
	virtual WrapperFile * clone() const;

	/** String converter */
	virtual String __repr__() const;

	/** Description file path accessor */
	void setDescriptionFilePath(const FileName & path);
	FileName getDescriptionFilePath() const;

	/** Wrapper data accessor */
	void setWrapperData(const WrapperData & data);
	const WrapperData & getWrapperData() const;

	/** Write the internal data to a wrapper file */
	void writeFile(const FileName & pathToFile);

	/** Stream out the internal data */
	String toString() const;


      protected:

	/** Where the description file of the wrapper is located */
	FileName descriptionFilePath_;

	/** The data that the wrapper may exchange with the platform */
	WrapperData data_;

      private:

	/** Initialization */
	void init() const
	  throw(WrapperFileParsingException);

	/** Finalization */
	void done() const;

	/** File parsing function */
	void parseFile(const FileName & pathToFile)
	  throw(WrapperFileParsingException);

	/** Stream parsing function */
	void parseStream(const String & stream);

      } ; /* class WrapperFile */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_WRAPPERFILE_HXX */
