//                                               -*- C++ -*-
/**
 *  @file  LowDiscrepancySequenceImplementation.cxx
 *  @brief This class is a top-level class for low discrepancy sequences
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy$
 *  @date:   $LastChangedDate$
 *  Id:      $Id$
 */
#include "LowDiscrepancySequenceImplementation.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Stat
    {
      /**
       * @class LowDiscrepancySequenceImplementation
       */

      typedef Common::NotYetImplementedException NotYetImplementedException;
      typedef Common::InvalidRangeException	 InvalidRangeException;

      CLASSNAMEINIT(LowDiscrepancySequenceImplementation);

      /* Default constructor */
      LowDiscrepancySequenceImplementation::LowDiscrepancySequenceImplementation() :
	Base::Common::PersistentObject(),
	isInitialized_(false)
      {
	// Nothing to do
      }


      /* Constructor with parameters */
      LowDiscrepancySequenceImplementation::LowDiscrepancySequenceImplementation(const UnsignedLong dimension) :
	Base::Common::PersistentObject(),
	isInitialized_(false),
	dimension_(dimension)
      {
	// Nothing to do
      }


      /* Virtual constructor */
      LowDiscrepancySequenceImplementation * LowDiscrepancySequenceImplementation::clone() const
      {
	return new LowDiscrepancySequenceImplementation(*this);
      }


      /* initialize the sequence */
      void LowDiscrepancySequenceImplementation::initialize(const UnsignedLong dimension)
      {
	throw NotYetImplementedException(HERE); // this function is supposed to be pure virtual
      }


      /* Has the sequence been initialized ? */
      Bool LowDiscrepancySequenceImplementation::isInitialized() const
      {
	return isInitialized_;
      }


      /* Dimension accessor*/
      UnsignedLong LowDiscrepancySequenceImplementation::getDimension() const
      {
	return dimension_;
      }


      /* Generate a low discrepancy vector of numbers uniformly distributed over [0, 1) */
      LowDiscrepancySequenceImplementation::NumericalPoint LowDiscrepancySequenceImplementation::generate()
      {
	throw NotYetImplementedException(HERE); // this function is supposed to be pure virtual
      }


      /* Generate a sample of pseudo-random vectors of numbers uniformly distributed over [0, 1) */
      NumericalSample LowDiscrepancySequenceImplementation::generate(const UnsignedLong size)
      {
	NumericalSample sequenceSample(size, getDimension());
	if(isInitialized())
	  {
	    for(UnsignedLong i = 0; i < size ; ++i)
	      {
		sequenceSample[i] = generate();
	      }
	  }
	return sequenceSample;
      }


      /* Compute the star discrepancy of a sample uniformly distributed over [0, 1) */
      NumericalScalar LowDiscrepancySequenceImplementation::ComputeStarDiscrepancy(const NumericalSample & sample)
      {
	// computationnaly heavy function : O(N²), let N the size of the sample
	const UnsignedLong size(sample.getSize());

	// discrepancy is the maximum of the local discrepancy
	NumericalScalar discrepancy(0.0);
	for(UnsignedLong i = 0; i < size; ++ i)
	  {
	    NumericalScalar local(ComputeLocalDiscrepancy(sample, sample[i]));
	    if(local > discrepancy)
	      discrepancy = local;
	  }
	return discrepancy;
      }


      /* String converter */
      String LowDiscrepancySequenceImplementation::__repr__() const
      {
	OSS oss;
	oss << "class=" << LowDiscrepancySequenceImplementation::GetClassName()
	    << " isInitialized=" << isInitialized_
	    << " dimension=" << dimension_;
	return oss;
      }


      /* Compute the local discrepancy of a sample, given a multidimensionnal interval */
      NumericalScalar LowDiscrepancySequenceImplementation::ComputeLocalDiscrepancy(const NumericalSample & sample,
										    const NumericalPoint & interval)
      {
	const UnsignedLong dimension(sample.getDimension());

	// calculate the Lebesgue's measure of the interval
	NumericalScalar measure(1.0);
	for(UnsignedLong k = 0; k < dimension; ++ k)
	  measure *= interval[k];
	
	// calculate number of inner points
	const UnsignedLong size(sample.getSize());
	UnsignedLong innerPoint(0);
	for(UnsignedLong j = 0; j < size; ++ j)
	  {
	    // determine if the point is in the interval
	    Bool inner(true);
	    UnsignedLong k(0);
	    while(inner && (k < dimension))
	      {
		// if one component is outside interval
		if(sample[j][k] > interval[k]) 
		  inner = false;
		++ k;
	      }
	    if(inner)
	      ++ innerPoint;
	  }
	return fabs((NumericalScalar)innerPoint / size - measure);
      }


    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */
