/*                                               -*- C++ -*- */
/*
 * @file  csv_parser.yy
 * @brief The parser definition in order to read CSV files
 *
 * (C) Copyright 2005-2006 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * @author $LastChangedBy: dutka $
 * @date   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 */

/* This parser is based on RFC 4180 from www.ietf.org */

%pure-parser
%parse-param {OT::Base::Stat::CSVParserState & theState}
%parse-param {yyscan_t yyscanner}
%parse-param {FILE * theFile}
%parse-param {OT::Base::Stat::NumericalSampleImplementation &impl}
%parse-param {OT::UnsignedLong & theDimension}
%parse-param {const char * theSeparator}
%lex-param   {yyscan_t yyscanner}
%lex-param   {FILE * theFile}
%lex-param   {const char * theSeparator}
%defines
%name-prefix="csv"

%union {
  double real;
  const char * st;
}

%token <real>      REAL
%token <st>        STRING
%token <st>        COMMA
%token             CRLF
%token             DOUBLEQUOTE
%token             CHAR

%left              COMMA
%left              CRLF

%type <st>         escapedData
%type <st>         data
%type <st>         CRLF
%type <st>         DOUBLEQUOTE
%type <st>         error



%{
#include <string>
#include "csv_parser_state.hxx"
#include "NumericalSampleImplementation.hxx"
#include "Log.hxx"

typedef OT::Base::Common::Log Log;
typedef OT::Base::Stat::CSVParserState CSVParserState;
typedef void*                 yyscan_t;
int yylex                     (YYSTYPE *lvalp, yyscan_t yyscanner, FILE * theFile, const char * theSeparator);
int yyerror                   (OT::Base::Stat::CSVParserState & theState, yyscan_t yyscanner, FILE * theFile, OT::Base::Stat::NumericalSampleImplementation &impl, OT::UnsignedLong & theDimension, const char * theSeparator, const char *s);
int csvget_lineno             (yyscan_t yyscanner);


std::string ToString(double val)
{
  std::ostringstream oss;
  oss << val;
  return oss.str();
}

void clearPoint(OT::Base::Stat::CSVParserState & theState, yyscan_t yyscanner)
{
  theState.Point = OT::Base::Type::NumericalPoint();
}
void printPoint(OT::Base::Stat::CSVParserState & theState, yyscan_t yyscanner, OT::Base::Stat::NumericalSampleImplementation &impl, OT::UnsignedLong & theDimension)
{
  Log::Debug(OT::OSS() << "file " << theState.theFileName << " line " << csvget_lineno(yyscanner) << ": Point=" << theState.Point.__repr__());

  if (theDimension == 0) {
    theDimension = theState.Point.getDimension();
    impl.add(theState.Point);
  } else if (theDimension == theState.Point.getDimension()) impl.add(theState.Point);
}


void clearHeader(OT::Base::Stat::CSVParserState & theState, yyscan_t yyscanner)
{
  theState.Header = OT::Base::Type::Description();
}
void printHeader(OT::Base::Stat::CSVParserState & theState, yyscan_t yyscanner, OT::Base::Stat::NumericalSampleImplementation &impl, OT::UnsignedLong & theDimension)
{
  Log::Debug(OT::OSS() << "file " << theState.theFileName << " line " << csvget_lineno(yyscanner) << ": Header=" << theState.Header.__repr__());

  if (theDimension == 0) {
    theDimension = theState.Header.getSize();
    impl.setDescription(theState.Header);
  }
}
%}


%start CSVFile
%%

CSVFile:  endOfFile
	| recordSet endOfFile
	;

recordSet: Record 
	|  recordSet CRLF Record 
	;

Record: { clearHeader(theState,yyscanner); clearPoint(theState,yyscanner); theState.errors = false; }
	record
        { if (!theState.errors) { printHeader(theState,yyscanner,impl,theDimension); printPoint(theState,yyscanner,impl,theDimension); } } ;

record:   field
	| record COMMA field
	| record error
	;

field:    { theState.St = ""; } escapedField    { theState.Header.add(theState.St); }
| { theState.St = ""; } nonEscapedField { if (theState.Type==CSVParserState::RealField) theState.Point.add(theState.Val); else theState.Header.add(theState.St); }
	;



escapedField: { theState.Type = CSVParserState::StringField; } DOUBLEQUOTE escapedDataList DOUBLEQUOTE ;

escapedDataList: escapedData 
	|        escapedDataList escapedData
	;

escapedData: data
	|    COMMA                   { theState.St += $1; }
	|    CRLF                    { theState.St += '\n'; }
	|    DOUBLEQUOTE DOUBLEQUOTE { theState.St += '"'; }
	;

nonEscapedField: { theState.Type = CSVParserState::NotSet; } data 
	|        { theState.Type = CSVParserState::NotSet; } error data
	;

data:     REAL   { if (theState.Type==CSVParserState::NotSet) { theState.Val = $1; theState.Type = CSVParserState::RealField; } else { theState.St += ToString($1); }  }
	| STRING { theState.St += $1; theState.Type = CSVParserState::StringField; }
	;

endOfFile: /* Empty */
	| CRLF
	;

%%

int yyerror(OT::Base::Stat::CSVParserState & theState, yyscan_t yyscanner, FILE * theFile, OT::Base::Stat::NumericalSampleImplementation &impl, OT::UnsignedLong & theDimension, const char * theSeparator, const char *s) {
  Log::Info(OT::OSS() << "file " << theState.theFileName << " line " << csvget_lineno(yyscanner) << " is ignored: " << s);
  theState.errors = true;
  return 0;
}
