//                                               -*- C++ -*-
/**
 *  @file  Collection.hxx
 *  @brief Collection defines top-most collection strategies
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Collection.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_COLLECTION_HXX
#define OPENTURNS_COLLECTION_HXX

#include <vector>
#include <algorithm>         // for std::copy
#include "OTprivate.hxx"
#include "OSS.hxx"
#include "OStream.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {
      /**
       * @class Collection
       *
       * Collection defines top-most collection strategies
       */
      
      /* Forward declaration */
      template <class T> class Collection;


      /** Comparison operator (friend) */
      template <class T> inline
      Bool operator == (const Collection<T> & lhs,
			const Collection<T> & rhs)
      {
	return lhs.coll__ == rhs.coll__;
      }


      /** Ordering operator (friend) */
      template <class T> inline
      Bool operator < (const Collection<T> & lhs,
		       const Collection<T> & rhs)
      {
	return lhs.coll__ < rhs.coll__;
      }


      /** Stream operator */
      template <class T> inline
      std::ostream & operator << (std::ostream & os,
				  const Collection<T> & collection)
      {
	return os << collection.__repr__();
      }

      template <class T> inline
      Common::OStream & operator << (Common::OStream & OS,
				     const Collection<T> & collection)
      {
	return OS << collection.__str__();
      }





      template <class T>
      class Collection
      {
      public:

	typedef T ElementType;
	typedef T ValueType;

	typedef typename std::vector<T>                       InternalType;
	typedef typename InternalType::iterator               iterator;
	typedef typename InternalType::const_iterator         const_iterator;
	typedef typename InternalType::reverse_iterator       reverse_iterator;
	typedef typename InternalType::const_reverse_iterator const_reverse_iterator;

	typedef Common::InvalidArgumentException              InvalidArgumentException;

      protected:

	/* The actual collection is a STL vector */
	InternalType coll__;

      public:

	/** Default constructor */
	Collection()
	  : coll__()
	{
	  // Nothing to do
	}

	/** Constructor that pre-allocate size elements */
	Collection(const UnsignedLong size)
	  : coll__(size)
	{
	  // Nothing to do
	}

	/** Constructor that pre-allocate size elements with value */
	Collection(const UnsignedLong size,
		   const T & value)
	  : coll__(size, value)
	{
	  // Nothing to do
	}

	/** Constructor from a range of elements */
	template <typename InputIterator>
	Collection(const InputIterator first,
		   const InputIterator last)
	  : coll__(first, last)
	{
	  // Nothing to do
	}

#ifndef SWIG
	/** Erase the elements between first and last */
	inline
	iterator erase(const iterator first,
		       const iterator last)
	{
	  if ( (first < begin()) ||
	       (first > end())   ||
	       (last  < begin()) ||
	       (last  > end())   )
	    throw InvalidArgumentException(HERE) << "Can NOT erase value outside of collection";
	  return coll__.erase(first, last);
	}

	/** Erase the elements pointed by position */
	inline
	iterator erase(iterator position)
	{
	  if ( (position < begin()) ||
	       (position > end())   )
	    throw InvalidArgumentException(HERE) << "Can NOT erase value outside of collection";
	  return coll__.erase(position);
	}
#endif

	/** Clear all elements of the collection */
	inline
	void clear()
	{
	  coll__.clear();
	}

	/** Assign elements to the collection */
	template <typename InputIterator>
	inline
	void assign(const InputIterator first,
		    const InputIterator last)
	{
	  coll__.assign(first, last);
	}

#ifndef SWIG
	/** Operator[]() gives access to the elements of the collection */
	inline
	T & operator [] (const UnsignedLong i)
	{
	  return coll__.at(i);
	}

	/** Operator[]() gives access to the elements of the const collection */
	inline
	const T & operator [] (const UnsignedLong i) const
	{
	  return coll__.at(i);
	}
#endif
	
	/* Method __getitem__() is for Python */
	inline
	T __getitem__(const UnsignedLong i) const
	{
	  return coll__.at(i);
	}

	/* Method __getitem__() is for Python */
	/*inline
	T & __getitem__(UnsignedLong i)
	{
	  return coll__.at(i);
	  }*/

	/* Method __setitem__() is for Python */
	inline
	void __setitem__(const UnsignedLong i,
			 const T val)
	{
	  coll__.at(i) = val;
	}

	/** At() gives access to the elements of the collection but throws an exception if bounds are overcome */
	inline
	T & at(const UnsignedLong i)
	{
	  return coll__.at(i);
	}

	/** At() gives access to the elements of the const collection but throws an exception if bounds are overcome */
	inline
	const T & at(const UnsignedLong i) const
	{
	  return coll__.at(i);
	}

	/** Method add() appends an element to the collection */
	inline
	void add(const T & elt)
	{
	  coll__.push_back(elt);
	}

	/** Method getSize() returns the number of elements of the collection */
	inline
	UnsignedLong getSize() const
	{
	  return coll__.size();
	}

	/** Method resize() changes the size of the Collection. If the new size is smaller than the older one, the last elements are thrown away, else the new elements are setted to the default value of the element type */
	inline
	void resize(const UnsignedLong newSize)
	{
	  coll__.resize(newSize);
	}

	/** Method empty() returns true if there is no element in the collection */
	inline
	Bool isEmpty() const
	{
	  return coll__.empty();
	}

#ifndef SWIG
	/** Method begin() points to the first element of the collection */
	inline
	iterator begin()
	{
	  return coll__.begin();
	}
	inline
	const_iterator begin() const
	{
	  return coll__.begin();
	}

	/** Method end() points beyond the last element of the collection */
	inline
	iterator end()
	{
	  return coll__.end();
	}
	inline
	const_iterator end() const
	{
	  return coll__.end();
	}

	/** Method rbegin() points to the last element of the collection */
	inline
	reverse_iterator rbegin()
	{
	  return coll__.rbegin();
	}
	inline
	const_reverse_iterator rbegin() const
	{
	  return coll__.rbegin();
	}

	/** Method rend() points before the first element of the collection */
	inline
	reverse_iterator rend()
	{
	  return coll__.rend();
	}
	inline
	const_reverse_iterator rend() const
	{
	  return coll__.rend();
	}
#endif

      protected:

	inline
	String toString(Bool full) const
	{
	  OSS oss(full);
	  oss << "[";
	  std::copy( begin(), end(), OSS_iterator<T>(oss, ",") );
	  oss << "]";	
	  return oss;
	}


      public:

	/** String converter */
	inline String __repr__() const { return toString(true); }
	inline String __str__() const { return toString(false); }


#ifndef SWIG
	/* Friend operator */
	template <class U> friend inline
	Bool operator == (const Collection<U> & lhs, const Collection<U> & rhs);

	/* Friend operator */
	template <class U> friend inline
	Bool operator < (const Collection<U> & lhs, const Collection<U> & rhs);
#endif

      }; /* class Collection */


    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_COLLECTION_HXX */
