//                                               -*- C++ -*-
/**
 *  @file  Matrix.cxx
 *  @brief Matrix implements the classical mathematical matrix
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Matrix.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "Matrix.hxx"
#include "MatrixImplementation.hxx"
#include "SymmetricMatrix.hxx"
#include "IdentityMatrix.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      CLASSNAMEINIT(Matrix);

      /* Default constructor */
      Matrix::Matrix()
	: Common::TypedInterfaceObject<MatrixImplementation>(new MatrixImplementation())
      {
        // Nothing to do
      }
      
      /* Constructor with size (rowDim and colDim) */
      /* The matrix is made up of a collection of rowDim*colDim elements */
      /* The matrix is viewed as a set of column vectors read one after another */
      Matrix::Matrix(const UnsignedLong rowDim,
      		     const UnsignedLong colDim)
	: Common::TypedInterfaceObject<MatrixImplementation>(new MatrixImplementation(rowDim, colDim))
      {
        // Nothing to do
      }
   
      /* Constructor from external collection */
      /* If the dimensions of the matrix and of the collection */
      /* do not match, either the collection is truncated */
      /* or the rest of the matrix is filled with zeros */
      Matrix::Matrix(const UnsignedLong rowDim,
		     const UnsignedLong colDim,
	             const Collection<NumericalScalar> & elementsValues)
	: Common::TypedInterfaceObject<MatrixImplementation>(new MatrixImplementation(rowDim, colDim, elementsValues))
      {
        // Nothing to do
      }
      
      /* Constructor with implementation */
      Matrix::Matrix(const Implementation & i)
	: Common::TypedInterfaceObject<MatrixImplementation>(i) 
      {
        // Nothing to do
      }


      /* String converter */
      String Matrix::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " implementation=" << getImplementation()->__repr__();
      }


      /* Get the dimensions of the matrix : number of rows */
      const UnsignedLong Matrix::getNbRows() const
      {
        return getImplementation()->getNbRows();
      }
	
      /* Get the dimensions of the matrix : number of columns */
      const UnsignedLong Matrix::getNbColumns() const
      {
        return getImplementation()->getNbColumns();
      }
      
      /* Operator () gives access to the elements of the matrix (to modify these elements) */
      /* The element of the matrix is designated by its row number i and its column number j */
      /* the first element of the matrix is m(0,0) */
      NumericalScalar & Matrix::operator() (const UnsignedLong i,
					    const UnsignedLong j) throw(InvalidDimensionException)
      {
	copyOnWrite();
        return (*getImplementation())(i,j);
      }
	
      /* Operator () gives access to the elements of the matrix (read only) */
      /* The element of the matrix is designated by its row number i and its column number j */
      const NumericalScalar & Matrix::operator() (const UnsignedLong i,
						  const UnsignedLong j)  const throw(InvalidDimensionException)
      {
	return (*getImplementation())(i,j);
      }

      
      /* Matrix transpose */
      Matrix Matrix::transpose () const
      {
	return Matrix(new MatrixImplementation(getImplementation()->transpose()));
      }
      
      /* Matrix additions (must have the same dimensions) */
      Matrix Matrix::operator+ (const Matrix & m) const throw(InvalidDimensionException)
      {
        return Matrix(new MatrixImplementation(*getImplementation() + *(m.getImplementation()) ));
      }
      
      Matrix Matrix::operator+ (const SymmetricMatrix & m) const throw(InvalidDimensionException)
      {
        (m.getImplementation())->symmetrize();
        return Matrix(new MatrixImplementation(*getImplementation() + *(m.getImplementation()) ));
      }
      
      /* Matrix substractions (must have the same dimensions) */
      Matrix Matrix::operator- (const Matrix & m) const throw(InvalidDimensionException)
      {
	return Matrix(new MatrixImplementation(*getImplementation() - *(m.getImplementation()) ));
      }
      
      Matrix Matrix::operator- (const SymmetricMatrix & m) const throw(InvalidDimensionException)
      {
        (m.getImplementation())->symmetrize();
	return Matrix(new MatrixImplementation(*getImplementation() - *(m.getImplementation()) ));
      }
      
      /* Matrix multiplications (must have consistent dimensions) */
      Matrix Matrix::operator* (const Matrix & m) const throw(InvalidDimensionException)
      {
        return Matrix(new MatrixImplementation(getImplementation()->genProd(*(m.getImplementation())) ));
      }
      
      Matrix Matrix::operator* (const SymmetricMatrix & m) const throw(InvalidDimensionException)
      {
        return Matrix(new MatrixImplementation(getImplementation()->symProd(*(m.getImplementation()),'R') ));
      }
      
      Matrix Matrix::operator* (const IdentityMatrix & m) const throw(InvalidDimensionException)
      {
        return *this;
      }
      
      /* Multiplication with a NumericalPoint (must have consistent dimensions) */
      NumericalPoint Matrix::operator* (const NumericalPoint & pt) const throw(InvalidDimensionException)
      {
        return getImplementation()->genVectProd(pt) ;
      }
      
      /* Multiplication with a NumericalScalar */
      Matrix Matrix::operator* (const NumericalScalar s) const
      {
        return Matrix(new MatrixImplementation(*getImplementation() * s ));
      }
      
      /* Division by a NumericalScalar*/
      Matrix Matrix::operator/ (const NumericalScalar s) const throw(InvalidArgumentException)
      {
        return Matrix(new MatrixImplementation(*getImplementation() / s ));
      }
      
      /* Resolution of a linear system */
      NumericalPoint Matrix::solveLinearSystem(const NumericalPoint & b) const throw(InvalidDimensionException)
      {
        return getImplementation()->solveLinearSystemRect(b);
      }
      
      Matrix Matrix::solveLinearSystem(const Matrix & b) const throw(InvalidDimensionException)
      {
        return Matrix(new MatrixImplementation(getImplementation()->solveLinearSystemRect(*(b.getImplementation()))));
      }
      
      /* Empty returns true if there is no element in the matrix */
      const Bool Matrix::isEmpty() const
      {
        return getImplementation()->isEmpty() ;
      }
      

      /* Comparison operator */
      Bool Matrix::operator == (const Matrix & rhs) const 
      {
        const Matrix &lhs(*this);
	return (*(lhs.getImplementation()) == *(rhs.getImplementation()) );
      }



    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
