
//                                               -*- C++ -*-
/**
 *  @file  MatrixImplementation.cxx
 *  @brief MatrixImplementation implements the classical mathematical MatrixImplementation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: MatrixImplementation.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <cstdlib>

#include "MatrixImplementation.hxx"
#include "PersistentObjectFactory.hxx"
#include "Lapack.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      CLASSNAMEINIT(MatrixImplementation);
      
      typedef Common::ResourceMap ResourceMap;

      static Common::Factory<MatrixImplementation> RegisteredFactory("MatrixImplementation");

      // All the pivots with a magnitude less than this threshold are considered as zero
      const NumericalScalar MatrixImplementation::DefaultSmallPivot = 1.0e-7;//strtod(ResourceMap::GetInstance().get("MatrixImplementation-DefaultSmallPivot").c_str(), NULL);

      /* Default constructor */
      MatrixImplementation::MatrixImplementation()
	: PersistentCollection<NumericalScalar>(),
	  nbRows_(0),
	  nbColumns_(0)
      {
        // Nothing to do
      }

      /* Constructor with size (rowDim and colDim) */
      /* The MatrixImplementation is made up of a collection of rowDim*colDim elements */
      /* The MatrixImplementation is viewed as a set of column vectors read one after another */
      MatrixImplementation::MatrixImplementation(const UnsignedLong rowDim,
      		                                 const UnsignedLong colDim)
	: PersistentCollection<NumericalScalar>(rowDim * colDim, 0.0), 
	  nbRows_(rowDim),
	  nbColumns_(colDim)
      {
        // Nothing to do
      }
      
      /* Constructor from external collection */
      MatrixImplementation::MatrixImplementation(const UnsignedLong rowDim,
                                                 const UnsignedLong colDim,
						 const Collection<NumericalScalar> & elementsValues)
	: PersistentCollection<NumericalScalar>(rowDim * colDim, 0.0),
	  nbRows_(rowDim),
	  nbColumns_(colDim)
      {
        const UnsignedLong matrixSize = std::min(rowDim * colDim, elementsValues.getSize());
        for(UnsignedLong i = 0; i < matrixSize; ++i)
	  {
	    (*this)[i] = elementsValues[i];
	  }
      }

      /* Virtual constructor */
      MatrixImplementation * MatrixImplementation::clone() const
      {
	return new MatrixImplementation(*this);
      }


      /* String converter */
      String MatrixImplementation::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " name=" << getName()
		     << " rows=" << getNbRows()
		     << " columns=" << getNbColumns()
		     << " values=" << PersistentCollection<NumericalScalar>::__repr__();
      }
      
      String MatrixImplementation::__str__() const
      {
	return __repr__();
      }

     /* Operator () gives access to the elements of the MatrixImplementation (to modify these elements) */
      /* The element of the MatrixImplementation is designated by its row number i and its column number j */
      /* the first element of the MatrixImplementation is m(0,0) */
      NumericalScalar & MatrixImplementation::operator() (const UnsignedLong i,
							  const UnsignedLong j) throw(InvalidDimensionException)
      {
        if ((i >= nbRows_) || (j >= nbColumns_)) throw InvalidDimensionException(HERE);
	
        return (*this)[this->convertPosition(i, j)];
      }
	
      /* Operator () gives access to the elements of the MatrixImplementation (read only) */
      /* The element of the MatrixImplementation is designated by its row number i and its column number j */
      const NumericalScalar & MatrixImplementation::operator() (const UnsignedLong i,
								const UnsignedLong j)  const throw(InvalidDimensionException)
      {
        if ((i >= nbRows_) || (j >= nbColumns_)) throw InvalidDimensionException(HERE);
	
        return (*this)[this->convertPosition(i, j)];
      }


      /* Get the dimensions of the MatrixImplementation : number of rows */
      const UnsignedLong MatrixImplementation::getNbRows() const
      {
        return nbRows_;
      }
	
      /* Get the dimensions of the MatrixImplementation : number of columns */
      const UnsignedLong MatrixImplementation::getNbColumns() const
      {
        return nbColumns_;
      }
      
      /* Get the dimensions of the MatrixImplementation : dimension (square matrix : nbRows_) */
      const UnsignedLong MatrixImplementation::getDimension() const
      {
        return nbRows_;
      }
      
      /* MatrixImplementation transpose */
      MatrixImplementation MatrixImplementation::transpose () const
      {
        MatrixImplementation trans(nbColumns_, nbRows_);
	UnsignedLong i, j;
	for (i = 0; i < nbColumns_; ++i)
	  for (j = 0; j < nbRows_; ++j)
	    trans(i, j) = this->operator()(j, i);
	return trans;
      }
      
      /* MatrixImplementation addition (must have the same dimensions) */
      MatrixImplementation MatrixImplementation::operator+ (const MatrixImplementation & m) const throw(InvalidDimensionException)
      {
        if ((nbRows_ != m.getNbRows() ) || (nbColumns_ != m.getNbColumns() )) throw InvalidDimensionException(HERE);
	MatrixImplementation add(m);
	int size(nbRows_ * nbColumns_);
	double alpha(1.0);
	int one(1);
	DAXPY_F77(&size, &alpha, const_cast<double*>(&((*this)[0])), &one, &add[0], &one);

	return add;
      }
      
      /* MatrixImplementation substraction (must have the same dimensions) */
      MatrixImplementation MatrixImplementation::operator- (const MatrixImplementation & m) const throw(InvalidDimensionException)
      {
        if ((nbRows_ != m.getNbRows() ) || (nbColumns_ != m.getNbColumns() )) throw InvalidDimensionException(HERE);
      
	MatrixImplementation sub(*this);
	int size(nbRows_ * nbColumns_);
	double alpha(-1.0);
	int one(1);
	DAXPY_F77(&size, &alpha, const_cast<double*>(&(m[0])), &one, &sub[0], &one);

	return sub;	
      }
      
      /* MatrixImplementation multiplications (must have consistent dimensions) */
      MatrixImplementation MatrixImplementation::genProd (const MatrixImplementation & m) const throw(InvalidDimensionException)
      {
        if (nbColumns_ != m.nbRows_) throw InvalidDimensionException(HERE) << "Invalid dimensions in matrix/matrix product left="
									   << nbRows_ << "x" << nbColumns_
									   << " right=" << m.getNbRows() << "x"
									   << m.getNbColumns();
	
	MatrixImplementation mult(nbRows_, m.nbColumns_);
	char transa('N');
	char transb('N');
	int m_(nbRows_);
	int k_(nbColumns_);
	int n_(m.nbColumns_);
	double alpha(1.0);
	double beta(0.0);
	int ltransa(1);
	int ltransb(1);
	DGEMM_F77(&transa, &transb, &m_, &n_, &k_, &alpha, const_cast<double*>(&((*this)[0])), &m_, const_cast<double*>(&(m[0])), &k_, &beta, &mult[0], &m_, &ltransa, &ltransb);

        return mult;
      }
      
      MatrixImplementation MatrixImplementation::symProd (const MatrixImplementation & m,
							  const char symSide) const throw(InvalidDimensionException)
      {
        if (nbColumns_ != m.nbRows_) throw InvalidDimensionException(HERE) << "Invalid dimensions in matrix/matrix product left="
									   << nbRows_ << "x" << nbColumns_
									   << " right=" << m.getNbRows() << "x"
									   << m.getNbColumns();
	
	MatrixImplementation mult(nbRows_, m.nbColumns_);
	char side(symSide);
	char uplo('U');
	int m_(nbRows_);
	int k_(nbColumns_);
	int n_(m.nbColumns_);
	double alpha(1.0);
	double beta(0.0);
	int lside(1);
	int luplo(1);
	DSYMM_F77(&side, &uplo, &m_, &n_, &alpha, const_cast<double*>(&((*this)[0])), &m_, const_cast<double*>(&(m[0])), &k_, &beta, &mult[0], &m_, &lside, &luplo);
	
        return mult;
      }
      
      /* Multiplications with a NumericalPoint (must have consistent dimensions) */
      NumericalPoint MatrixImplementation::genVectProd (const NumericalPoint & pt) const throw(InvalidDimensionException)
      {
        if (nbColumns_ != pt.getDimension() ) throw InvalidDimensionException(HERE) << "Invalid dimension in matrix/vector product";
	
	NumericalPoint prod(nbRows_);
	char trans('N');
	int m_(nbRows_);
	int n_(nbColumns_);
	int one(1);
	double alpha(1.0);
	double beta(0.0);
	int ltrans(1);

	DGEMV_F77(&trans, &m_, &n_, &alpha, const_cast<double*>(&((*this)[0])), &m_, const_cast<double*>(&(pt[0])), &one, &beta, &prod[0], &one, &ltrans);

	return prod;
      }
      
      NumericalPoint MatrixImplementation::symVectProd (const NumericalPoint & pt) const throw(InvalidDimensionException)
      {
        if (nbColumns_ != pt.getDimension() ) throw InvalidDimensionException(HERE) << "Invalid dimension in matrix/vector product";
	
	NumericalPoint prod(nbRows_);
	char uplo('U');
	int n(nbRows_);
	int one(1);
	double alpha(1.0);
	double beta(0.0);
	int luplo(1);
	DSYMV_F77(&uplo, &n, &alpha, const_cast<double*>(&((*this)[0])), &n, const_cast<double*>(&(pt[0])), &one, &beta, &prod[0], &one, &luplo);

	return prod;
      }
      
      /* Multiplication with a NumericalScalar */
      MatrixImplementation MatrixImplementation::operator* (const NumericalScalar s) const
      {
        MatrixImplementation scalprod(*this);
	double alpha(s);
	int one(1);
	int n_(nbRows_ * nbColumns_);
	DSCAL_F77(&n_, &alpha, &scalprod[0], &one);

	return scalprod;
      }
      
      /* Division by a NumericalScalar*/
      MatrixImplementation MatrixImplementation::operator/ (const NumericalScalar s) const throw(InvalidArgumentException)
      {
        if (s == 0.) throw InvalidArgumentException(HERE);
	
        return this->operator*(1/s);
      }
      
      /* Integer power, general matrix */
      MatrixImplementation MatrixImplementation::genPower(const UnsignedLong n) const
      {
	Bool first(true);
	UnsignedLong exponent(n);
	MatrixImplementation y;
	MatrixImplementation z(*this);
	while (exponent > 0)
	  {
	    // t is the right bit of exponent
	    const UnsignedLong t(exponent % 2);
	    // remove last bit from exponent
	    exponent /= 2;
	    // if right bit is 1
	    if (t != 0)
	      {
		// if it is the rightmost bit equals to 1
		if (first)
		  {
		    first = false;
		    y = z;
		  }
		else y = y.genProd(z);
		// if no more bit to consider
		if (exponent == 0) return y;
	      }
	    // Square the contribution
	    z = z.genProd(z);
	  }
	return y;
      }

      /* Integer power, symmetric matrix */
      MatrixImplementation MatrixImplementation::symPower(const UnsignedLong n) const
      {
	Bool first(true);
	UnsignedLong exponent(n);
	MatrixImplementation y;
	MatrixImplementation z(*this);
	while (exponent > 0)
	  {
	    // t is the right bit of exponent
	    const UnsignedLong t(exponent % 2);
	    // remove last bit from exponent
	    exponent /= 2;
	    // if right bit is 1
	    if (t != 0)
	      {
		// if it is the rightmost bit equals to 1
		if (first)
		  {
		    first = false;
		    y = z;
		  }
		else y = y.symProd(z, 'L');
		// if no more bit to consider
		if (exponent == 0) return y;
	      }
	    // Square the contribution
	    z = z.symProd(z, 'L');
	  }
	return y;
      }

      /* Empty returns true if there is no element in the MatrixImplementation */
      const Bool MatrixImplementation::isEmpty() const
      {
	return ((nbRows_ == 0)  || (nbColumns_ == 0) || (PersistentCollection<NumericalScalar>::isEmpty()));
      }
      

      /* Comparison operator */
      Bool MatrixImplementation::operator == (const MatrixImplementation & rhs) const 
      {
        const MatrixImplementation &lhs(*this);
	Bool equality = true;

	if (&lhs != &rhs) { // Not the same object
	  const PersistentCollection<NumericalScalar> & refLhs = static_cast<const PersistentCollection<NumericalScalar> >(lhs);
	  const PersistentCollection<NumericalScalar> & refRhs = static_cast<const PersistentCollection<NumericalScalar> >(rhs);
	  
	  equality = ( lhs.nbRows_==rhs.nbRows_ && lhs.nbColumns_==rhs.nbColumns_ && refLhs==refRhs);
	}

	return equality;
      }
      
      /* Symmetrize MatrixImplementation in case it is a symmetric matrix (stored as a triangular matrix) */
      void MatrixImplementation::symmetrize() const
      {
	MatrixImplementation & refThis = * const_cast<MatrixImplementation *>(this);
        UnsignedLong i,j;
        for (i = 0; i < nbRows_; ++i)
	  for (j = 0; j < i; ++j)
	    refThis[this->convertPosition(i, j)] = this->operator[](this->convertPosition(j, i));
      }
      
      /* Resolution of a linear system : rectangular matrix
       * MX = b, M is an mxn matrix, b is an mxq matrix and
       * X is an nxq matrix */
      MatrixImplementation MatrixImplementation::solveLinearSystemRect (const MatrixImplementation & b) const throw (InvalidDimensionException)
      {
        if (nbRows_ != b.getNbRows()) throw InvalidDimensionException(HERE);
	
	MatrixImplementation A(*this);
	int m(A.getNbRows());
	int n(A.getNbColumns());
	// B is an extended copy of b, it must be large enought to store the solution, see LAPACK documentation
	int p = std::max(m, n);
	int q = b.getNbColumns();
	MatrixImplementation B(p, q);
	for(UnsignedLong j = 0; j < static_cast<UnsignedLong>(q); ++j)
	  {
	    for (UnsignedLong i = 0; i < static_cast<UnsignedLong>(m); ++i)
	      {
		B(i, j) = b(i, j);
	      }
	  }
	int nrhs = q;
	int lwork=-1;
	double lwork_d;
	int info;
	std::vector<int> jpiv(n);
	double rcond(DefaultSmallPivot);
	int rank;
	DGELSY_F77(&m, &n, &nrhs, &A[0], &m, &B[0], &p, &jpiv[0], &rcond, &rank, &lwork_d, &lwork, &info);
	lwork = static_cast<int>(lwork_d);
	std::vector<double> work(lwork);
	DGELSY_F77(&m, &n, &nrhs, &A[0], &m, &B[0], &p, &jpiv[0], &rcond, &rank, &work[0], &lwork, &info);
	MatrixImplementation result(n, q);
	for(UnsignedLong j = 0; j < static_cast<UnsignedLong>(q); ++j)
	  {
	    for (UnsignedLong i = 0; i < static_cast<UnsignedLong>(n); ++i)
	      {
		result(i, j) = B(i, j);
	      }
	  }
        return result;
      }
  
      /* Resolution of a linear system : rectangular matrix
       * Mx = b, M is an mxn matrix, b is an m-dimensional
       * vector and x is an n-dimensional vector */
      NumericalPoint MatrixImplementation::solveLinearSystemRect (const NumericalPoint & b) const throw (InvalidDimensionException)
      {
	const UnsignedLong m(b.getDimension());
	if (nbRows_ != m) throw InvalidDimensionException(HERE);
	// Convert the vector b into an mx1 matrix B
	MatrixImplementation B(m, 1);
	for (UnsignedLong i = 0; i < m; ++i) B(i, 0) = b[i];
	// Solve the matrix linear system
	MatrixImplementation solution(solveLinearSystemRect(B));
	// Convert the matrix result into an n-dimensional vector
	const UnsignedLong resultDimension(getNbColumns());
	NumericalPoint result(resultDimension);
	for (UnsignedLong i = 0; i < resultDimension; ++i) result[i] = solution(i, 0);
        return result;
      }
  
      /* Resolution of a linear system : square matrix */
      MatrixImplementation MatrixImplementation::solveLinearSystemSquare (const MatrixImplementation & b) const throw (InvalidDimensionException)
      {
        if (nbColumns_ != b.getNbRows() ) throw InvalidDimensionException(HERE);
	
	MatrixImplementation A(*this);
	MatrixImplementation B(b);
	int m = A.getDimension();
	int nrhs = B.getNbColumns();
	int info;
	std::vector<int> ipiv(m);
	DGESV_F77(&m, &nrhs, &A[0], &m, &ipiv[0], &B[0], &m, &info);
        return B;
      }

      /* Resolution of a linear system : square matrix */
      NumericalPoint MatrixImplementation::solveLinearSystemSquare (const NumericalPoint & b) const throw (InvalidDimensionException)
      {
	const UnsignedLong dimension(b.getDimension());
	MatrixImplementation B(dimension, 1);
	for (UnsignedLong i = 0; i < dimension; ++i) B(i, 0) = b[i];
	MatrixImplementation solution(solveLinearSystemSquare(B));
	NumericalPoint result(dimension);
	for (UnsignedLong i = 0; i < dimension; ++i) result[i] = solution(i, 0);
        return result;
      }

      /* Resolution of a linear system : symmetric matrix */
      MatrixImplementation MatrixImplementation::solveLinearSystemSym (const MatrixImplementation & b) const throw (InvalidDimensionException)
      {
        if (nbColumns_ != b.getNbRows() ) throw InvalidDimensionException(HERE);
	
	MatrixImplementation A(*this);
	char uplo='U';
	MatrixImplementation B(b);
	int n = A.getDimension();
	int nrhs = B.getNbColumns();
	int lwork=-1;
	double lwork_d;
	int info;
	std::vector<int> ipiv(n);
	int luplo(1);
	DSYSV_F77(&uplo, &n, &nrhs, &A[0], &n, &ipiv[0], &B[0], &n, &lwork_d, &lwork, &info, &luplo);
	lwork = static_cast<int>(lwork_d);
	std::vector<double> work(lwork);
	DSYSV_F77(&uplo, &n, &nrhs, &A[0], &n, &ipiv[0], &B[0], &n, &work[0], &lwork, &info, &luplo);
        return B;
      }
      
      /* Resolution of a linear system : symmetric matrix */
      NumericalPoint MatrixImplementation::solveLinearSystemSym (const NumericalPoint & b) const throw (InvalidDimensionException)
      {
	const UnsignedLong dimension(b.getDimension());
	MatrixImplementation B(dimension, 1);
	for (UnsignedLong i = 0; i < dimension; ++i) B(i, 0) = b[i];
	MatrixImplementation solution(solveLinearSystemSym(B));
	NumericalPoint result(dimension);
	for (UnsignedLong i = 0; i < dimension; ++i) result[i] = solution(i, 0);
        return result;
      }
      
      /* Compute determinant */
      NumericalScalar MatrixImplementation::computeDeterminant () const
      {
        MatrixImplementation A(*this);
	int n = A.getDimension();
	std::vector<int> ipiv (n);
	int info;
        NumericalScalar determinant = 1.0;
	
	// LU Factorization with LAPACK
	DGETRF_F77(&n, &n, &A[0], &n, &ipiv[0], &info);
	
	// Determinant computation
        for (UnsignedLong i = 0; i < ipiv.size(); ++i) {
	  determinant *= A[i * (ipiv.size() + 1)];
	  if (ipiv[i] != int(i + 1)) determinant = -determinant;
	}
	
	return determinant;
      }
      
      /* Compute determinant for a symmetric matrix */
      NumericalScalar MatrixImplementation::computeDeterminantSym () const
      {
        char uplo='U';
        MatrixImplementation A(*this);
	int n = A.getDimension();
	std::vector<int> ipiv (n);
	int info;
        NumericalScalar determinant=1.0;
	int lwork=-1;
	double lwork_d;
	int luplo(1);
        
	// LU Factorization with LAPACK
	DSYTRF_F77(&uplo, &n, &A[0], &n, &ipiv[0],&lwork_d, &lwork, &info, &luplo);
	lwork = static_cast<int>(lwork_d);
	std::vector<double> work(lwork);
	DSYTRF_F77(&uplo, &n, &A[0], &n, &ipiv[0],&work[0], &lwork, &info, &luplo);

	// Determinant computation
        for (UnsignedLong i = 0; i < ipiv.size(); ++i) {
	  determinant *= A[i * (ipiv.size() + 1)];
	  if (ipiv[i] != int(i + 1)) determinant = -determinant;
	}
	
	return determinant;
      }
      
      /* Compute eigenvalues of a square matrix */
      MatrixImplementation::NumericalComplexCollection MatrixImplementation::computeEigenValuesSquare () const
      {
        char jobvl = 'N';
	char jobvr = 'N';
        MatrixImplementation A(*this);
	int n = A.getDimension();
	NumericalPoint wr (n, 0.);
	NumericalPoint wi (n, 0.);
	double vl;
	double vr;
	int ldvl=1;
	int ldvr=1;	
	int lwork=-1;
	double lwork_d;
	int info;
        int ljobvl(1);
	int ljobvr(1);
	
	DGEEV_F77(&jobvl, &jobvr, &n, &A[0], &n, &wr[0], &wi[0], &vl, &ldvl, &vr, &ldvr, &lwork_d, &lwork, &info, &ljobvl, &ljobvr);
	lwork = static_cast<int>(lwork_d);
	std::vector<double> work(lwork);
	DGEEV_F77(&jobvl, &jobvr, &n, &A[0], &n, &wr[0], &wi[0], &vl, &ldvl, &vr, &ldvr, &work[0], &lwork, &info, &ljobvl, &ljobvr);
	NumericalComplexCollection eigenValues(n);
	for (UnsignedLong i = 0; i < static_cast<UnsignedLong>(n); ++i)
	  {
	    eigenValues[i] = NumericalComplex(wr[i], wi[i]);
	  }
	return eigenValues;
      }
      
      /* Compute eigenvalues of a symmetric matrix */
      NumericalPoint MatrixImplementation::computeEigenValuesSym () const
      {
        char jobz = 'N';
	char uplo = 'U';
        MatrixImplementation A(*this);
	int n = A.getDimension();
	NumericalPoint w (n, 0.);
	int lwork=-1;
	double lwork_d;
	int info;
        int ljobz(1);
        int luplo(1);

	DSYEV_F77(&jobz, &uplo, &n, &A[0], &n, &w[0], &lwork_d, &lwork, &info, &ljobz, &luplo);
	lwork = static_cast<int>(lwork_d);
	std::vector<double> work(lwork);
	DSYEV_F77(&jobz,&uplo,&n,&A[0],&n,&w[0],&work[0],&lwork,&info, &ljobz, &luplo);
	return w;
      }
      
      /* Check if the matrix is SPD */
      Bool MatrixImplementation::isPositiveDefinite() const
      {
	MatrixImplementation A(*this);
        int info;
        int n=A.getDimension();
        char uplo='U';
	int luplo(1);

        DPOTRF_F77(&uplo, &n, &A[0], &n, &info, &luplo);
	
	return (info == 0) ;
      }
	
      /* Build the Cholesky factorization of the matrix */
      MatrixImplementation MatrixImplementation::computeCholesky() const
      {
        MatrixImplementation A(*this);
        int info;
        int n=A.getDimension();
        char uplo='L';
	int luplo(1);

        DPOTRF_F77(&uplo, &n, &A[0], &n, &info, &luplo);
	for (UnsignedLong i = 0; i < (UnsignedLong)(n); ++i)
	  for (UnsignedLong j = i + 1; j < (UnsignedLong)(n); ++j)
	    A(i, j) = 0.0;
	return A;
      }

      /* Method save() stores the object through the StorageManager */
      void MatrixImplementation::save(const StorageManager::Advocate & adv) const
      {
	PersistentCollection<NumericalScalar>::save(adv);
	adv.writeValue("nbRows_",    nbRows_);
	adv.writeValue("nbColumns_", nbColumns_);
      }

      /* Method load() reloads the object from the StorageManager */
      void MatrixImplementation::load(const StorageManager::Advocate & adv)
      {
	PersistentCollection<NumericalScalar>::load(adv);

	String name;
	UnsignedLong value;
	StorageManager::List objList = adv.getList(StorageManager::UnsignedLongEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "nbRows_")    nbRows_ = value;
	    if (name == "nbColumns_") nbColumns_ = value;
	  }
	}
      }


    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
