//                                               -*- C++ -*-
/**
 *  @file  WeightedExperiment.cxx
 *  @brief Abstract top-level view of an weightedExperiment plane
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu, 26 jun 2008) $
 *  Id:      $Id: WeightedExperiment.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include <cstdlib>

#include "OTprivate.hxx"
#include "WeightedExperiment.hxx"
#include "Exception.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      CLASSNAMEINIT(WeightedExperiment);

      typedef Base::Common::NotYetImplementedException NotYetImplementedException;
      typedef Base::Common::InvalidArgumentException   InvalidArgumentException;
      typedef Base::Common::ResourceMap                ResourceMap;

      const UnsignedLong WeightedExperiment::DefaultSize = 100;//strtoul(ResourceMap::GetInstance().get("WeightedExperiment-DefaultSize").c_str(), NULL, 0);

      /* Default constructor */
      WeightedExperiment::WeightedExperiment(const String & name):
	ExperimentImplementation(name),
	distribution_(),
	size_(DefaultSize),
	weight_(DefaultSize, 1.0 / DefaultSize)
      {
	// Nothing to do
      }

      /* Constructor with parameters */
      WeightedExperiment::WeightedExperiment(const UnsignedLong size,
					     const String & name):
	ExperimentImplementation(name),
	distribution_(),
	size_(0),
	weight_(0)
      {
	// Check if the size is valid
	setSize(size);
	weight_ = NumericalPoint(size_, 1.0 / size_);
      }

      /* Constructor with parameters */
      WeightedExperiment::WeightedExperiment(const Distribution & distribution,
					     const UnsignedLong size,
					     const String & name):
	ExperimentImplementation(name),
	distribution_(distribution),
	size_(0),
	weight_(0)
      {
	// Check if the size is valid
	setSize(size);
	weight_ = NumericalPoint(size_, 1.0 / size_);
      }

      /* Virtual constructor */
      WeightedExperiment * WeightedExperiment::clone() const
      {
	return new WeightedExperiment(*this);
      }

      /* String converter */
      String WeightedExperiment::__repr__() const
      {
	OSS oss;
	oss << "class=" << GetClassName()
	    << " name=" << getName ()
	    << " distribution=" << distribution_
	    << " size=" << size_;
	return oss;
      }

      /* Distribution accessor */
      void WeightedExperiment::setDistribution(const Distribution & distribution)
      {
	distribution_ = distribution;
      }

      WeightedExperiment::Distribution WeightedExperiment::getDistribution() const
      {
	return distribution_;
      }

      /* Size accessor */
      void WeightedExperiment::setSize(const UnsignedLong size)
      {
	if (size == 0) throw InvalidArgumentException(HERE) << "Error: the size must be > 0.";
	size_ = size;
      }

      UnsignedLong WeightedExperiment::getSize() const
      {
	return size_;
      }

      /* Sample generation */
      WeightedExperiment::NumericalSample WeightedExperiment::generate()
      {
	throw NotYetImplementedException(HERE);
      }

      /* Weight accessor */
      WeightedExperiment::NumericalPoint WeightedExperiment::getWeight() const
      {
	return weight_;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
