//                                               -*- C++ -*-
/**
 *  @file  MetaModelAlgorithm.cxx
 *  @brief Interface for regression methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2008-06-05 12:11:56 +0200 (Thu, 05 Jun 2008) $
 *  Id:      $Id: MetaModelAlgorithm.cxx 846 2008-06-05 10:11:56Z schueller $
 */

#include "MetaModelAlgorithm.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      CLASSNAMEINIT(MetaModelAlgorithm);

      /* Constructor with parameters */
      MetaModelAlgorithm::MetaModelAlgorithm(const NumericalSample & dataIn,
					     const NumericalMathFunction & inputFunction,
					     const String & name)
	throw(InvalidArgumentException):
 	PersistentObject(name),
 	dataIn_(dataIn),
	dataOut_(NumericalSample(0, inputFunction.getOutputNumericalPointDimension())),
 	inputFunction_(inputFunction)
      {
	if (!inputFunction.getEvaluationImplementation()->isActualImplementation()) throw InvalidArgumentException(HERE) << "Error: the given function must have an actual implementation";
	if (inputFunction.getInputNumericalPointDimension() != dataIn.getDimension()) throw InvalidArgumentException(HERE) << "Error: the input data dimension and the input dimension of the function must be the same, here input dimension=" << dataIn.getDimension() << " and input dimension of the function=" << inputFunction.getInputNumericalPointDimension();
      }

      /* Constructor with parameters */
      MetaModelAlgorithm::MetaModelAlgorithm(const NumericalSample & dataIn,
					     const NumericalSample & dataOut,
					     const String & name)
	throw(InvalidArgumentException):
	PersistentObject(name),
	dataIn_(dataIn),
	dataOut_(NumericalSample(0, dataOut.getDimension())),
	inputFunction_(NumericalMathFunction())
      {
	setDataOut(dataOut);
      }

      /* String converter */
      String MetaModelAlgorithm::__repr__() const
      {
	OSS oss;
	oss << "class=" << GetClassName()
	    << " name=" << getName()
	    << " dataIn=" << dataIn_
	    << " dataOut=" << dataOut_
	    << " function=" << inputFunction_
	    << " responseSurface=" << responseSurface_;
	return oss;
      }

      /* Response surface computation */
      void MetaModelAlgorithm::run()
      {
	throw NotYetImplementedException(HERE);
      }


      /* DataIn accessor */
      MetaModelAlgorithm::NumericalSample MetaModelAlgorithm::getDataIn() const
      {
	return dataIn_;
      }

      /* DataOut accessor */
      MetaModelAlgorithm::NumericalSample MetaModelAlgorithm::getDataOut()
      {
	// If the response surface has been defined with an input function and the output data have not already been computed, compute them
	if (inputFunction_.getEvaluationImplementation()->isActualImplementation() && (dataOut_.getSize() == 0)) dataOut_ = inputFunction_(dataIn_);
        return dataOut_;
      }

      void MetaModelAlgorithm::setDataOut(const NumericalSample & dataOut)
	throw(InvalidArgumentException)
      {
	if (inputFunction_.getEvaluationImplementation()->isActualImplementation()) throw InvalidArgumentException(HERE) << "Error: cannot set the output data in a response surface defined with a function, here function=" << inputFunction_;
	if (dataOut.getSize() != dataIn_.getSize()) throw InvalidArgumentException(HERE) << "Error: the output data must have the same size than the input data, here output size=" << dataOut.getSize() << " and input size=" << dataIn_.getSize();
	dataOut_ = dataOut;
      }


      /* Function accessor */
      MetaModelAlgorithm::NumericalMathFunction MetaModelAlgorithm::getInputFunction() const
      {
	return inputFunction_;
      }

      /* Response surface accessor */
      MetaModelAlgorithm::NumericalMathFunction MetaModelAlgorithm::getResponseSurface() const
      {
	return responseSurface_;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
