//                                               -*- C++ -*-
/**
 *  @file  OrthogonalUniVariatePolynomialFactory.cxx
 *  @brief This an abstract class for 1D polynomial factories
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "OrthogonalUniVariatePolynomialFactory.hxx"
#include "PersistentObjectFactory.hxx"
#include "OSS.hxx"
#include "SquareMatrix.hxx"
#include "Normal.hxx"
#include "Exception.hxx"
#include "Lapack.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(OrthogonalUniVariatePolynomialFactory);
	
      static Base::Common::Factory<OrthogonalUniVariatePolynomialFactory> RegisteredFactory("OrthogonalUniVariatePolynomialFactory");

      typedef Base::Common::NotYetImplementedException              NotYetImplementedException;
      typedef Base::Common::InvalidArgumentException                InvalidArgumentException;
      typedef Base::Type::SquareMatrix                              SquareMatrix;
      typedef Distribution::Normal                                  Normal;
      typedef OrthogonalUniVariatePolynomialFactory::Coefficients   Coefficients; 
      typedef OrthogonalUniVariatePolynomialFactory::Distribution   Distribution; 

      /* Default constructor */
      OrthogonalUniVariatePolynomialFactory::OrthogonalUniVariatePolynomialFactory()
	: Base::Common::PersistentObject(),
	  measure_(Normal()),
	  cache_()
      {
	// Nothing to do
      }


      /* Constructor */
      OrthogonalUniVariatePolynomialFactory::OrthogonalUniVariatePolynomialFactory(const Distribution & measure)
	: Base::Common::PersistentObject(),
          measure_(measure),
	  cache_()
      {
	// Nothing to do
      }


      /* Virtual constructor */
      OrthogonalUniVariatePolynomialFactory * OrthogonalUniVariatePolynomialFactory::clone() const
      {
	return new OrthogonalUniVariatePolynomialFactory(*this);
      }


      /* String converter */
      String OrthogonalUniVariatePolynomialFactory::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " measure=" << measure_;
      }


      /* The method to get the polynomial of any degree */
      OrthogonalUniVariatePolynomialFactory::UniVariatePolynomial OrthogonalUniVariatePolynomialFactory::build(const UnsignedLong degree) const
      {
	if ( cache_.find( degree ) != cache_.end() ) return UniVariatePolynomial( cache_[ degree ] );
	else {
	  // There should be no problem to call build with degree-1 and degree-2 because P0 and P1 are already in cache
	  // and we should never go here with degree==0 or degree==1
	  UniVariatePolynomial Pn_minus_1 = build( degree - 1 );
	  UniVariatePolynomial Pn_minus_2 = build( degree - 2 );
	  Coefficients Pn_coefficients = buildNewCoefficients( Pn_minus_1.getCoefficients(), Pn_minus_2.getCoefficients() );
	  cache_[degree] = Pn_coefficients;
	  return UniVariatePolynomial(Pn_coefficients);
	}
      }

      
      /* Measure accessor */
      OrthogonalUniVariatePolynomialFactory::Distribution OrthogonalUniVariatePolynomialFactory::getMeasure() const
      {
	return measure_;
      }


      /* Method save() stores the object through the StorageManager */
      void OrthogonalUniVariatePolynomialFactory::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
	// TODO : save cache
      }


      /* Method load() reloads the object from the StorageManager */
      void OrthogonalUniVariatePolynomialFactory::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
	// TODO : load cache
      }


      /* Determine the coefficients of the first orthogonal Polynomial (degree = 0) of the family */
      Coefficients OrthogonalUniVariatePolynomialFactory::getP0Coefficients() const
      {
	throw NotYetImplementedException(HERE);
      }
      
      /* Determine the coefficients of the second orthogonal Polynomial (degree = 1) of the family */
      Coefficients OrthogonalUniVariatePolynomialFactory::getP1Coefficients() const
      {
	throw NotYetImplementedException(HERE);
      }
      
      /* Calculate the coefficients of recurrence a1 a2 a3 a4 */
      Coefficients OrthogonalUniVariatePolynomialFactory::getRecurrenceCoefficients(const UnsignedLong n) const
      {
	throw NotYetImplementedException(HERE);
      }

      /* Polynomial recurrence build the new coefficients of the polynomial Pn according to the relation
         Pn(x) = (a0 * x + a1) * Pn-1(x) + a2 * Pn-2(x) */
      Coefficients OrthogonalUniVariatePolynomialFactory::buildNewCoefficients(const Coefficients & Pn_minus_1_Coefficients,
									       const Coefficients & Pn_minus_2_Coefficients) const
      {
	Coefficients recurrenceCoefficients(getRecurrenceCoefficients(Pn_minus_1_Coefficients.getSize()));
	UniVariatePolynomial Pn_minus_1(Pn_minus_1_Coefficients);
	UniVariatePolynomial Pn_minus_2(Pn_minus_2_Coefficients);
	UniVariatePolynomial Pn(Pn_minus_1.incrementDegree(1) * recurrenceCoefficients[0] + Pn_minus_1 * recurrenceCoefficients[1] + Pn_minus_2 * recurrenceCoefficients[2]);
	return Pn.getCoefficients();
      }

      /* Cache initialization */
      void OrthogonalUniVariatePolynomialFactory::initializeCache()
      {
	cache_[0] = getP0Coefficients();
	cache_[1] = getP1Coefficients();
      }

      /* Root of the polynomial of degree n as the eigenvalues of the associated Toeplitz matrix
	 If the recurrence relation is written in the form:
	 P_n(x) = (A_{n-1} x + B_{n-1}) P_{n-1}(x) - C_{n-1} P_{n-2}(x)
	 the polynomial P_n(x) can be written as:
	 P_n(x) = det(K x + M), where K = diag(A_0,...,A_{n-1}) and M is a tridiagonal matrix. Finding the roots of P_n(x)
         is thus equivalent to finding the eigenvalues of -K^{-1}M
       */
      OrthogonalUniVariatePolynomialFactory::NumericalPoint OrthogonalUniVariatePolynomialFactory::getRoots(const UnsignedLong n) const
      {
	if (n == 0) throw InvalidArgumentException(HERE) << "Error: cannot compute the roots of a constant polynomial.";
        // gauss integration rule
	char jobz('V');
	int ljobz(1);
	NumericalPoint d(n);
	NumericalPoint e(n - 1);
	Coefficients coefficients(getP1Coefficients());
	NumericalScalar alphaPrec(coefficients[1]);
	d[0] = -coefficients[0] / alphaPrec;
	for (UnsignedLong i = 1; i < n; ++i)
	  {
	    coefficients = getRecurrenceCoefficients(i + 1);
	    d[i]     = -coefficients[1] / coefficients[0];
	    e[i - 1] = sqrt(-coefficients[2] / (coefficients[0] * alphaPrec));
	    alphaPrec = coefficients[0];
	  }
	int ldz(n);
	SquareMatrix z(n);
	NumericalPoint work(2 * n - 2);
	int info;
	DSTEV_F77(&jobz, &ldz, &d[0], &e[0], &z(0, 0), &ldz, &work[0], &info, &ljobz);
	return d;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
