//                                               -*- C++ -*-
/**
 *  @file  Histogram.cxx
 *  @brief The Histogram distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Histogram.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "Histogram.hxx"
#include "RandomGenerator.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Histogram);

      /* Default constructor */
      Histogram::Histogram()
        : NonEllipticalDistribution("Histogram"),
          firstX_(0.0),
	  collection_(PairCollection(0)),
	  cumulatedWidth_(0),
	  surface_(0)
      {
        setPairCollection(PairCollection(1, Pair(1.0, 1.0)));
        setDimension( 1 );
        computeRange();
      }

      /* Parameters constructor */
      Histogram::Histogram(const NumericalScalar first,
                           const PairCollection & collection)
        : NonEllipticalDistribution("Histogram"),
          firstX_(first),
	  collection_(PairCollection(0)),
	  cumulatedWidth_(0),
	  surface_(0)
      {
        setPairCollection(collection);
        setDimension( 1 );
        computeRange();
      }

      /* Comparison operator */
      Bool Histogram::operator ==(const Histogram & other) const {
        Bool sameObject = false;

        if (this != &other) { // Other is NOT me, so I have to realize the comparison
          // sameObject = ...
          // TODO: Write Histogram::operator ==(...)
          sameObject = true;
        } else sameObject = true;

        return sameObject;
      }

      /* String converter */
      String Histogram::__repr__() const {
        OSS oss;
        oss << "class=" << Histogram::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " first=" << firstX_
            << " pair collection=" << collection_;
        return oss;
      }

      /* Virtual constructor */
      Histogram * Histogram::clone() const
      {
        return new Histogram(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void Histogram::computeRange()
      {
        const UnsignedLong size(cumulatedWidth_.getDimension());
        if (size == 0) return;
        setRange(Interval(firstX_, firstX_ + cumulatedWidth_[size - 1]));
      }

      /* Get one realization of the distribution */
      Histogram::NumericalPoint Histogram::getRealization() const
      {
        return computeQuantile(RandomGenerator::Generate());
      }


      /* Get the DDF of the distribution */
      Histogram::NumericalPoint Histogram::computeDDF(const NumericalPoint & point) const
      {
        return NumericalPoint(1, 0.0);
      }


      /* Get the PDF of the distribution */
      NumericalScalar Histogram::computePDF(const NumericalPoint & point) const
      {
        NumericalScalar x(point[0] - firstX_);
        const UnsignedLong size(collection_.getSize());
        if ((x <= 0.0) || (x >= cumulatedWidth_[size - 1])) return 0.0;
        NumericalScalar lower(0.0);
        for (UnsignedLong i = 0; i < size; i++)
          {
            x -= lower;
            lower = collection_[i].l_;
            if (x < lower) return collection_[i].h_;
          }
        // Should never go there
        return 0.0;
      }


      /* Get the CDF of the distribution */
      NumericalScalar Histogram::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
        NumericalScalar x(point[0] - firstX_);
        const UnsignedLong size(collection_.getSize());
        if (x <= 0.0) return (tail ? 1.0 : 0.0);
        if (x >= cumulatedWidth_[size - 1]) return (tail ? 0.0 : 1.0);
        NumericalScalar lower(0.0);
        NumericalScalar cdf(0.0);
        // FIXME
        for (UnsignedLong i = 0; i < size; i++)
          {
            x -= lower;
            lower = collection_[i].l_;
            cdf += collection_[i].s_;;
            if (x <= lower) return cdf + (x - lower) * collection_[i].h_;
          }
        // Should never go there
        return 1.0;
      }

      /** Get the PDFGradient of the distribution */
      Histogram::NumericalPoint Histogram::computePDFGradient(const NumericalPoint & point) const
      {
        const UnsignedLong size(collection_.getSize());
        NumericalPoint pdfGradient(1 + 2 * size, 0.0);
        NumericalScalar x(point[0] - firstX_);
        if ((x <= 0.0) || (x >= cumulatedWidth_[size - 1])) return pdfGradient;
        NumericalScalar pdf(computePDF(point));
        // Gradient with respect to firstX_ is null
        // Gradient with respect to (li, hi)
        for (UnsignedLong i = 0; i < size; i++)
          {
            NumericalScalar hi(collection_[i].h_);
            pdfGradient[2 * i + 1] = -hi * pdf;
            pdfGradient[2 * i + 2] = (1.0 / hi - collection_[i].l_) * pdf / surface_;
          }
        return pdfGradient;
      }

      /** Get the CDFGradient of the distribution */
      Histogram::NumericalPoint Histogram::computeCDFGradient(const NumericalPoint & point) const
      {
        return NumericalPoint(0);
      }

      /* Get the quantile of the distribution */
      NumericalScalar Histogram::computeScalarQuantile(const NumericalScalar prob,
                                                       const NumericalScalar initialGuess,
                                                       const NumericalScalar initialStep,
                                                       const NumericalScalar precision) const
      {
        const UnsignedLong size(collection_.getSize());
        // Research of the containing bin
        UnsignedLong index(UnsignedLong(prob * size));
        NumericalScalar currentProba(cumulatedSurface_[index]);
        UnsignedLong currentIndex(index);
        // Basic search: upper bound. The loop must end because cumulatedSurface_[size - 1] = 1.0 and prob < 1.0
        while (prob >= currentProba)
          {
            currentIndex++;
            currentProba = cumulatedSurface_[currentIndex];
          }
        // At the end of the loop, we are sure that currentProba > prob
        // If index < currentIndex, it means that prob is associated with bin number currentIndex. Do a linear interpolation.
        if (index < currentIndex)
          {
            // currentIndex is now the number of the bin associated with prob
            return firstX_ + cumulatedWidth_[currentIndex] + collection_[currentIndex].l_ * (prob - currentProba) / collection_[currentIndex].s_;
          }
        // Here we know that we have to go downstairs. We must check that currentIndex remains >= 0 in the loop.
        while ((prob < currentProba) && (currentIndex > 0))
          {
            currentIndex--;
            currentProba = cumulatedSurface_[currentIndex];
          }
        // At the end of the loop, either prob < cumulatedSurface_[0], which means that prob is associated with the first bin...
        if (currentIndex == 0)
          {
            return firstX_ + collection_[0].l_ * prob / currentProba;
          }
        // ... or prob >= cumulatedSurface_[currentIndex], which means that prob is associated with the bin number currentIndex + 1. Do a linear interpolation.
        return firstX_ + cumulatedWidth_[currentIndex] + collection_[currentIndex + 1].l_ * (prob - currentProba) / collection_[currentIndex + 1].s_;
      }

      /* Get the mean of the distribution */
      Histogram::NumericalPoint Histogram::getMean() const throw(NotDefinedException)
      {
        NumericalScalar mean(firstX_);
        const UnsignedLong size(collection_.getSize());
        NumericalScalar lower(0.0);
        for (UnsignedLong i = 0; i < size; i++)
          {
            NumericalScalar length(collection_[i].l_);
            NumericalScalar upper(lower + length);
            mean += 0.5 * collection_[i].s_ * (lower + upper);
            lower = upper;
          }
        return NumericalPoint(1, mean);
      }

      /* Get the covariance of the distribution */
      Histogram::CovarianceMatrix Histogram::getCovariance() const throw(NotDefinedException)
      {
        NumericalScalar value(0.0);
        const UnsignedLong size(collection_.getSize());
        CovarianceMatrix covariance(1);
        // Since variance is invariant by translation, we center the data for numerical stability
        NumericalScalar lower(firstX_ - getMean()[0]);
        for (UnsignedLong i = 0; i < size; i++)
          {
            NumericalScalar length(collection_[i].l_);
            NumericalScalar upper(lower + length);
            value += collection_[i].s_ * (lower * lower + lower * upper + upper * upper);
            lower = upper;
          }
        covariance(0, 0) = value / 3.0;
        return covariance;
      }

      /* Parameters value and description accessor */
      Histogram::NumericalPointWithDescriptionCollection Histogram::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        const UnsignedLong size(collection_.getSize());
        NumericalPointWithDescription point(1 + 2 * size);
        Description description(1 + 2 * size);
        point[0] = firstX_;
        description[0] = "first";
        for (UnsignedLong i = 0; i < size; i++)
          {
            point[2 * i + 1] = collection_[i].h_;
            point[2 * i + 2] = collection_[i].l_;
            OSS oss;
            oss << "h_" << i;
            description[2 * i + 1] = oss;
            oss.clear();
            oss << "l_" << i;
            description[2 * i + 2] = oss;
          }
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }




      /* Interface specific to Histogram */

      /* First point accessor */
      void Histogram::setFirst(const NumericalScalar first)
      {
        firstX_ = first;
        computeRange();
      }

      NumericalScalar Histogram::getFirst() const
      {
        return firstX_;
      }


      /* Collection accessor */
      void Histogram::setPairCollection(const PairCollection & collection)
      {
        surface_ = 0.0;
        const UnsignedLong size(collection.getSize());
        cumulatedWidth_ = NumericalPoint(size);
        cumulatedSurface_ = NumericalPoint(size);
        // first, check that all the heights and widths are >=0
        for (UnsignedLong i = 0; i < size; i++)
          {
            NumericalScalar height(collection[i].h_);
            if (height < 0.0) throw InvalidArgumentException(HERE) << "Error: all the heights must be >= 0, here values=" << collection;
            NumericalScalar length(collection[i].l_);
            if (length <= 0.0) throw InvalidArgumentException(HERE) << "Error: all the widths must be > 0, here value=" << collection;
            surface_ += height * length;
            cumulatedWidth_[i] = length + (i == 0 ? 0 : cumulatedWidth_[i - 1]);
            cumulatedSurface_[i] = surface_;
          }
        // Normalization
        collection_ = PairCollection(size);
        NumericalScalar normalizationFactor(1.0 / surface_);
        for (UnsignedLong i = 0; i < size; i++)
          {
            collection_[i].h_ = collection[i].h_ * normalizationFactor;
            collection_[i].l_ = collection[i].l_;
            collection_[i].s_ = collection[i].s_ * normalizationFactor;
            cumulatedSurface_[i] *= normalizationFactor;
          }
        computeRange();
      }

      Histogram::PairCollection Histogram::getPairCollection() const
      {
        return collection_;
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
