//                                               -*- C++ -*-
/**
 *  @file  Weibull.hxx
 *  @brief The Weibull distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Weibull.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_WEIBULL_HXX
#define OPENTURNS_WEIBULL_HXX

#include "NonEllipticalDistribution.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Weibull
       *
       * The Weibull distribution.
       */
      class Weibull
        : public Model::NonEllipticalDistribution
      {
        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
        typedef Model::NonEllipticalDistribution                     NonEllipticalDistribution; // required by SWIG
        typedef NonEllipticalDistribution::NumericalPoint            NumericalPoint;
        typedef NonEllipticalDistribution::NumericalSample           NumericalSample;
        typedef NonEllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
        typedef NonEllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
        typedef NonEllipticalDistribution::NotDefinedException       NotDefinedException;
        typedef NonEllipticalDistribution::StorageManager            StorageManager;

        enum ParameterSet { ALPHABETA, MUSIGMA };

        /** Default constructor */
        Weibull();

        /** Parameters constructor */
        Weibull(const NumericalScalar arg1,
                const NumericalScalar arg2,
                const NumericalScalar gamma = 0.0,
                const ParameterSet set = ALPHABETA)
          throw (InvalidArgumentException);


        /** Comparison operator */
        Bool operator ==(const Weibull & other) const;

        /** String converter */
	String __repr__() const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        virtual Weibull * clone() const;

        /** Get one realization of the Weibull distribution */
        NumericalPoint getRealization() const;

        /** Get the DDF of the distribution, i.e. the gradient of its PDF w.r.t. point */
        using NonEllipticalDistribution::computeDDF;
        NumericalPoint computeDDF(const NumericalPoint & point) const;

        /** Get the PDF of the distribution, i.e. P(point < X < point+dx) = PDF(point)dx + o(dx) */
        using NonEllipticalDistribution::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the distribution, i.e. P(X <= point) = CDF(point). If tail=true, compute P(X >= point) */
        using NonEllipticalDistribution::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the gradient of the PDF w.r.t the parameters of the distribution */
        NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the gradient of the CDF w.r.t the parameters of the distribution */
        NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

      protected:
        /** Get the quantile of the distribution, i.e the value Xp such that P(X <= Xp) = prob */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                              const NumericalScalar initialGuess = 0.0,
                                              const NumericalScalar initialStep = 1.0,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

      public:
        /** Get the mean of the distribution */
        NumericalPoint getMean() const throw(NotDefinedException);

        /** Get the standard deviation of the distribution */
        NumericalPoint getStandardDeviation() const throw(NotDefinedException);

        /** Get the skewness of the distribution */
        NumericalPoint getSkewness() const throw(NotDefinedException);

        /** Get the kurtosis of the distribution */
        NumericalPoint getKurtosis() const throw(NotDefinedException);

        /** Get the covariance of the distribution */
        CovarianceMatrix getCovariance() const throw(NotDefinedException);

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;
        using NonEllipticalDistribution::setParametersCollection;
        void setParametersCollection(const NumericalPointCollection & parametersCollection);

        /* Interface specific to Weibull */

        /** Beta accessor */
        void setBeta(const NumericalScalar beta)
          throw(InvalidArgumentException);
        NumericalScalar getBeta() const;

        /** Alpha accessor */
        void setAlpha(const NumericalScalar alpha)
          throw(InvalidArgumentException);
        NumericalScalar getAlpha() const;

        /** Mu accessor */
        void setMuSigma(const NumericalScalar mu,
                        const NumericalScalar sigma)
          throw(InvalidArgumentException);
        NumericalScalar getMu() const;

        /** Sigma accessor */
        NumericalScalar getSigma() const;

        /** Gamma accessor */
        void setGamma(const NumericalScalar gamma);
        NumericalScalar getGamma() const;

        /** Method save() stores the object through the StorageManager */
        void save(const StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(const StorageManager::Advocate & adv);

      protected:


      private:

        /** Compute the numerical range of the distribution given the parameters values */
        void computeRange();

        /** The main parameter set of the distribution */
        NumericalScalar alpha_;
        NumericalScalar beta_;
        NumericalScalar gamma_;



      }; /* class Weibull */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_WEIBULL_HXX */
