//                                               -*- C++ -*-
/**
 *  @file  ComposedNumericalMathHessianImplementation.cxx
 *  @brief The class that implements the composition between numerical math hessians
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: ComposedNumericalMathHessianImplementation.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "ComposedNumericalMathHessianImplementation.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ComposedNumericalMathHessianImplementation);

      static Common::Factory<ComposedNumericalMathHessianImplementation> RegisteredFactory("ComposedNumericalMathHessianImplementation");

      /* Default constructor */
      ComposedNumericalMathHessianImplementation::ComposedNumericalMathHessianImplementation(const GradientImplementation & p_leftGradient,
											     const HessianImplementation & p_leftHessian,
											     const EvaluationImplementation & p_rightFunction,
											     const GradientImplementation & p_rightGradient,
											     const HessianImplementation & p_rightHessian)
	/* throw(InvalidArgumentException) */
	: NumericalMathHessianImplementation(),
	  p_leftGradient_(p_leftGradient),
	  p_leftHessian_(p_leftHessian),
	  p_rightFunction_(p_rightFunction),
	  p_rightGradient_(p_rightGradient),
	  p_rightHessian_(p_rightHessian)
      {
	UnsignedLong p_rightInputDimension(p_rightFunction->getInputDimension());
	UnsignedLong p_rightOutputDimension(p_rightFunction->getOutputDimension());
	UnsignedLong p_leftInputDimension(p_leftHessian->getInputDimension());
	UnsignedLong p_leftOutputDimension(p_leftHessian->getOutputDimension());
	// We check that the dimensions of the input parameters are compatible
	if ((p_leftInputDimension != p_rightOutputDimension) || (p_leftGradient->getInputDimension() != p_leftInputDimension) || (p_leftGradient->getOutputDimension() != p_leftOutputDimension) || (p_rightGradient->getInputDimension() != p_rightInputDimension) || (p_rightGradient->getOutputDimension() != p_rightOutputDimension) || (p_rightHessian->getInputDimension() != p_rightInputDimension) || (p_rightHessian->getOutputDimension() != p_rightOutputDimension)) throw InvalidArgumentException(HERE) << "The left gradient dimensions are not compatible with the right function dimensions, or the right gradient dimensions are not compatible with the right function dimensions";
      }

      /* Virtual constructor */
      ComposedNumericalMathHessianImplementation * ComposedNumericalMathHessianImplementation::clone() const
      {
	return new ComposedNumericalMathHessianImplementation(*this);
      }

      /* Comparison operator */
      Bool ComposedNumericalMathHessianImplementation::operator ==(const ComposedNumericalMathHessianImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String ComposedNumericalMathHessianImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << ComposedNumericalMathHessianImplementation::GetClassName()
	    << " name=" << getName()
	    << " leftGradient=" << p_leftGradient_->__repr__()
	    << " leftHessian=" << p_leftHessian_->__repr__()
	    << " rightFunction=" << p_rightFunction_->__repr__()
	    << " rightGradient=" << p_rightGradient_->__repr__()
	    << " rightHessian=" << p_rightHessian_->__repr__();
	return oss;
      }
  
      /* Method hessian() returns the symmetric tensor of the second gradient of the function at point */
      ComposedNumericalMathHessianImplementation::SymmetricTensor ComposedNumericalMathHessianImplementation::hessian(const NumericalPoint & in) const
	/* throw(InvalidArgumentException,InternalException) */
      {
	callsNumber_++;
	// Hessian of the composed function fog, g:R^n->R^p, f:R^p->R^q so fog:R^n->R^q
	NumericalPoint p_rightIn(p_rightFunction_->operator()(in));
	Matrix p_rightGradientIn(p_rightGradient_->gradient(in));
	SymmetricTensor p_rightHessianIn(p_rightHessian_->hessian(in));
	Matrix p_leftGradientIn(p_leftGradient_->gradient(p_rightIn));
	SymmetricTensor p_leftHessianIn(p_leftHessian_->hessian(p_rightIn));
	// Get the several dimensions
	UnsignedLong inputDimension(getInputDimension());
	UnsignedLong outputDimension(getOutputDimension());
	UnsignedLong innerDimension(p_rightIn.getDimension());
	SymmetricTensor result(inputDimension, outputDimension);
	/* We unroll the formula:
	 *  D2(h)(x) = D2(f)(g(x))D(g)(x)D(g)(x)+D(f)(g(x))D2(g)(x)
	 *  after transposition to convert jacobian and hessian into
	 *  the associated gradients
	 */
	for (UnsignedLong i = 0; i < inputDimension; i++)
	  {
	    for (UnsignedLong j = 0; j <= i; j++)
	      {
		for (UnsignedLong k = 0; k < outputDimension; k++)
		  {
		    result(i, j, k) = 0.0;
		    for (UnsignedLong m = 0; m < innerDimension; m++)
		      {
			result(i, j, k) += p_leftGradientIn(m, k) * p_rightHessianIn(i, j, m);
			for (UnsignedLong n = 0; n < p_rightIn.getDimension(); n++)
			  {
			    result(i, j, k) += p_leftHessianIn(m, n, k) * p_rightGradientIn(i, m) * p_rightGradientIn(j, n);
			  } // n
		      } // m
		  } // k
	      } // j
	  } // i
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong ComposedNumericalMathHessianImplementation::getInputDimension() const
	/* throw(InternalException) */
      {
	return p_rightHessian_->getInputDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong ComposedNumericalMathHessianImplementation::getOutputDimension() const
	/* throw(InternalException) */
      {
	return p_leftHessian_->getOutputDimension();
      }
     
      /* Method save() stores the object through the StorageManager */
      void ComposedNumericalMathHessianImplementation::save(StorageManager::Advocate & adv) const
      {
	NumericalMathHessianImplementation::save(adv);
	adv.saveAttribute( "leftGradient_", *p_leftGradient_ );
	adv.saveAttribute( "leftHessian_", *p_leftHessian_ );
	adv.saveAttribute( "rightFunction_", *p_rightFunction_ );
	adv.saveAttribute( "rightGradient_", *p_rightGradient_ );
	adv.saveAttribute( "rightHessian_", *p_rightHessian_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void ComposedNumericalMathHessianImplementation::load(StorageManager::Advocate & adv)
      {
	Common::TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluationValue;
	Common::TypedInterfaceObject<NumericalMathGradientImplementation> gradientValue;
	Common::TypedInterfaceObject<NumericalMathHessianImplementation> hessianValue;
	NumericalMathHessianImplementation::load(adv);
	adv.loadAttribute( "leftGradient_", gradientValue );
	p_leftGradient_ = gradientValue.getImplementation();
	adv.loadAttribute( "leftHessian_", hessianValue );
	p_leftHessian_ = hessianValue.getImplementation();
	adv.loadAttribute( "rightFunction_", evaluationValue );
	p_rightFunction_ = evaluationValue.getImplementation();
	adv.loadAttribute( "rightGradient_", gradientValue );
	p_rightGradient_ = gradientValue.getImplementation();
	adv.loadAttribute( "rightHessian_", hessianValue );
	p_rightHessian_ = hessianValue.getImplementation();
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
