//                                               -*- C++ -*-
/**
 *  @file  WrapperFile.cxx
 *  @brief This class provides all the treatments for wrapper file manipulation
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: WrapperFile.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <vector>                 // for std::vector
#include <string>                 // for std::string
#include <sstream>                // for std::istringstream
#include <cstdlib>                // for getenv
#include <sys/types.h>            // for stat
#include <sys/stat.h>             // for stat
#include <unistd.h>               // for stat
#include "OSS.hxx"
#include "WrapperFile.hxx"
#include "Path.hxx"
#include "Exception.hxx"
#include "Log.hxx"
#include "WrapperCommon.h"
#include "XMLToolbox.hxx"

#ifdef XML_SUPPORTED
#include "XMLTags.hxx"
#endif

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      using Common::XMLParserException;
      using Common::WrapperFileParsingException;
      using Common::FileNotFoundException;
      using Common::NotYetImplementedException;
      using Common::InternalException;
      using Common::Log;
      using Common::Path;

      /* The file name extension */
#ifdef XML_SUPPORTED
      const String WrapperFile::extension_ = ".xml";
      static const FileName DTDFileName    = "wrapper.dtd";
#else
      const String WrapperFile::extension_ = ".txt";
#endif



      CLASSNAMEINIT(WrapperFile);

      /* Default constructor */
      WrapperFile::WrapperFile()
	: PersistentObject(),
	  descriptionFilePath_(),
	  data_()
      {
	// Nothing to do
      }

      /* Constructor from file */
      WrapperFile::WrapperFile(const FileName & pathToFile)
	/* throw(WrapperFileParsingException) */
	: PersistentObject(),
	  descriptionFilePath_(pathToFile),
	  data_()
      {
	init();
	Log::Info(OSS() << "Working on wrapper description file " << pathToFile);
	parseFile(pathToFile);
	done();
      }

      /* Virtual constructor */
      WrapperFile * WrapperFile::clone() const
      {
	return new WrapperFile(*this);
      }


      /* String converter */
      String WrapperFile::__repr__() const
      {
	OSS oss;
	oss << "class=" << getClassName()
	    << " path=" << getDescriptionFilePath()
            << " data={" << data_
	    << "}";
	return oss;
      }

#ifdef XML_SUPPORTED

      using Common::XMLDoc;
      using Common::XML;

      /* Standard Initialization */
      void WrapperFile::init() const
	/* throw(WrapperFileParsingException) */
      {
	// Nothing to do
      }

      /* Standard finalization */
      void WrapperFile::done() const
      {
	// Nothing to do
      }


      /* Document parsing function */
      static WrapperData parseDocument(const XMLDoc & doc)
	/* throw(WrapperFileParsingException) */
      {
	WrapperData data;

	const FileName DTDPath = Path::GetStandardWrapperDirectory() + "/" + DTDFileName;
	Bool ok = doc.hasDTD() ? doc.validate() : doc.validate( XMLTag_wrapper, DTDPath );
	if (!ok) throw WrapperFileParsingException(HERE) << "The wrapper description does not conform to DTD";
	else Log::Debug( OSS() << "The wrapper description is valid according to DTD (" << DTDPath << ")" );

	// Check it is an OpenTURNS' one
	XML::Node wrapperElt = XML::GetRootNode( doc );
	if (wrapperElt == NULL) throw WrapperFileParsingException(HERE) << "Wrapper description has no root element";
	if (! XML::IsElement( wrapperElt, XMLTag_wrapper ))
	  throw WrapperFileParsingException(HERE) << "Wrapper description has an invalid root element (" << XML::GetNodeName( wrapperElt )
						  << ") at line " << XML::GetNodeLineNumber( wrapperElt );

	XML::Node libraryElt     = XML::FindElementByName( wrapperElt, XMLTag_library );
	XML::Node libraryPathElt = XML::FindElementByName( libraryElt, XMLTag_path );
	String  libraryPath      = XML::GetNodeValue( libraryPathElt );
	data.setLibraryPath( libraryPath );

	Log::Debug(OSS() << "Read library path                     : " << libraryPath );



	XML::Node descriptionElt  = XML::FindElementByName( libraryElt, XMLTag_description );
	XML::Node variableListElt = XML::FindElementByName( descriptionElt, XMLTag_variable_list );
	WrapperData::VariableListType variableList;
	Bool hasOneOrMoreInputVariables  = false;
	Bool hasOneOrMoreOutputVariables = false;
	if (XML::IsElement(variableListElt)) {
	  for (XML::Node current = XML::GetFirstChild(variableListElt); current; current = XML::GetNextNode(current)) {
	    if (XML::IsElement(current, XMLTag_variable)) {
	      WrapperDataVariable variable;

	      XML::Node commentElt = XML::FindElementByName( current, XMLTag_comment );
	      if (commentElt) variable.comment_ = XML::GetNodeValue( commentElt );
	      
	      XML::Node unitElt    = XML::FindElementByName( current, XMLTag_unit );
	      if (unitElt) variable.unit_ = XML::GetNodeValue( unitElt );
	      
	      XML::Node regexpElt  = XML::FindElementByName( current, XMLTag_regexp );
	      if (regexpElt) variable.regexp_ = XML::GetNodeValue( regexpElt );
	      
	      XML::Node formatElt  = XML::FindElementByName( current, XMLTag_format );
	      if (formatElt) variable.format_ = XML::GetNodeValue( formatElt );

	      variable.id_  = XML::GetAttributeByName( current, XMLTag_id );
	      
	      String type_ = XML::GetAttributeByName( current, XMLTag_type );
	      if (type_ == XMLAttr_in) {
		variable.type_ = WrapperDataVariableType::IN;
		hasOneOrMoreInputVariables = true;
	      }
	      else if (type_ == XMLAttr_out) {
		variable.type_ = WrapperDataVariableType::OUT;
		hasOneOrMoreOutputVariables = true;
	      }
	      else {
		/* We should never go here if the wrapper file has been validated according to the DTD */
		throw WrapperFileParsingException(HERE) << "Unknown type (" << type_
							<< ") for variable in wrapper description at line "
							<< XML::GetNodeLineNumber( current );
	      }
	      
	      String computedGradient_ = XML::GetAttributeByName( current, XMLTag_computed_gradient );
	      if (computedGradient_ == XMLAttr_yes) variable.gradient_ = WrapperComputedGradient::YES;
	      else if (computedGradient_ == XMLAttr_no) variable.gradient_ = WrapperComputedGradient::NO;
	      else if (computedGradient_.empty()) variable.gradient_ = WrapperComputedGradient::NO;
	      else {
		/* We should never go here if the wrapper file has been validated according to the DTD */
		throw WrapperFileParsingException(HERE) << "Unknown " << XMLTag_computed_gradient << " attribute (" << computedGradient_
							<< ") for variable in wrapper description at line "
							<< XML::GetNodeLineNumber( current );
	      }
	      
	      Log::Debug(OSS() << "Read data variable id                 : " << variable.id_);
	      Log::Debug(OSS() << "               ... type               : " << WrapperListElementTypeAsString[variable.type_]);
	      Log::Debug(OSS() << "               ... computed-gradient  : " << WrapperProvidedAsString[variable.gradient_]);
	      Log::Debug(OSS() << "               ... comment            : " << variable.comment_);
	      Log::Debug(OSS() << "               ... unit               : " << variable.unit_);
	      Log::Debug(OSS() << "               ... regexp             : " << variable.regexp_);
	      Log::Debug(OSS() << "               ... format             : " << variable.format_);
	      
	      variableList.add( variable );
	    }
	  }
	}
	data.setVariableList( variableList );

	if (! hasOneOrMoreInputVariables)  Log::Warn( "Wrapper description doesn't define any input  variable" );
	if (! hasOneOrMoreOutputVariables) Log::Warn( "Wrapper description doesn't define any output variable" );

	WrapperFunctionDescription functionDesc;
	XML::Node functionElt = XML::FindElementByName( descriptionElt, XMLTag_function );
	functionDesc.name_     = XML::GetNodeValue( functionElt );

	String functionProvided_  = XML::GetAttributeByName( functionElt, XMLTag_provided );
	if (functionProvided_ == XMLAttr_yes) functionDesc.provided_ = WrapperSymbolProvided::YES;
	else if (functionProvided_ == XMLAttr_no) functionDesc.provided_ = WrapperSymbolProvided::NO;
	else if (functionProvided_.empty()) functionDesc.provided_ = WrapperSymbolProvided::NO;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown " << XMLTag_provided << " attribute (" << functionProvided_
						  << ") for variable in wrapper description at line "
						  << XML::GetNodeLineNumber( functionElt );
	}
	data.setFunctionDescription( functionDesc );

	Log::Debug(OSS() << "Read function name                    : " << functionDesc.name_);
	Log::Debug(OSS() << "               ... provided           : " << WrapperProvidedAsString[functionDesc.provided_]);




	WrapperFunctionDescription gradientDesc;
	XML::Node gradientElt = XML::FindElementByName( descriptionElt, XMLTag_gradient );
	gradientDesc.name_    = XML::GetNodeValue( gradientElt );

	String gradientProvided_  = XML::GetAttributeByName( gradientElt, XMLTag_provided );
	if (gradientProvided_ == XMLAttr_yes) gradientDesc.provided_ = WrapperSymbolProvided::YES;
	else if (gradientProvided_ == XMLAttr_no) gradientDesc.provided_ = WrapperSymbolProvided::NO;
	else if (gradientProvided_.empty()) gradientDesc.provided_ = WrapperSymbolProvided::NO;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown " << XMLTag_provided << " attribute (" << gradientProvided_
						  << ") for variable in wrapper description at line "
						  << XML::GetNodeLineNumber( gradientElt );
	}
	data.setGradientDescription( gradientDesc );

	Log::Debug(OSS() << "Read gradient name                    : " << gradientDesc.name_);
	Log::Debug(OSS() << "               ... provided           : " << WrapperProvidedAsString[gradientDesc.provided_]);




	WrapperFunctionDescription hessianDesc;
	XML::Node hessianElt = XML::FindElementByName( descriptionElt, XMLTag_hessian );
	hessianDesc.name_     = XML::GetNodeValue( hessianElt );

	String hessianProvided_  = XML::GetAttributeByName( hessianElt, XMLTag_provided );
	if (hessianProvided_ == XMLAttr_yes) hessianDesc.provided_ = WrapperSymbolProvided::YES;
	else if (hessianProvided_ == XMLAttr_no) hessianDesc.provided_ = WrapperSymbolProvided::NO;
	else if (hessianProvided_.empty()) hessianDesc.provided_ = WrapperSymbolProvided::NO;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown " << XMLTag_provided << " attribute (" << hessianProvided_
						  << ") for variable in wrapper description at line "
						  << XML::GetNodeLineNumber( hessianElt );
	}
	data.setHessianDescription( hessianDesc );

	Log::Debug(OSS() << "Read hessian  name                    : " << hessianDesc.name_);
	Log::Debug(OSS() << "               ... provided           : " << WrapperProvidedAsString[hessianDesc.provided_]);



	XML::Node externalCodeElt = XML::FindElementByName( wrapperElt, XMLTag_external_code );
	XML::Node dataElt         = XML::FindElementByName( externalCodeElt, XMLTag_data );
	WrapperData::FileListType fileList;
	WrapperFrameworkData framework;
	Bool hasOneOrMoreInputFiles  = false;
	Bool hasOneOrMoreOutputFiles = false;
	if (XML::IsElement(dataElt)) {
	  for (XML::Node current = XML::GetFirstChild(dataElt); current; current = XML::GetNextNode(current)) {
	    if (XML::IsElement(current, XMLTag_file)) {
	      WrapperDataFile file;

	      XML::Node nameElt         = XML::FindElementByName( current, XMLTag_name );
	      if (nameElt) file.name_   = XML::GetNodeValue( nameElt );
	      
	      XML::Node pathElt         = XML::FindElementByName( current, XMLTag_path );
	      file.path_                = XML::GetNodeValue( pathElt );
	      
	      XML::Node substElt        = XML::FindElementByName( current, XMLTag_subst );
	      if (substElt) file.subst_ = XML::GetNodeValue( substElt );
	      
	      file.id_                  = XML::GetAttributeByName( current, XMLTag_id );
	      
	      String type_              = XML::GetAttributeByName( current, XMLTag_type );
	      if (type_ == XMLAttr_in) {
		file.type_ = WrapperDataFileType::IN;
		hasOneOrMoreInputFiles = true;
	      }
	      else if (type_ == XMLAttr_out) {
		file.type_ = WrapperDataFileType::OUT;
		hasOneOrMoreOutputFiles = true;
	      }
	      else {
		/* We should never go here if the wrapper file has been validated according to the DTD */
		throw WrapperFileParsingException(HERE) << "Unknown type (" << type_
							<< ") for file in wrapper description at line "
							<< XML::GetNodeLineNumber( current );
	      }
	      
	      Log::Debug(OSS() << "Read data file id                     : " << file.id_);
	      Log::Debug(OSS() << "           ... type                   : " << WrapperListElementTypeAsString[file.type_]);
	      Log::Debug(OSS() << "           ... name                   : " << file.name_);
	      Log::Debug(OSS() << "           ... path                   : " << file.path_);
	      Log::Debug(OSS() << "           ... subst                  : " << file.subst_);

	      fileList.add( file );

	    } else if (XML::IsElement(current, XMLTag_framework)) {
	      XML::Node studyidElt                    = XML::FindElementByName( current, XMLTag_study_id );
	      if (studyidElt) {
		std::istringstream iss( XML::GetNodeValue( studyidElt ) );
		iss >> framework.studyid_;
	      }
	      
	      XML::Node studycaseElt                  = XML::FindElementByName( current, XMLTag_study_case );
	      if (studycaseElt) framework.studycase_  = XML::GetNodeValue( studycaseElt );
	      
	      XML::Node componentnameElt              = XML::FindElementByName( current, XMLTag_component_name );
	      if (componentnameElt) framework.componentname_  = XML::GetNodeValue( componentnameElt );
	      
	      Log::Debug(OSS() << "Read data framework studyid           : " << framework.studyid_);
	      Log::Debug(OSS() << "                ... studycase         : " << framework.studycase_);
	      Log::Debug(OSS() << "                ... component name    : " << framework.componentname_);
	    }
	  }
	}
	data.setFileList( fileList );
	data.setFrameworkData( framework );


	WrapperParameter parameters;
	XML::Node wrapModeElt = XML::FindElementByName( externalCodeElt, XMLTag_wrap_mode );
	String  wrapType_      = XML::GetAttributeByName( wrapModeElt, XMLTag_type );
	if (wrapType_ == XMLAttr_static_link) parameters.mode_ = WrapperMode::STATICLINK;
	else if (wrapType_ == XMLAttr_dynamic_link) parameters.mode_ = WrapperMode::DYNAMICLINK;
	else if (wrapType_ == XMLAttr_fork) parameters.mode_ = WrapperMode::FORK;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown type (" << wrapType_
						  << ") for " << XMLTag_wrap_mode << " in wrapper description at line "
						  << XML::GetNodeLineNumber( wrapModeElt );
	}

	String  wrapState_    = XML::GetAttributeByName( wrapModeElt, XMLTag_state );
	if (wrapState_ == XMLAttr_shared) parameters.state_ = WrapperState::SHARED;
	else if (wrapState_ == XMLAttr_specific) parameters.state_ = WrapperState::SPECIFIC;
	else if (wrapState_.empty()) parameters.state_ = WrapperState::SHARED;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown state (" << wrapState_
						  << ") for " << XMLTag_wrap_mode << " in wrapper description at line "
						  << XML::GetNodeLineNumber( wrapModeElt );
	}


	XML::Node inDataTransferElt = XML::FindElementByName( wrapModeElt, XMLTag_in_data_transfer );
	String  inMode_              = XML::GetAttributeByName( inDataTransferElt, XMLTag_mode );
	if (inMode_ == XMLAttr_files) parameters.in_ = WrapperDataTransfer::FILES;
	else if (inMode_ == XMLAttr_pipe) parameters.in_ = WrapperDataTransfer::PIPE;
	else if (inMode_ == XMLAttr_arguments) parameters.in_ = WrapperDataTransfer::ARGUMENTS;
	else if (inMode_ == XMLAttr_socket) parameters.in_ = WrapperDataTransfer::SOCKET;
	else if (inMode_ == XMLAttr_corba) parameters.in_ = WrapperDataTransfer::CORBA;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown mode (" << inMode_
						  << ") for " << XMLTag_in_data_transfer << " in wrapper description at line "
						  << XML::GetNodeLineNumber( inDataTransferElt );
	}


	XML::Node outDataTransferElt = XML::FindElementByName( wrapModeElt, XMLTag_out_data_transfer );
	String  outMode_              = XML::GetAttributeByName( outDataTransferElt, XMLTag_mode );
	if (outMode_ == XMLAttr_files) parameters.out_ = WrapperDataTransfer::FILES;
	else if (outMode_ == XMLAttr_pipe) parameters.out_ = WrapperDataTransfer::PIPE;
	else if (outMode_ == XMLAttr_arguments) parameters.out_ = WrapperDataTransfer::ARGUMENTS;
	else if (outMode_ == XMLAttr_socket) parameters.out_ = WrapperDataTransfer::SOCKET;
	else if (outMode_ == XMLAttr_corba) parameters.out_ = WrapperDataTransfer::CORBA;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown mode (" << outMode_
						  << ") for " << XMLTag_out_data_transfer << " in wrapper description at line "
						  << XML::GetNodeLineNumber( outDataTransferElt );
	}


	XML::Node commandElt = XML::FindElementByName( externalCodeElt, XMLTag_command );
	parameters.command_   = XML::GetNodeValue( commandElt );

	data.setParameters( parameters );

	Log::Debug(OSS() << "Read wrapper mode                     : " << WrapperConfigurationModeAsString[parameters.mode_]);
	Log::Debug(OSS() << "         ... state                    : " << WrapperConfigurationStateAsString[parameters.state_]);
	Log::Debug(OSS() << "         ... input  transfer mode     : " << WrapperDataTransferModeAsString[parameters.in_]);
	Log::Debug(OSS() << "         ... output transfer mode     : " << WrapperDataTransferModeAsString[parameters.out_]);
	Log::Debug(OSS() << "         ... command                  : " << parameters.command_);

	if ( (parameters.in_ == WrapperDataTransfer::FILES) &&
	     (! hasOneOrMoreInputFiles) ) Log::Warn( OSS() << "Wrapper description defines input transfer mode as '"
						     << WrapperDataTransferModeAsString[parameters.in_]
						     << "' but doasn't define any input file" );
	if ( (parameters.out_ == WrapperDataTransfer::FILES) &&
	     (! hasOneOrMoreOutputFiles) ) Log::Warn( OSS() << "Wrapper description defines output transfer mode as '"
						      << WrapperDataTransferModeAsString[parameters.out_]
						      << "' but doasn't define any output file" );

	Log::Debug( "Wrapper description successfully parsed" );

	return data;
      }

      /* Stream parsing function */
      void WrapperFile::parseStream(const String & stream)
      {
	// @todo: write stream parsing function
	Log::Debug( "Try parsing stream" );

	// Open the stream...
	XMLDoc doc( stream.c_str(), stream.size() );

	// ... and parse it
	data_ = parseDocument( doc );

	Log::Debug( "Stream successfully parsed" );
      }

      /* Standard parsing function */
      void WrapperFile::parseFile(const FileName & pathToFile)
	/* throw(WrapperFileParsingException) */
      {
	Log::Debug(OSS() << "Try parsing file " << pathToFile);

	// Load the wrapper file...
	XMLDoc doc( pathToFile );

	// ... and parse it
	data_ = parseDocument( doc );

	Log::Debug(OSS() << "File " << pathToFile << " successfully parsed");
      }


      /* Make a new document from the internal data */
      static XMLDoc makeDocument(const WrapperFile & wrapperFile)
      {
	WrapperData theData = wrapperFile.getWrapperData();
	XMLDoc doc;

	// Set the DTD of the wrapper file
	const FileName DTDPath = Path::GetStandardWrapperDirectory() + "/" + DTDFileName;
	//XML::SetDTD( doc, XMLTag_wrapper, DTDPath );

	// Create <wrappper> node
	XML::Node wrapper = XML::NewNode( XMLTag_wrapper );
	XML::SetRootNode( doc, wrapper );

	// Create <library> node
	XML::Node library = XML::NewNode( XMLTag_library );
	XML::AddChild( wrapper, library );

	// Create <path> node
	XML::Node libraryPath = XML::NewNode( XMLTag_path, theData.getLibraryPath() );
	XML::AddChild( library, libraryPath );

	// Create <description> node
	XML::Node description = XML::NewNode( XMLTag_description );
	XML::AddChild( library, description );

	// Create <variable-list> node
	XML::Node variableList = XML::NewNode( XMLTag_variable_list );
	XML::AddChild( description, variableList );

	// Create <variable> node
	const WrapperData::VariableListType vList = theData.getVariableList();
	for (WrapperData::VariableListType::const_iterator it = vList.begin(); it != vList.end(); ++it) {
	  const WrapperDataVariable & theVar = (*it);
	  XML::Node variable = XML::NewNode( XMLTag_variable );
	  XML::SetAttribute( variable, XMLTag_id, theVar.id_ );
	  XML::SetAttribute( variable, XMLTag_type, (theVar.type_ != WrapperDataVariableType::OUT) ? XMLAttr_in : XMLAttr_out );
	  if (theVar.gradient_ == WrapperComputedGradient::YES) XML::SetAttribute( variable, XMLTag_computed_gradient, XMLAttr_yes );
	  XML::AddChild( variableList, variable );

	  // Create <comment> node
	  if (! theVar.comment_.empty()) {
	    XML::Node comment = XML::NewNode( XMLTag_comment, theVar.comment_ );
	    XML::AddChild( variable, comment );
	  }

	  // Create <unit> node
	  if (! theVar.unit_.empty()) {
	    XML::Node unit = XML::NewNode( XMLTag_unit, theVar.unit_ );
	    XML::AddChild( variable, unit );
	  }

	  // Create <regexp> node
	  if (! theVar.regexp_.empty()) {
	    XML::Node regexp = XML::NewNode( XMLTag_regexp, theVar.regexp_ );
	    XML::AddChild( variable, regexp );
	  }

	  // Create <format> node
	  if (! theVar.format_.empty()) {
	    XML::Node format = XML::NewNode( XMLTag_format, theVar.format_ );
	    XML::AddChild( variable, format );
	  }
	} /* end for ( it = vList.begin() ; ... ) */


	// Create <function> node
	const WrapperFunctionDescription funcDescription = theData.getFunctionDescription();
	XML::Node function = XML::NewNode( XMLTag_function, funcDescription.name_ );
	XML::SetAttribute( function, XMLTag_provided, (funcDescription.provided_ == WrapperSymbolProvided::YES) ? XMLAttr_yes : XMLAttr_no );
	XML::AddChild( description, function );

	// Create <gradient> node
	const WrapperFunctionDescription gradDescription = theData.getGradientDescription();
	XML::Node gradient = XML::NewNode( XMLTag_gradient, gradDescription.name_ );
	XML::SetAttribute( gradient, XMLTag_provided, (gradDescription.provided_ == WrapperSymbolProvided::YES) ? XMLAttr_yes : XMLAttr_no );
	XML::AddChild( description, gradient );

	// Create <hessian> node
	const WrapperFunctionDescription hessDescription = theData.getHessianDescription();
	XML::Node hessian = XML::NewNode( XMLTag_hessian, hessDescription.name_ );
	XML::SetAttribute( hessian, XMLTag_provided, (hessDescription.provided_ == WrapperSymbolProvided::YES) ? XMLAttr_yes : XMLAttr_no );
	XML::AddChild( description, hessian );



	// Create <external-code> node
	XML::Node externalCode = XML::NewNode( XMLTag_external_code );
	XML::AddChild( wrapper, externalCode );

	// Create <data> node
	XML::Node data = XML::NewNode( XMLTag_data );
	XML::AddChild( externalCode, data );

	// Create <file> node
	const WrapperData::FileListType fList = theData.getFileList();
	for (WrapperData::FileListType::const_iterator it = fList.begin(); it != fList.end(); ++it) {
	  const WrapperDataFile & theFile = (*it);
	  XML::Node file = XML::NewNode( XMLTag_file );
	  XML::SetAttribute( file, XMLTag_id, theFile.id_ );
	  XML::SetAttribute( file, XMLTag_type, (theFile.type_ != WrapperDataFileType::OUT) ? XMLAttr_in : XMLAttr_out );
	  XML::AddChild( data, file );

	  // Create <name> node
	  if (! theFile.name_.empty()) {
	    XML::Node name = XML::NewNode( XMLTag_name, theFile.name_ );
	    XML::AddChild( file, name );
	  }

	  // Create <path> node
	  if (! theFile.path_.empty()) {
	    XML::Node path = XML::NewNode( XMLTag_path, theFile.path_ );
	    XML::AddChild( file, path );
	  }

	  // Create <subst> node
	  if (! theFile.subst_.empty()) {
	    XML::Node subst = XML::NewNode( XMLTag_subst, theFile.subst_ );
	    XML::AddChild( file, subst );
	  }
	} /* end for ( it = fList.begin() ; ... ) */

	// Create <framework> node
	XML::Node framework = XML::NewNode( XMLTag_framework );
	XML::AddChild( data, framework );

	// Create <study-id> node
	const WrapperFrameworkData frameworkData = theData.getFrameworkData();
	XML::Node studyId = XML::NewNode( XMLTag_study_id, OSS() << frameworkData.studyid_ );
	XML::AddChild( framework, studyId );

	// Create <study-case> node
	XML::Node studyCase = XML::NewNode( XMLTag_study_case, frameworkData.studycase_ );
	XML::AddChild( framework, studyCase );

	// Create <component-name> node
	XML::Node componentName = XML::NewNode( XMLTag_component_name, frameworkData.componentname_ );
	XML::AddChild( framework, componentName );

	// Create <wrap-mode> node
	const WrapperParameter & parameters = theData.getParameters();
	XML::Node wrapMode = XML::NewNode( XMLTag_wrap_mode );
	XML::SetAttribute( wrapMode, XMLTag_type, WrapperConfigurationModeAsString[ parameters.mode_ ]);
	XML::SetAttribute( wrapMode, XMLTag_state, WrapperConfigurationStateAsString[ parameters.state_ ]);
	XML::AddChild( externalCode, wrapMode );

	// Create <in-data-transfer> node
	XML::Node inDataTransfer = XML::NewNode( XMLTag_in_data_transfer );
	XML::SetAttribute( inDataTransfer, XMLTag_mode, WrapperDataTransferModeAsString[ parameters.in_ ]);
	XML::AddChild( wrapMode, inDataTransfer );

	// Create <out-data-transfer> node
	XML::Node outDataTransfer = XML::NewNode( XMLTag_out_data_transfer );
	XML::SetAttribute( outDataTransfer, XMLTag_mode, WrapperDataTransferModeAsString[ parameters.out_ ]);
	XML::AddChild( wrapMode, outDataTransfer );

	// Create <command> node
	XML::Node command = XML::NewNode( XMLTag_command, parameters.command_ );
	XML::AddChild( externalCode, command );

	Bool ok = doc.validate( XMLTag_wrapper, DTDPath );
	if (!ok) throw InternalException(HERE) << "The generated wrapper does not conform to DTD (" << DTDPath << "). Report bug.";

	return doc;
      }

      /* Write the internal data to a wrapper file */
      void WrapperFile::writeFile(const FileName & pathToFile)
      {
	XMLDoc doc = makeDocument( *this );

	// Write out the document to a file
	doc.save( pathToFile );

	setDescriptionFilePath( pathToFile );
      }

      /* Stream out the internal data */
      String WrapperFile::toString() const
      {
	XMLDoc doc = makeDocument( *this );

	// Stream out the data
	return doc.__repr__();
      }

#else

      /* Standard Initialization */
      void WrapperFile::init() const
	/* throw(WrapperFileParsingException) */
      {
	// Nothing to do
      }

      /* Standard finalization */
      void WrapperFile::done() const
      {
	// Nothing to do
      }

      /* Stream parsing function */
      void WrapperFile::parseStream(const String & stream)
      {
	// @todo: write stream parsing function
	throw WrapperFileParsingException(HERE) << "In void WrapperFile::parseStream(const String & stream). Not yet implemented.";
      }

      /* Standard parsing function */
      void WrapperFile::parseFile(const FileName & pathToFile)
	/* throw(WrapperFileParsingException) */
      {
	// @todo: write file parsing function
	throw WrapperFileParsingException(HERE) << "In void WrapperFile::parseFile(const FileName & pathToFile). Not yet implemented.";
      }

      /* Write the internal data to a wrapper file */
      void WrapperFile::writeFile(const FileName & pathToFile)
      {
	// @todo: write function
	throw NotYetImplementedException(HERE) << "In void WrapperFile::writeFile(const FileName & pathToFile).";
      }

      /* Stream out the internal data */
      String WrapperFile::toString() const
      {
	// @todo: write function
	throw NotYetImplementedException(HERE) << "In void WrapperFile:toString().";
      }
#endif


      /* Description file path accessor */
      void WrapperFile::setDescriptionFilePath(const FileName & path)
      {
	descriptionFilePath_ = path;
      }

      /* Description file path accessor */
      FileName WrapperFile::getDescriptionFilePath() const
      {
	return descriptionFilePath_;
      }

      /* Wrapper data accessor */
      void WrapperFile::setWrapperData(const WrapperData & data)
      {
	data_ = data;
      }

      /* Wrapper data accessor */
      const WrapperData & WrapperFile::getWrapperData() const
      {
	return data_;
      }





      /*
       * Find the path of a wrapper file from its name.
       */
      FileName WrapperFile::FindWrapperPathByName(const String & name)
	/* throw(NoWrapperFileFoundException) */
      {
	// Append the extension to the name
	FileName wrapperFileName = name + WrapperFile::extension_;
	Log::Debug( OSS() << "Transform '" << name << "' into '" << wrapperFileName << "'" );

	// Get the directory list...
	std::vector<FileName> directoryList = Path::GetWrapperDirectoryList();

	// ... and search it for the file
	FileName wrapperFullPath;
	try {
	  wrapperFullPath = Path::FindFileByNameInDirectoryList(wrapperFileName, directoryList);

	}
	catch (FileNotFoundException & ex) {
	  throw NoWrapperFileFoundException(HERE) << ex;
	}

	return wrapperFullPath;

      } /* end findWrapperPathByName */



      /*
       * Build a wrapper from a name
       */
      WrapperFile WrapperFile::FindWrapperByName(const String & name)
	/* throw(NoWrapperFileFoundException) */
      {
	// We need to constructs object in the body because of some useful temporary
	// It also speeds the creation of the object because the wrapper file is parsed only once
	FileName wrapperPath;
	  
	// Find the path of the wrapper file that describes the numerical
	// math function interface
	Log::Debug(OSS() << "Try loading a wrapper for function '" << name << "'");
	wrapperPath = WrapperFile::FindWrapperPathByName( name );
	  
	// Open the wrapper file and get the name of the library and the names of
	// the function, its gradient and its hessian if defined
	// If the gradient or the hessian are not defined, ask for default ones
	// Then create the actual functions
	// Get also the data read inside the wrapper file and pass them to the wrapper
	WrapperFile wrapperFile( wrapperPath );
	wrapperFile.setName( name );

	return wrapperFile;
      } /* end FindWrapperByName */


      /*
       * Build a wrapper from a stream
       */
      WrapperFile WrapperFile::BuildWrapperFromStream(const String & xmlStream)
      {
	WrapperFile wrapper;
	wrapper.parseStream( xmlStream );
	return wrapper;
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
