//                                               -*- C++ -*-
/**
 *  @file  FiniteDifferenceGradient.cxx
 *  @brief Class for the creation of a numerical math gradient implementation
 *         form a numerical math evaluation implementation by using centered
 *         finite difference formula.
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: slama $
 *  @date:   $LastChangedDate: 2008-11-19 16:10:42 +0100 (mer, 19 nov 2008) $
 *  Id:      $Id: FiniteDifferenceGradient.cxx 1023 2008-11-19 15:10:42Z slama $
 */

#include "FiniteDifferenceGradient.hxx"
#include "PersistentObjectFactory.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Diff {

      CLASSNAMEINIT(FiniteDifferenceGradient);

      typedef Base::Func::NumericalMathEvaluationImplementation                             NumericalMathEvaluationImplementation;
      typedef Base::Common::NotYetImplementedException   NotYetImplementedException;

      /* Default constructor */
      FiniteDifferenceGradient::FiniteDifferenceGradient()
        : NumericalMathGradientImplementation()
      {
        // Nothing to do
      }

      /* First Parameter constructor  */
      FiniteDifferenceGradient::FiniteDifferenceGradient(
                                                         const NumericalPoint & epsilon,
                                                         const EvaluationImplementation & p_evaluation)
        : NumericalMathGradientImplementation(),
          p_evaluation_(p_evaluation),
          finiteDifferenceStep_(epsilon)
      {
        /* Check if the dimension of the constant term is compatible with the linear and quadratic terms */
        if (epsilon.getDimension() != p_evaluation->getInputDimension())
          throw InvalidDimensionException(HERE) << "Epsilon dimension is incompatible with the given evaluation";

        /* Check if any epsilon component is exactly zero */
        for (UnsignedLong i = 0; i < epsilon.getDimension(); i++)
          {
            if (epsilon[i] == 0.0)
              throw InvalidArgumentException(HERE) << "At least one of the components of epsilon is equal to 0.0, namely component " << i;
          }
      }

      /* SecondParameter constructor */
      FiniteDifferenceGradient::FiniteDifferenceGradient(const NumericalScalar epsilon,
                                                         const EvaluationImplementation & p_evaluation)
        :
        NumericalMathGradientImplementation(),
        p_evaluation_(p_evaluation),
        finiteDifferenceStep_(NumericalPoint(p_evaluation->getInputDimension(), epsilon))
      {
        // Check if epsilon is exactly zero
        if (epsilon == 0.0)
          throw InvalidArgumentException(HERE) << "The given scalar epsilon is equal to 0.0";
      }

      /*  Parameter constructor  with FiniteDifferenceStep*/
      FiniteDifferenceGradient::FiniteDifferenceGradient(
                                                         const FiniteDifferenceStep & finiteDifferenceStep,
                                                         const EvaluationImplementation & p_evaluation)
        /* throw(InvalidDimensionException) */
        :
        NumericalMathGradientImplementation(),
        p_evaluation_(p_evaluation),
        finiteDifferenceStep_(finiteDifferenceStep)
      {
        NumericalPoint epsilon(getEpsilon());
        //Check if the dimension of the constant term is compatible with the linear and quadratic terms
        if (epsilon.getDimension() != p_evaluation->getInputDimension())
          throw InvalidDimensionException(HERE) << "Epsilon dimension is incompatible with the given evaluation";

        //Check if any epsilon component is exactly zero
        for (UnsignedLong i = 0; i < epsilon.getDimension(); i++)
          {
            if (epsilon[i] == 0.0)
              throw InvalidArgumentException(HERE) << "At least one of the components of epsilon is equal to 0.0, namely component " << i;
          }
      }

      /* Comparison operator */
      Bool FiniteDifferenceGradient::operator ==(const FiniteDifferenceGradient & other) const
      {
        return (getEpsilon() == other.getEpsilon());
      }

      /* String converter */
      String FiniteDifferenceGradient:: __repr__() const {
        OSS oss;
        oss << "class=" << FiniteDifferenceGradient::GetClassName()
            << " name=" << getName()
            << " evaluation=" << p_evaluation_-> __repr__();
        return oss;
      }

      /* Accessor for epsilon */
      FiniteDifferenceGradient::NumericalPoint FiniteDifferenceGradient::getEpsilon() const
      {
        return finiteDifferenceStep_.getEpsilon();
      }

      /* Accessor for the evaluation */
      FiniteDifferenceGradient::EvaluationImplementation FiniteDifferenceGradient::getEvaluation() const
      {
        return p_evaluation_;
      }

      /* Accessor for input point dimension */
      UnsignedLong FiniteDifferenceGradient::getInputDimension() const
      {
        return p_evaluation_->getInputDimension();
      }

      /* Accessor for output point dimension */
      UnsignedLong FiniteDifferenceGradient::getOutputDimension() const
      {
        return p_evaluation_->getOutputDimension();
      }

      /* Accessor for the finite difference step */
      void FiniteDifferenceGradient::setFiniteDifferenceStep(const FiniteDifferenceStep & finiteDifferenceStep)
      {
        finiteDifferenceStep_ = finiteDifferenceStep;
      }

      FiniteDifferenceGradient::FiniteDifferenceStep FiniteDifferenceGradient::getFiniteDifferenceStep() const
      {
        return finiteDifferenceStep_;
      }

      /* Method save() stores the object through the StorageManager */
      void FiniteDifferenceGradient::save(StorageManager::Advocate & adv) const
      {
        NumericalMathGradientImplementation::save(adv);
        adv.saveAttribute( "evaluation_", *p_evaluation_ );
        adv.saveAttribute( "finiteDifferenceStep_", finiteDifferenceStep_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void FiniteDifferenceGradient::load(StorageManager::Advocate & adv)
      {
        NumericalMathGradientImplementation::load(adv);
        Base::Common::TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluation;
        adv.loadAttribute( "evaluation_", evaluation );
        p_evaluation_ = evaluation.getImplementation();
        adv.loadAttribute( "finiteDifferenceStep_", finiteDifferenceStep_ );
      }


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
