//                                               -*- C++ -*-
/**
 *  @file  DualLinearCombinationHessianImplementation.cxx
 *  @brief The hessian part of functional linear combination of vectors
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "DualLinearCombinationHessianImplementation.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "SymmetricMatrix.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      CLASSNAMEINIT(DualLinearCombinationHessianImplementation);

      static Base::Common::Factory<DualLinearCombinationHessianImplementation> RegisteredFactory("DualLinearCombinationHessianImplementation");

      typedef Type::SymmetricMatrix SymmetricMatrix;

      /* Default constructor */
      DualLinearCombinationHessianImplementation::DualLinearCombinationHessianImplementation()
        : NumericalMathHessianImplementation(),
          evaluation_()
      {
        // Nothing to do
      }


      /* Parameters constructor */
      DualLinearCombinationHessianImplementation::DualLinearCombinationHessianImplementation(const DualLinearCombinationEvaluationImplementation & evaluation)
        : NumericalMathHessianImplementation(),
          evaluation_(evaluation)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      DualLinearCombinationHessianImplementation * DualLinearCombinationHessianImplementation::clone() const
      {
        return new DualLinearCombinationHessianImplementation(*this);
      }


      /* Hessian method */
      DualLinearCombinationHessianImplementation::SymmetricTensor DualLinearCombinationHessianImplementation::hessian(const NumericalPoint & inP) const
      /* throw(InvalidArgumentException, InternalException) */
      {
        const UnsignedLong inputDimension(getInputDimension());
        if (inP.getDimension() != inputDimension) throw InvalidArgumentException(HERE) << "Error: the given point has an invalid dimension. Expect a dimension " << inputDimension << ", got " << inP.getDimension();
        const UnsignedLong size(evaluation_.functionsCollection_.getSize());
        const UnsignedLong outputDimension(getOutputDimension());
        SymmetricTensor result(inputDimension, outputDimension);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            const SymmetricTensor hessianI(evaluation_.functionsCollection_[i].hessian(inP));
            const NumericalPoint coefficientI(evaluation_.coefficients_[i]);
            for (UnsignedLong j = 0; j < inputDimension; ++j)
              for (UnsignedLong k = 0; k <= j; ++k)
                for (UnsignedLong n = 0; n < outputDimension; ++n)
                  result(j, k, n) += hessianI(j, k, 0) * coefficientI[n];
          }
        return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong DualLinearCombinationHessianImplementation::getInputDimension() const
      /* throw(InternalException) */
      {
        return evaluation_.getInputDimension();
      }

      /* Accessor for output point dimension */
      UnsignedLong DualLinearCombinationHessianImplementation::getOutputDimension() const
      /* throw(InternalException) */
      {
        return evaluation_.getOutputDimension();
      }

      /* String converter */
      String DualLinearCombinationHessianImplementation::__repr__() const
      {
        return OSS() << "class=" << GetClassName()
                     << " evaluation=" << evaluation_;
      }


      /* Method save() stores the object through the StorageManager */
      void DualLinearCombinationHessianImplementation::save(StorageManager::Advocate & adv) const
      {
        PersistentObject::save(adv);
        adv.saveAttribute( "evaluation_", evaluation_ );
      }


      /* Method load() reloads the object from the StorageManager */
      void DualLinearCombinationHessianImplementation::load(StorageManager::Advocate & adv)
      {
        PersistentObject::load(adv);
        adv.loadAttribute( "evaluation_", evaluation_ );
      }



    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
