//                                               -*- C++ -*-
/**
 *  @file  ProductPolynomialEvaluationImplementation.cxx
 *  @brief This is a nD polynomial build as a product of n 1D polynomial
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "ProductPolynomialEvaluationImplementation.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      using Func::UniVariatePolynomial;
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<UniVariatePolynomial>);

      static Common::Factory<PersistentCollection<UniVariatePolynomial> > RegisteredFactory("PersistentCollection<UniVariatePolynomial>");
    } /* namespace Type */

    namespace Func
    {

      CLASSNAMEINIT(ProductPolynomialEvaluationImplementation);

      static Common::Factory<ProductPolynomialEvaluationImplementation> RegisteredFactory("ProductPolynomialEvaluationImplementation");

      typedef Common::NotYetImplementedException NotYetImplementedException;

      /* Default constructor */
      ProductPolynomialEvaluationImplementation::ProductPolynomialEvaluationImplementation()
        : NumericalMathEvaluationImplementation(),
          polynomials_()
      {
        // Nothing to do
      }


      /* Constructor */
      ProductPolynomialEvaluationImplementation::ProductPolynomialEvaluationImplementation(const PolynomialCollection & coll)
        : NumericalMathEvaluationImplementation(),
          polynomials_(coll)
      {
        // Nothing to do
        Description description(0);
        for (UnsignedLong i = 0; i < getInputDimension(); ++i) description.add(OSS() << "x" << i);
        for (UnsignedLong i = 0; i < getOutputDimension(); ++i) description.add(OSS() << "y" << i);
        setDescription(description);
      }


      /* Virtual constructor */
      ProductPolynomialEvaluationImplementation * ProductPolynomialEvaluationImplementation::clone() const
      {
        return new ProductPolynomialEvaluationImplementation(*this);
      }


      /* String converter */
      String ProductPolynomialEvaluationImplementation::__repr__() const
      {
        return OSS() << "class=" << GetClassName()
                     << ", polynomials=" << polynomials_;
      }

      String ProductPolynomialEvaluationImplementation::__str__(const String & offset) const
      {
        const UnsignedLong size(polynomials_.getSize());
        if (size == 0) return "";
        const Description description(getInputDescription());
        if (size == 1) return polynomials_[0].__str__("", description[0]);
        OSS oss(false);
        oss << offset;
        for (UnsignedLong i = 0; i < size; ++i) oss << (i > 0 ? "*" : "") << "(" << polynomials_[i].__str__("", description[i]) << ")";
        return oss;
      }


      /* Operator (): Evaluate a product of 1D polynomials for one sample */
      ProductPolynomialEvaluationImplementation::NumericalPoint ProductPolynomialEvaluationImplementation::operator() (const NumericalPoint & inP) const
      /* throw(InvalidArgumentException, InternalException) */
      {
        const UnsignedLong inDimension(inP.getDimension());
        if (inDimension != getInputDimension()) throw InvalidArgumentException(HERE) << "Error: trying to evaluate a ProductPolynomialFunction with an argument of invalid dimension";
        NumericalScalar productEvaluation(1.0) ;
        for (UnsignedLong i = 0; i < inDimension; ++i) productEvaluation *= polynomials_[i](inP[i]);
        return NumericalPoint(1, productEvaluation);
      }


      /* Accessor for input point dimension */
      UnsignedLong ProductPolynomialEvaluationImplementation::getInputDimension() const
      /* throw(InternalException) */
      {
        return polynomials_.getSize();
      }

      /* Accessor for output point dimension */
      UnsignedLong ProductPolynomialEvaluationImplementation::getOutputDimension() const
      /* throw(InternalException) */
      {
        return 1;
      }


      /* Method save() stores the object through the StorageManager */
      void ProductPolynomialEvaluationImplementation::save(StorageManager::Advocate & adv) const
      {
        NumericalMathEvaluationImplementation::save(adv);
        adv.saveAttribute( "polynomials_", polynomials_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void ProductPolynomialEvaluationImplementation::load(StorageManager::Advocate & adv)
      {
        NumericalMathEvaluationImplementation::load(adv);
        adv.loadAttribute( "polynomials_", polynomials_ );
      }


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
