//                                               -*- C++ -*-
/**
 *  @file  CovarianceMatrix.cxx
 *  @brief The class CovarianceMatrix implements blank free samples
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-06-30 10:19:34 +0200 (Thu, 30 Jun 2011) $
 *  Id:      $Id: CovarianceMatrix.cxx 1972 2011-06-30 08:19:34Z schueller $
 */
#include "CovarianceMatrix.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Stat
    {

      CLASSNAMEINIT(CovarianceMatrix);

      /* Default constructor */
      CovarianceMatrix::CovarianceMatrix()
        : SymmetricMatrix(0)
      {
        // Nothing to do
      }

      /* Constructor with size (dim, which is the same for nbRows_ and nbColumns_ )*/
      CovarianceMatrix::CovarianceMatrix(const UnsignedLong dim)
        : SymmetricMatrix(dim)
      {
        // Initialize the correlation matrix to the identity matrix
        for(UnsignedLong i = 0; i < dim; ++i) operator()(i, i) = 1.0;
      }

      /* Constructor from external collection */
      /* If the dimensions of the matrix and of the collection */
      /* do not match, either the collection is truncated */
      /* or the rest of the matrix is filled with zeros */
      CovarianceMatrix::CovarianceMatrix(const UnsignedLong dim,
                                         const OT::Base::Type::Collection<NumericalScalar> &elementsValues)
        : SymmetricMatrix(dim,elementsValues)
      {
        // Nothing to do
      }

      /* Constructor with implementation */
      CovarianceMatrix::CovarianceMatrix(const Implementation & i)
        : SymmetricMatrix(i)
      {
        // Nothing to do
      }

      /* String converter */
      String CovarianceMatrix::__repr__() const
      {
        checkSymmetry();
        return OSS() << "class=" << getClassName()
                     << " dimension=" << this->getDimension()
                     << " implementation=" << getImplementation()->__repr__();
      }

      /* CovarianceMatrix transpose */
      CovarianceMatrix CovarianceMatrix::transpose () const
      {
        return *this;
      }

      /* CovarianceMatrix addition (must have the same dimensions) */
      CovarianceMatrix CovarianceMatrix::operator + (const CovarianceMatrix & m) const /* throw(InvalidDimensionException) */
      {
        checkSymmetry();
        return CovarianceMatrix(new Type::MatrixImplementation(*getImplementation() + *(m.getImplementation()) ));
      }

      /* CovarianceMatrix multiplication (must have consistent dimensions) */
      CovarianceMatrix CovarianceMatrix::operator * (const IdentityMatrix & m) const /* throw(InvalidDimensionException) */
      {
        return *this;
      }

      /* Check if the matrix is SPD */
      Bool CovarianceMatrix::isPositiveDefinite(const Bool keepIntact)
      {
        checkSymmetry();
        return getImplementation()->isPositiveDefinite(keepIntact);
      }

      /* Build the Cholesky factorization of the matrix */
      Type::SquareMatrix CovarianceMatrix::computeCholesky(const Bool keepIntact)
      {
        checkSymmetry();
        return Type::SquareMatrix(this->getDimension(),getImplementation()->computeCholesky(keepIntact));
      }





    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */
