//                                               -*- C++ -*-
/**
 *  @file  NumericalSampleImplementation.cxx
 *  @brief The class NumericalSampleImplementation implements blank free samples
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-07-06 12:03:57 +0200 (Wed, 06 Jul 2011) $
 *  Id:      $Id: NumericalSampleImplementation.cxx 1991 2011-07-06 10:03:57Z schueller $
 */
#include <limits>        // std::numeric_limits
#include <map>
#include <cmath>
#include <fstream>
#include <iomanip>
#include <algorithm>
#include <iterator>
#include <cstdlib>
#include <cstdio>        // std::fopen, std::errno
#include <cstring>       // std::strerror

#include "OTconfig.hxx"
#include "NumericalSampleImplementation.hxx"
#include "StorageManager.hxx"
#include "PersistentObjectFactory.hxx"
#include "Log.hxx"
#include "Exception.hxx"
#include "ResourceMap.hxx"
#include "Path.hxx"
#include "Os.hxx"
#include "TBB.hxx"

#include "csv_parser_state.hxx"
#include "csv_parser.hh"
#include "csv_lexer.h"

int csvparse (OT::Base::Stat::CSVParserState & theState, yyscan_t yyscanner, FILE * theFile, OT::Base::Stat::NumericalSampleImplementation &impl, OT::UnsignedLong & theDimension, const char * separator);


namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<NumericalPoint>);

      static Common::Factory<PersistentCollection<NumericalPoint> > RegisteredFactory("PersistentCollection<NumericalPoint>");
    } /* namespace Type */



    namespace Stat
    {

      using Common::InvalidArgumentException;

      NSI_point::NSI_point(NumericalSampleImplementation * p_nsi, const UnsignedLong index)
        : p_nsi_(p_nsi), index_(index), dimension_(p_nsi->dimension_) {}

      NSI_point & NSI_point::operator = (const NSI_point & rhs)
      {
        if ( (this != &rhs) && (getDimension() == rhs.getDimension()) )
          std::copy( rhs.begin(), rhs.end(), begin() );
        return *this;
      }

      NSI_point & NSI_point::operator = (const NSI_const_point & rhs)
      {
        if ( getDimension() == rhs.getDimension() )
          std::copy( rhs.begin(), rhs.end(), begin() );
        return *this;
      }

      NSI_point & NSI_point::operator = (const Type::NumericalPoint & rhs)
      {
        if ( getDimension() == rhs.getDimension() )
          std::copy( rhs.begin(), rhs.end(), begin() );
        return *this;
      }

      NumericalScalar & NSI_point::operator [] (UnsignedLong i)
      {
        return p_nsi_->data_[index_ * dimension_ + i];
      }

      const NumericalScalar & NSI_point::operator [] (const UnsignedLong i) const
      {
        return p_nsi_->data_[index_ * dimension_ + i];
      }

      NumericalScalar & NSI_point::at (UnsignedLong i)
      {
        return p_nsi_->data_.at(index_ * dimension_ + i);
      }

      const NumericalScalar & NSI_point::at (const UnsignedLong i) const
      {
        return p_nsi_->data_.at(index_ * dimension_ + i);
      }

      NSI_point & NSI_point::operator += (const NSI_point & other)
      {
        if (getDimension() != other.getDimension())
          throw InvalidArgumentException(HERE)
            << "NumericalPoints of different dimensions cannot be added (LHS dimension = "
            << getDimension()
            << "; RHS dimension = "
            << other.getDimension();

        for (UnsignedLong i = 0; i < getDimension(); ++i) (*this)[i] += other[i];
        return *this;
      }

      NSI_point & NSI_point::operator -= (const NSI_point & other)
      {
        if (getDimension() != other.getDimension())
          throw InvalidArgumentException(HERE)
            << "NumericalPoints of different dimensions cannot be added (LHS dimension = "
            << getDimension()
            << "; RHS dimension = "
            << other.getDimension();

        for (UnsignedLong i = 0; i < getDimension(); ++i) (*this)[i] -= other[i];
        return *this;
      }

      NSI_point & NSI_point::operator += (const Type::NumericalPoint & other)
      {
        if (getDimension() != other.getDimension())
          throw InvalidArgumentException(HERE)
            << "NumericalPoints of different dimensions cannot be added (LHS dimension = "
            << getDimension()
            << "; RHS dimension = "
            << other.getDimension();

        for (UnsignedLong i = 0; i < getDimension(); ++i) (*this)[i] += other[i];
        return *this;
      }

      NSI_point & NSI_point::operator -= (const Type::NumericalPoint & other)
      {
        if (getDimension() != other.getDimension())
          throw InvalidArgumentException(HERE)
            << "NumericalPoints of different dimensions cannot be added (LHS dimension = "
            << getDimension()
            << "; RHS dimension = "
            << other.getDimension();

        for (UnsignedLong i = 0; i < getDimension(); ++i) (*this)[i] -= other[i];
        return *this;
      }

      NSI_point & NSI_point::operator *= (const NumericalScalar val)
      {
        for(UnsignedLong i = 0; i < getDimension(); ++i) (*this)[i] *= val;
        return *this;
      }


      bool operator == (const NSI_point & lhs, const NSI_point & rhs)
      {
        return (lhs.getDimension() == rhs.getDimension()) &&
          std::equal(lhs.begin(), lhs.end(), rhs.begin());
      }

      bool operator != (const NSI_point & lhs, const NSI_point & rhs)
      {
        return ! (lhs == rhs);
      }

      bool operator < (const NSI_point & lhs, const NSI_point & rhs)
      {
        return std::lexicographical_compare(lhs.begin(), lhs.end(),
                                            rhs.begin(), rhs.end(),
                                            std::less<NumericalScalar>());
      }

      bool operator > (const NSI_point & lhs, const NSI_point & rhs)
      {
        return !( lhs <= rhs );
      }

      bool operator <= (const NSI_point & lhs, const NSI_point & rhs)
      {
        return std::lexicographical_compare(lhs.begin(), lhs.end(),
                                            rhs.begin(), rhs.end(),
                                            std::less_equal<NumericalScalar>());
      }

      bool operator >= (const NSI_point & lhs, const NSI_point & rhs)
      {
        return !( lhs < rhs );
      }





      NSI_const_point::NSI_const_point(const NumericalSampleImplementation * p_nsi, const UnsignedLong index)
        : p_nsi_(p_nsi), index_(index), dimension_(p_nsi->dimension_) {}

      NSI_const_point::NSI_const_point(const NSI_point & point)
        : p_nsi_(point.p_nsi_), index_(point.index_), dimension_(point.dimension_) {}

      bool operator == (const NSI_const_point & lhs, const NSI_const_point & rhs)
      {
        return (lhs.getDimension() == rhs.getDimension()) &&
          std::equal(lhs.begin(), lhs.end(), rhs.begin());
      }

      const NumericalScalar & NSI_const_point::operator [] (const UnsignedLong i) const
      {
        return p_nsi_->data_[index_ * dimension_ + i];
      }

      const NumericalScalar & NSI_const_point::at (const UnsignedLong i) const
      {
        return p_nsi_->data_.at(index_ * dimension_ + i);
      }

      bool operator != (const NSI_const_point & lhs, const NSI_const_point & rhs)
      {
        return ! (lhs == rhs);
      }

      bool operator < (const NSI_const_point & lhs, const NSI_const_point & rhs)
      {
        return std::lexicographical_compare(lhs.begin(), lhs.end(),
                                            rhs.begin(), rhs.end(),
                                            std::less<NumericalScalar>());
      }

      bool operator > (const NSI_const_point & lhs, const NSI_const_point & rhs)
      {
        return !( lhs <= rhs );
      }

      bool operator <= (const NSI_const_point & lhs, const NSI_const_point & rhs)
      {
        return std::lexicographical_compare(lhs.begin(), lhs.end(),
                                            rhs.begin(), rhs.end(),
                                            std::less_equal<NumericalScalar>());
      }

      bool operator >= (const NSI_const_point & lhs, const NSI_const_point & rhs)
      {
        return !( lhs < rhs );
      }







      typedef NumericalSampleImplementation (*BuildMethod) (const FileName & fileName);
      using Common::NotYetImplementedException;
      using Common::OutOfBoundException;
      using Common::Log;
      using Common::ResourceMap;
      using Common::Path;
      using Common::Os;

      static Common::Factory<NumericalSampleImplementation> RegisteredFactory("NumericalSampleImplementation");


      /*
       * This class implements a map that stores a function pointer to a factory that builds
       * a NumericalSampleImplementation according to a file format
       */
      class BuildMethodMap : public std::map<NumericalSampleImplementation::ExternalFileFormat, BuildMethod>
      {
        typedef std::map<NumericalSampleImplementation::ExternalFileFormat, BuildMethod> ParentType;
        ParentType & table_;

      public:
        BuildMethodMap()
          : std::map<NumericalSampleImplementation::ExternalFileFormat, BuildMethod>(),
            table_(*this)
        {
          table_[NumericalSampleImplementation::CSV] = NumericalSampleImplementation::BuildFromCSVFile;
        }

        const BuildMethod & operator[] (const NumericalSampleImplementation::ExternalFileFormat & format) const
        {
          return table_[format];
        }

      }; /* end class BuildMethodMap */


      static const BuildMethodMap SampleImportationFactoryMap;



      CLASSNAMEINIT(NumericalSampleImplementation);


      /* Constructor from file */
      NumericalSampleImplementation NumericalSampleImplementation::GetFromFile(const FileName & fileName,
                                                                               const ExternalFileFormat format)
      {
        return SampleImportationFactoryMap[format](fileName);
      }

      /* Factory of NumericalSampleImplementation from CSV file */
      NumericalSampleImplementation NumericalSampleImplementation::BuildFromCSVFile(const FileName & fileName) /* throw(FileNotFoundException, InternalException) */
      {
        const String csvSeparator = ResourceMap::Get( "csv-file-separator" );

        yyscan_t scanner = 0;
        NumericalSampleImplementation impl(0, 0);
        impl.setName(fileName);

        FILE * theFile = std::fopen(fileName.c_str(),"r");
        if (!theFile)
          { // theFile can not be found. Errno is set
            throw FileNotFoundException(HERE) << "Can NOT open file '" << fileName
                                              << "'. Reason: " << std::strerror(errno);
          }
        CSVParserState state;
        state.theFileName = fileName;

        csvlex_init(&scanner);
        csvparse(state, scanner, theFile, impl, impl.dimension_, csvSeparator.c_str());
        csvlex_destroy(scanner);
        std::fclose(theFile);
        // Check the description
        if (impl.p_description_.isNull() || (impl.p_description_->getSize() != impl.getDimension()))
          {
            const UnsignedLong dimension(impl.getDimension());
            Description defaultDescription(dimension);
            for (UnsignedLong i = 0; i < dimension; ++i)
              defaultDescription[i] = String(OSS() << "data_" << i);
            impl.setDescription(defaultDescription);
          }
        return impl;
      }

      /* Store a sample in a temporary text file, one realization by line. Returns the file name. */
      String NumericalSampleImplementation::storeToTemporaryFile() const
      {
        const String dataFileName(Path::BuildTemporaryFileName("RData.txt.XXXXXX"));
        std::ofstream dataFile(dataFileName.c_str());
        // Fill-in the data file
        for (UnsignedLong i = 0; i < size_; ++i)
          {
            String separator = "";
            for (UnsignedLong j = 0; j < dimension_; ++j, separator = " ")
              dataFile << separator << std::setprecision(16) << operator[](i)[j];
            dataFile << Os::GetEndOfLine();
          }
        dataFile.close();
        return dataFileName;
      }

      /* Export a sample as a matrix, one row by realization, in a format suitable to exchange with R */
      String NumericalSampleImplementation::streamToRFormat() const
      {
        OSS oss;
        oss.setPrecision(16);
        oss << "matrix(c(";
        String separator("");
        for (UnsignedLong j = 0; j < dimension_; ++j)
          for (UnsignedLong i = 0; i < size_; ++i, separator = ",")
            oss << separator << operator[](i)[j];
        oss << "), nrow=" << size_ << ", ncol=" << dimension_ << ")";
        return oss;
      }

      /* Default constructor is private */
      NumericalSampleImplementation::NumericalSampleImplementation()
        : Common::PersistentObject(),
          size_(0),
          dimension_(0),
          data_(size_ * dimension_, 0.0),
          p_description_()
      {
        // Nothing to do
      }

      /* Standard constructor */
      NumericalSampleImplementation::NumericalSampleImplementation(const UnsignedLong size,
                                                                   const UnsignedLong dim)
        : Common::PersistentObject(),
          size_(size),
          dimension_(dim),
          data_(size_ * dimension_, 0.0),
          p_description_()
      {
        // Nothing to do
      }

      /* Constructor from a NumericalPoint */
      NumericalSampleImplementation::NumericalSampleImplementation(const UnsignedLong size,
                                                                   const NumericalPoint & point)
        : Common::PersistentObject(),
          size_(size),
          dimension_(point.getDimension()),
          data_(size_ * dimension_, 0.0),
          p_description_()
      {
        for (UnsignedLong i = 0; i < size_; ++i)
          for (UnsignedLong j = 0; j < dimension_; ++j)
            data_[i * dimension_ + j] = point[j];
      }


      /* Constructor from a collection of NumericalPoint */
      NumericalSampleImplementation::NumericalSampleImplementation(const Type::Collection<NumericalPoint> & coll)
        : Common::PersistentObject(),
          size_(coll.getSize()),
          dimension_((coll.getSize()>0) ? coll[0].getDimension() : 0),
          data_(size_ * dimension_, 0.0),
          p_description_()
      {
        for (UnsignedLong i = 0; i < size_; ++i)
          for (UnsignedLong j = 0; j < dimension_; ++j)
            data_[i * dimension_ + j] = coll[i][j];
      }

      /* Partial copy constructor */
      NumericalSampleImplementation::NumericalSampleImplementation(const NumericalSampleImplementation & other, iterator first, iterator last)
        : Common::PersistentObject(),
          size_(last - first),
          dimension_(other.getDimension()),
          data_(size_ * dimension_, 0.0),
          p_description_(other.p_description_)
      {
        std::copy( first, last, begin() );
      }


      /* Virtual constructor */
      NumericalSampleImplementation * NumericalSampleImplementation::clone() const
      {
        return new NumericalSampleImplementation(*this);
      }



      void NumericalSampleImplementation::swap_points(const UnsignedLong a, const UnsignedLong b)
      {
        std::swap_ranges( &data_[ a * dimension_ ], &data_[ (a + 1) * dimension_ ], &data_[ b * dimension_ ] );
      }

      void NumericalSampleImplementation::swap_range_points(const UnsignedLong fa, const UnsignedLong ta, const UnsignedLong fb)
      {
        for (UnsignedLong i = 0; i < ta - fa; ++i) swap_points( fa + i, fb + i );
      }



      /* Description Accessor */
      void NumericalSampleImplementation::setDescription(const Description & description)
      {
        if (description.getSize() != getDimension()) throw InvalidArgumentException(HERE) << "Error: the given description does not match the sample dimension.";
        p_description_ = description.getImplementation();
      }


      /* Description Accessor */
      NumericalSampleImplementation::Description NumericalSampleImplementation::getDescription() const
      {
        return p_description_.isNull() ? Description(getDimension()) : *p_description_;
      }

      /* Comparison function */
      Bool operator ==(const NumericalSampleImplementation & lhs, const NumericalSampleImplementation & rhs)
      {
        return (lhs.size_ == rhs.size_) && (lhs.dimension_ == rhs.dimension_) &&
          std::equal( lhs.begin(), lhs.end(), rhs.begin() );
      }

      void NumericalSampleImplementation::erase(iterator first, iterator last)
      {
        Type::PersistentCollection<NumericalScalar>::iterator f = data_.begin() + (first - begin()) * dimension_;
        Type::PersistentCollection<NumericalScalar>::iterator l = data_.begin() + (last - begin()) * dimension_;
        data_.erase( f, l );
        size_ -= last - first;
      }

      void NumericalSampleImplementation::erase(const UnsignedLong first, const UnsignedLong last)
      {
        Type::PersistentCollection<NumericalScalar>::iterator f = data_.begin() + first * dimension_;
        Type::PersistentCollection<NumericalScalar>::iterator l = data_.begin() + last * dimension_;
        data_.erase( f, l );
        size_ -= last - first;
      }

      void NumericalSampleImplementation::clear()
      {
        data_.clear();
        size_ = 0;
      }

      /* Method __contains__() is for Python */
      Bool NumericalSampleImplementation::contains(const NumericalPoint & val) const
      {
        for (UnsignedLong i = 0; i < size_; ++i) if ( (*this)[i] == val ) return true;
        return false;
      }

      /* String converter */
      String NumericalSampleImplementation::__repr__() const
      {
        OSS oss;
        oss << "class=" << NumericalSampleImplementation::GetClassName()
            << " name=" << getName()
            << " size=" << size_
            << " dimension=" << dimension_
            << " data=[";
        const char * sep = "";
        for(const_iterator it=begin(); it!=end(); ++it, sep=",") oss << sep << *it;
        oss << "]";
        return oss;
      }

      String NumericalSampleImplementation::__str__(const String & offset) const
      {
	// First, print the description if it is not empty.
        // If you use the getDescription() method you get a default value
        // for the description that is not stored in the sample, producing a spurious output
        const Bool printDescription = !p_description_.isNull() && (p_description_->getSize() == dimension_);

	size_t twidth = 0; // column title max width
        size_t lwidth = 0; // LHS number max width
        size_t rwidth = 0; // RHS number max width
        size_t iwidth = 0; // index max width

	if (printDescription) {
          for( UnsignedLong j=0; j<dimension_; ++j )
	    twidth = std::max( twidth, (*p_description_)[j].size() );
	}

        for( UnsignedLong i=0; i<size_; ++i )
          for( UnsignedLong j=0; j<dimension_; ++j ) {
            String st = OSS() << data_[i * dimension_ + j];
            size_t dotpos = st.find( '.' );
            lwidth = std::max( lwidth, (dotpos != String::npos) ? dotpos             : st.size() );
            rwidth = std::max( rwidth, (dotpos != String::npos) ? st.size() - dotpos : 0         );
          }

	if (twidth > lwidth + rwidth)
	  rwidth = twidth - lwidth;
	else
	  twidth = lwidth + rwidth;

	{
	  // Computing the size of the last index (max width of the indexes)
	  String sti = OSS() << size_-1;
	  iwidth = sti.size();
	}

	OSS oss;
	// Print the column title
	if (printDescription) {
	  oss << offset << String( iwidth , ' ' ) << "   [ ";
          const char * sep = "";
          for( UnsignedLong j=0; j<dimension_; ++j, sep=" " ) {
	    oss << sep << (*p_description_)[j] << String( twidth - (*p_description_)[j].size(), ' ' );
	  }
	  oss << " ]\n";
	}

        const char * newline = "";
        for( UnsignedLong i=0; i<size_; ++i, newline="\n" ) {
	  String sti = OSS() << i;
          oss << newline << offset << String( iwidth - sti.size(), ' ' ) << sti << " : [ ";
          const char * sep = "";
          for( UnsignedLong j=0; j<dimension_; ++j, sep=" " ) {
            String st = OSS() << data_[i * dimension_ + j];
            size_t dotpos = st.find( '.' );
            oss << sep << String( lwidth - ((dotpos != String::npos) ? dotpos : st.size()), ' ' )
                << st
                << String( rwidth - ((dotpos != String::npos) ? st.size() - dotpos : 0), ' ' );
          }
          oss << " ]";
        }
        return oss;
      }





      /* Appends an element to the collection */
      NumericalSampleImplementation & NumericalSampleImplementation::add(const NumericalPoint & point)
      /* throw(InvalidArgumentException)*/
      {
        if ( (dimension_ != 0) && (point.getDimension() != dimension_) )
          throw InvalidArgumentException(HERE) << "Point has invalid dimension ("
                                               << point.getDimension()
                                               << ") expected : "
                                               << getDimension();
        const UnsignedLong oldSize = size_;
        ++size_;
        data_.resize( size_ * dimension_ );
        //for(UnsignedLong j=0; j<dimension_; ++j)
        //  data_[(size_-1)*dimension_ + j] = point[j];
        memcpy( &data_[oldSize * dimension_], &point[0], dimension_ * sizeof(NumericalScalar) );
        return *this;
      }


      /* Appends another sample to the collection */
      NumericalSampleImplementation & NumericalSampleImplementation::add(const NumericalSampleImplementation & sample)
      /* throw(InvalidArgumentException)*/
      {
        if ( (dimension_ != 0) && (sample.getDimension() != dimension_) )
          throw InvalidArgumentException(HERE) << "Sample has invalid dimension ("
                                               << sample.getDimension()
                                               << ") expected : "
                                               << getDimension();
        const UnsignedLong oldSize = size_;
        size_ += sample.getSize();
        data_.resize( size_ * dimension_ );
        memmove( &data_[oldSize * dimension_], &(sample.data_[0]), sample.getSize() * dimension_ * sizeof(NumericalScalar) );
        return *this;
      }


      struct AddPolicy
      {
        typedef Type::NumericalPoint value_type;

        static inline value_type GetInvariant(const NumericalSampleImplementation & nsi)
        { return value_type(nsi.getDimension(), 0.0); }


        template <typename T>
        static inline value_type & inplace_op( value_type & a, const T & pt )
        {
          const UnsignedLong dim = a.getDimension();
          for (UnsignedLong i=0; i<dim; ++i) a[i] += pt[i];
          return a;
        }
      }; /* end struct AddPolicy */

      template <typename OP>
      struct ReductionFunctor {
        const NumericalSampleImplementation & nsi_;
        const OP & op_;
        typename OP::value_type accumulator_;

        ReductionFunctor(const NumericalSampleImplementation & nsi, const OP & op = OP())
          : nsi_(nsi), op_(op), accumulator_(OP::GetInvariant(nsi_)) {}

        ReductionFunctor(const ReductionFunctor & other, TBB::Split)
          : nsi_(other.nsi_), op_(other.op_), accumulator_(OP::GetInvariant(nsi_)) {}

        void operator() (const TBB::BlockedRange<UnsignedLong> & r)
        { for (UnsignedLong i = r.begin(); i != r.end(); ++i) op_.inplace_op( accumulator_, nsi_[i] ); }

        void join(const ReductionFunctor & other)
        { op_.inplace_op( accumulator_, other.accumulator_ ); }

      }; /* end struct ReductionFunctor */


      template <typename OP>
      class ParallelFunctor {
        NumericalSampleImplementation & nsi_;
        const OP & op_;
      public:
        ParallelFunctor(NumericalSampleImplementation & nsi, const OP & op) : nsi_(nsi), op_(op) {}

        void operator() (const TBB::BlockedRange<UnsignedLong> & r) const
        { for (UnsignedLong i = r.begin(); i != r.end(); ++i) op_.inplace_op( nsi_[i] ); }

      }; /* end class ParallelFunctor */



      /*
       * Gives the mean of the sample, based on the formula
       * mean = sum of the elements in the sample / size of the sample
       */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::computeMean() const
      {
        ReductionFunctor<AddPolicy> functor( *this );
        TBB::ParallelReduce( 0, size_, functor );
        return functor.accumulator_ * (1.0 / size_);
      }

      /*
       * Gives the covariance matrix of the sample, normalization by 1 / (size - 1) if size > 1
       */
      NumericalSampleImplementation::CovarianceMatrix NumericalSampleImplementation::computeCovariance() const
      {
        // Special case for a sample of size 1
        if (size_ == 1) return CovarianceMatrix(SquareMatrix(dimension_).getImplementation());

        const NumericalPoint mean(computeMean());
        CovarianceMatrix covariance(SquareMatrix(dimension_).getImplementation());
        for (UnsignedLong index = 0; index < size_; ++index) {
          const NumericalPoint realization( (*this)[index] );
          for (UnsignedLong i = 0; i < dimension_; ++i)
            for (UnsignedLong j = 0; j <= i; ++j)
              covariance(i, j) += (realization[i] - mean[i]) * (realization[j] - mean[j]);
        }

        const NumericalScalar alpha(1.0 / (size_ -1));
        for (UnsignedLong i = 0; i < dimension_; ++i)
          for (UnsignedLong j = 0; j <= i; ++j)
            covariance(i, j) *= alpha;

        return covariance;
      }

      /*
       * Gives the standard deviation of the sample, i.e. the square-root of the covariance matrix.
       */
      NumericalSampleImplementation::SquareMatrix NumericalSampleImplementation::computeStandardDeviation() const
      {
        return computeCovariance().computeCholesky();
      }

      /*
       * Gives the standard deviation of each component of the sample
       */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::computeStandardDeviationPerComponent() const
      {
        // Special case for a sample of size 1
        if (size_ == 1) return NumericalPoint(dimension_, 0.0);

        const NumericalPoint mean(computeMean());
        NumericalPoint sd(dimension_);
        for (UnsignedLong index = 0; index < size_; ++index) {
          NumericalPoint centeredRealization((*this)[index] - mean);
          for (UnsignedLong i = 0; i < dimension_; ++i)
            sd[i] += centeredRealization[i] * centeredRealization[i];
        }

        sd *= 1.0 / (size_ - 1);
        for (UnsignedLong i = 0; i < dimension_; ++i)
          sd[i] = sqrt(sd[i]);

        return sd;
      }



      /*
       * Gives the Pearson correlation matrix of the sample
       */
      NumericalSampleImplementation::CorrelationMatrix NumericalSampleImplementation::computePearsonCorrelation() const
      {
        CorrelationMatrix correlation(dimension_);
        if (dimension_ == 1) return correlation;

        const CovarianceMatrix covariance(computeCovariance());
        NumericalPoint sd(dimension_);
        for (UnsignedLong i = 0; i < dimension_; ++i) {
          sd[i] = sqrt( covariance(i,i) );
          for (UnsignedLong j = 0; j < i; ++j)
            correlation(i, j) = covariance(i, j) / (sd[i] * sd[j]);
        }

        return correlation;
      }

      struct Pair
      {
        NumericalScalar value_;
        UnsignedLong index_;
        Pair() : value_(0.0), index_() {}
        Pair(NumericalScalar value, UnsignedLong index) : value_(value), index_(index) {}
        Bool operator < (const Pair & other) const
        { return value_ < other.value_; }
      };

      typedef Type::Collection<Pair>                   PairCollection;
      typedef Type::Collection<PairCollection>         PairCollectionCollection;
      typedef Type::Collection<UnsignedLong>           UnsignedLongCollection;
      typedef Type::Collection<UnsignedLongCollection> UnsignedLongCollectionCollection;

      struct Comparison
      {
        // Main sorting key
        UnsignedLong first_;
        // Secondary sorting key
        UnsignedLong second_;
        // Pointer to the data
        const NumericalSampleImplementation & nsi_;
        // Sorting permutation
        NumericalSampleImplementation::UnsignedLongCollection permutation_;

        Comparison(const UnsignedLong first,
                   const NumericalSampleImplementation & nsi)
          : first_(first), second_(first), nsi_(nsi), permutation_(0)
        {
          const UnsignedLong size = nsi_.getSize();
          for (UnsignedLong i = 0; i < size; ++i) permutation_.add(i);
        }

        Comparison(const UnsignedLong first,
                   const UnsignedLong second,
                   const NumericalSampleImplementation & nsi)
          : first_(first), second_(second), nsi_(nsi), permutation_(0)
        {
          const UnsignedLong size = nsi_.getSize();
          for (UnsignedLong i = 0; i < size; ++i) permutation_.add(i);
        }

        Bool operator() (const UnsignedLong i, const UnsignedLong j) const
        {
          const NumericalScalar xI(nsi_[ permutation_[i] ][ first_  ]);
          const NumericalScalar xJ(nsi_[ permutation_[j] ][ first_  ]);
          const NumericalScalar yI(nsi_[ permutation_[i] ][ second_ ]);
          const NumericalScalar yJ(nsi_[ permutation_[j] ][ second_ ]);
          return ( (xI < xJ) || ((xI == xJ) && (yI < yJ)) );
        }
      }; // struct Comparison

      /* Ranked sample */
      NumericalSampleImplementation NumericalSampleImplementation::rank() const
      {
        NumericalSampleImplementation rankedSample(size_, dimension_);

        PairCollectionCollection sortedMarginalSamples(dimension_, PairCollection(size_));

        // Sort and rank all the marginal samples
        for (UnsignedLong i = 0; i < dimension_; ++i) {
          for (UnsignedLong j = 0; j < size_; ++j) {
            sortedMarginalSamples[i][j].value_ = (*this)[j][i];
            sortedMarginalSamples[i][j].index_ = j;
          }
          // sort
          TBB::ParallelSort(sortedMarginalSamples[i].begin(), sortedMarginalSamples[i].end());
          // rank
          for (UnsignedLong j = 0; j < size_; ++j)
            rankedSample[ sortedMarginalSamples[i][j].index_ ][i] = j;
        }

        return rankedSample;
      }

      /* Ranked component */
      NumericalSampleImplementation NumericalSampleImplementation::rank(const UnsignedLong index) const
      {
        if (index >= dimension_) throw OutOfBoundException(HERE) << "The requested index is too large, index=" << index << ", dimension=" << dimension_;

        NumericalSampleImplementation rankedSample(size_, 1);

        PairCollectionCollection sortedMarginalSamples(1, PairCollection(size_));

        // Sort and rank the marginal sample number index
        for (UnsignedLong j = 0; j < size_; ++j) {
          sortedMarginalSamples[0][j].value_ = (*this)[j][index];
          sortedMarginalSamples[0][j].index_ = j;
        }
        // sort
        TBB::ParallelSort(sortedMarginalSamples[0].begin(), sortedMarginalSamples[0].end());
        // rank
        for (UnsignedLong j = 0; j < size_; ++j)
          rankedSample[ sortedMarginalSamples[0][j].index_ ][0] = j;

        return rankedSample;
      }

      /* Sorted sample, component by component */
      NumericalSampleImplementation NumericalSampleImplementation::sort() const
      {
        NumericalSampleImplementation sortedSample(size_, dimension_);
        Type::Collection<NumericalScalar> component(size_);

        // Sort all the marginal samples
        for (UnsignedLong i = 0; i < dimension_; ++i) {
          for (UnsignedLong j = 0; j < size_; ++j)
            component[j] = (*this)[j][i];
          // sort
          TBB::ParallelSort(component.begin(), component.end());

          // copy
          for (UnsignedLong j = 0; j < size_; ++j)
            sortedSample[j][i] = component[j];
        } // loop over dimension

        return sortedSample;
      }

      /* Sorted sample, one component */
      NumericalSampleImplementation NumericalSampleImplementation::sort(const UnsignedLong index) const
      {
        if (index >= getDimension()) throw OutOfBoundException(HERE) << "The requested index is too large, index=" << index << ", dimension=" << getDimension();

        NumericalSampleImplementation sortedSample(size_, 1);
        Type::Collection<NumericalScalar> component(size_);

        // Sort the requested component
        for (UnsignedLong j = 0; j < size_; ++j)
          component[j] = (*this)[j][index];

        // sort
        TBB::ParallelSort(component.begin(), component.end());

        // copy
        for (UnsignedLong j = 0; j < size_; ++j)
          sortedSample[j][0] = component[j];

        return sortedSample;
      }

      /* Sorted according a component */
      NumericalSampleImplementation NumericalSampleImplementation::sortAccordingToAComponent(const UnsignedLong index) const
      {
        NumericalSampleImplementation rankedIndex(rank(index));
        NumericalSampleImplementation result(size_, dimension_);
        for (UnsignedLong i = 0; i < size_; ++i)
          result[static_cast<UnsignedLong>( round(rankedIndex[i][0]) ) ] = (*this)[i];

        return result;
      }

      /*
       * Gives the Spearman correlation matrix of the sample
       */
      NumericalSampleImplementation::CorrelationMatrix NumericalSampleImplementation::computeSpearmanCorrelation() const
      {
        return rank().computePearsonCorrelation();
      }

      /*
       * Gives the Kendall tau matrix of the sample
       * The correction for ties should be made according to http://www.statsdirect.com/help/nonparametric_methods/kend.htm
       */

      /* Private merge sort for the fast computation of Kendall's tau */
      uint64_t NumericalSampleImplementation::mergeSort(const UnsignedLong offset,
                                                        const UnsignedLong length,
                                                        const UnsignedLong activeDimension,
                                                        UnsignedLongCollection & ordering,
                                                        UnsignedLongCollection & buffer) const
      {
        uint64_t exchanges(0);
        // Array of length 1, no swap
        if (length == 1) return 0;

        if (length == 2) {
          const UnsignedLong index1(ordering[offset]);
          const UnsignedLong index2(ordering[offset + 1]);
          // Swap?
          const NumericalScalar y1((*this)[index1][activeDimension]);
          const NumericalScalar y2((*this)[index2][activeDimension]);
          if (y1 <= y2) return 0;
          ordering[offset] = index2;
          ordering[offset + 1] = index1;
          return 1;
        }

        const UnsignedLong length0(length / 2);
        const UnsignedLong length1(length - length0);
        const UnsignedLong middle(offset + length0);

        // Recursive sort of left and right parts
        exchanges += mergeSort(offset, length0, activeDimension, ordering, buffer);
        exchanges += mergeSort(middle, length1, activeDimension, ordering, buffer);
        const UnsignedLong index1(ordering[middle - 1]);
        const UnsignedLong index2(ordering[middle]);
        const NumericalScalar y1((*this)[index1][activeDimension]);
        const NumericalScalar y2((*this)[index2][activeDimension]);
        if (y1 < y2) return exchanges;
        UnsignedLong i(0);
        UnsignedLong j(0);
        UnsignedLong k(0);

        // Merge the left and right arrays
        while ((j < length0) || (k < length1)) {
          int d(0);
          const UnsignedLong index1(ordering[offset + j]);
          const UnsignedLong index2(ordering[middle + k]);
          if ( (k >= length1) ||
               ( (j < length0) &&
                 ((*this)[index1][activeDimension] <= (*this)[index2][activeDimension]) )) {
            buffer[i] = ordering[offset + j];
            d = i - j;
            ++j;

          } else {
            buffer[i] = ordering[middle + k];
            d = (offset + i) - (middle + k);
            ++k;
          }

          if (d > 0) exchanges += d;
          ++i;
        } // merge of the arrays

        for (UnsignedLong m = 0; m < length; ++m)
          ordering[offset + m] = buffer[m];

        return exchanges;
      }

      /* Lexical ordering between two dimensions: ranking with respect to i then to j if ties */
      NumericalSampleImplementation::UnsignedLongCollection NumericalSampleImplementation::lexicalRank(const UnsignedLong i,
                                                                                                       const UnsignedLong j) const
      {
        // Sort and rank all the marginal samples
        Comparison comparison(i, j, *this);
        std::stable_sort(comparison.permutation_.begin(), comparison.permutation_.end(), comparison);
        return comparison.permutation_;
      }

      /*
       * Gives the Kendall tau matrix of the sample, including ties correction
       */
      NumericalSampleImplementation::CorrelationMatrix NumericalSampleImplementation::computeKendallTau() const
      {
        CorrelationMatrix tau(dimension_);
        for (UnsignedLong i = 0; i < dimension_ - 1; ++i) {
          Comparison comparison(i, i, *this);
          std::stable_sort(comparison.permutation_.begin(), comparison.permutation_.end(), comparison);
          // Here, all the pairs (i, j) are sorted according to X_i
          Bool hasTies = false;
          for (UnsignedLong k = 1; !hasTies && (k < size_); ++k) {
            if ((*this)[ comparison.permutation_[k] ][i] == (*this)[ comparison.permutation_[k - 1] ][i])
              hasTies = true;
          }

          for (UnsignedLong j = i + 1; j < dimension_; ++j) {
            // If there are ties according to X_i, solve the ties by a lexical sorting of (X_i, X_j)
            UnsignedLongCollection ordering;
            if (hasTies) ordering = lexicalRank(i, j);
            else ordering = comparison.permutation_;

            // Check for ties
            // joint ties
            UnsignedLong first(0);
            NumericalScalar t(0.0);
            for (UnsignedLong k = 1; k < size_; ++k) {
              if ( ( (*this)[ordering[first]][i] != (*this)[ordering[k]][i] ) ||
                   ( (*this)[ordering[first]][j] != (*this)[ordering[k]][j] ) ) {
                t += (NumericalScalar(k - first) * NumericalScalar(k - first - 1)) / 2;
                first = k;
              }
            } // ties in x and y

            t += (NumericalScalar(size_ - first) * NumericalScalar(size_ - first - 1)) / 2;

            // ties in x
            first = 0;
            NumericalScalar u(0.0);
            for (UnsignedLong k = 1; k < size_; ++k) {
              if ((*this)[ordering[first]][i] != (*this)[ordering[k]][i]) {
                u += (NumericalScalar(k - first) * NumericalScalar(k - first - 1)) / 2;
                first = k;
              }
            } // ties in x

            u += (NumericalScalar(size_ - first) * NumericalScalar(size_ - first - 1)) / 2;

            // Initialize the working array of the merge sort
            UnsignedLongCollection buffer(size_);
            for (UnsignedLong k = 0; k < size_; ++k) buffer[k] = k;

            // swaps
            // It is casted into a NumericalScalar as it will be used in an expression using real values
            const NumericalScalar exchanges(mergeSort(0, size_, j, ordering, buffer));

            // ties in y after mergesort with counting
            first = 0;
            NumericalScalar v(0.0);
            for (UnsignedLong k = 1; k < size_; ++k) {
              if ( (*this)[ordering[first]][j] != (*this)[ordering[k]][j] ) {
                v += (NumericalScalar(k - first) * NumericalScalar(k - first - 1)) / 2;
                first = k;
              }
            } // ties in y after mergesort and counting

            v += (size_ - first) * (size_ - first - 1) / 2.;
            const NumericalScalar total = size_ * (size_ - 1) / 2.;
            if ((total == u) && (total == v)) tau(i, j) = 1.0;
            else tau(i, j) = ((total - (v + u - t)) - 2.0 * exchanges) / (sqrt((total - u) * (total - v)));
          } // Loop over j
        } // Loop over i

        return tau;
      }

      /*
       * Gives the range of the sample (by component)
       */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::computeRangePerComponent() const
      {
        return getMax() - getMin();
      }

      /*
       * Gives the median of the sample (by component)
       */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::computeMedianPerComponent() const
      {
        return computeQuantilePerComponent(0.5);
      }

      struct VariancePerComponentPolicy
      {
        typedef Type::NumericalPoint value_type;

        const value_type & mean_;
        const UnsignedLong dimension_;

        VariancePerComponentPolicy( const value_type & mean)
          : mean_(mean), dimension_(mean_.getDimension()) {}

        static inline value_type GetInvariant(const NumericalSampleImplementation & nsi)
        { return value_type(nsi.getDimension(), 0.0); }

        inline value_type & inplace_op( value_type & var, NSI_const_point point ) const
        {
          const value_type centeredRealization(point - mean_);
          for (UnsignedLong i = 0; i < dimension_; ++i)
            var[i] += centeredRealization[i] * centeredRealization[i];
          return var;
        }

        static inline value_type & inplace_op( value_type & var, const value_type & point )
        { return var += point; }

      }; /* end struct VariancePerComponentPolicy */

      /*
       * Gives the variance of the sample (by component)
       */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::computeVariancePerComponent() const
      {
        // Special case for a sample of size 1
        if (size_ == 1) return NumericalPoint(dimension_, 0.0);
        const NumericalPoint mean( computeMean() );

        ReductionFunctor<VariancePerComponentPolicy> functor( *this, mean );
        TBB::ParallelReduce( 0, size_, functor );
        functor.accumulator_ *= 1.0 / (size_ - 1);
        return functor.accumulator_;
      }

      /*
       * Gives the skewness of the sample (by component)
       */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::computeSkewnessPerComponent() const
      {
        // Special case for a sample of size 1
        if (size_ == 1) return NumericalPoint(dimension_, 0.0);

        const NumericalPoint mean(computeMean());
        NumericalPoint skewness(dimension_);
        NumericalPoint var(dimension_);

        for (UnsignedLong index = 0; index < size_; ++index) {
          NumericalPoint centeredRealization(operator[](index) - mean);
          for (UnsignedLong i = 0; i < dimension_; ++i) {
            // var = sum (Xi - Xmean)^2
            const NumericalScalar square(centeredRealization[i] * centeredRealization[i]);
            var[i] += square;
            // skewness = sum (Xi - Xmean)^3
            skewness[i] += square * centeredRealization[i];
          }
        }

        const NumericalScalar factor(1.0 / size_);
        const NumericalScalar factor1(1.0 / (size_ - 1));
        for (UnsignedLong i = 0; i < dimension_; ++i) {
          // skewness = 1 / size sum (Xi - Xmean)^3 / (var/(size-1))^3/2
          skewness[i] *= factor * pow(var[i] * factor1, -1.5);
        }
        return skewness;
      }

      /*
       * Gives the kurtosis of the sample (by component)
       */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::computeKurtosisPerComponent() const
      {
        // Special case for a sample of size 1
        if (size_ == 1) return NumericalPoint(dimension_, 0.0);

        const NumericalPoint mean(computeMean());
        NumericalPoint kurtosis(dimension_);
        NumericalPoint var(dimension_);

        for (UnsignedLong index = 0; index < size_; ++index) {
          const NumericalPoint centeredRealization(operator[](index) - mean);
          for (UnsignedLong i = 0; i < dimension_; ++i) {
            // var = sum (Xi - Xmean)^2
            const NumericalScalar square(centeredRealization[i] * centeredRealization[i]);
            var[i] += square;
            // kurtosis = sum (Xi - Xmean)^4
            kurtosis[i] += square * square;
          }
        }

        const NumericalScalar factor(1.0 / size_);
        const NumericalScalar factor1(1.0 / (size_ - 1));
        for (UnsignedLong i = 0; i < dimension_; ++i) {
          // kurtosis = 1 / size sum (Xi - Xmean)^4 / (var/(size-1))^2
          kurtosis[i] *= factor * pow(var[i] * factor1, -2);
        }

        return kurtosis;
      }

      /*
       * Gives the centered moment of order k of the sample (by component)
       */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::computeCenteredMomentPerComponent(const UnsignedLong k) const
      {
        if (size_ == 0) throw InvalidArgumentException(HERE) << "Cannot compute centered moments on an empty sample";

        // Special case: order 0, return (1,...,1)
        if (k == 0) return NumericalPoint(dimension_, 1.0);
        // Special case: order 1, return (0,...,0)
        if (k == 1) return NumericalPoint(dimension_, 0.0);

        // General case
        const NumericalPoint mean(computeMean());
        NumericalPoint moment(dimension_);
        for (UnsignedLong index = 0; index < size_; ++index) {
          const NumericalPoint centeredRealization(operator[](index) - mean);
          for (UnsignedLong i = 0; i < dimension_; ++i) {
            // moment[i] = sum Xc^k, the normalization is done later
            moment[i] += pow(centeredRealization[i], k);
          }
        }

        // Normalization by 1 / size
        return moment * (1.0 / size_);
      }

      /*
       * Gives the quantile per component of the sample
       */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::computeQuantilePerComponent(const NumericalScalar prob) const
      {
        // Special case for prob >= 1
        if (prob >= 1.0) return getMax();
        // Special case for prob <= 0.0
        if (prob <= 0.0) return getMin();

        const NumericalScalar scalarIndex(prob * size_);
        const UnsignedLong index(static_cast<UnsignedLong>( floor( scalarIndex ) ));
        const NumericalScalar beta(scalarIndex - index);
        const NumericalScalar alpha(1.0 - beta);
        NumericalPoint quantile(dimension_);
        NumericalPoint component(size_);
        for (UnsignedLong j = 0; j < dimension_; ++j) {
          for (UnsignedLong i = 0; i < size_; ++i)
            component[i] = operator[](i)[j];

          TBB::ParallelSort(component.begin(), component.end());

          if (index == size_ - 1) quantile[j] = component[index];
          else
            // Interpolation between the two adjacent empirical quantiles
            quantile[j] = alpha * component[index] + beta * component[index + 1];
        } // end for

        return quantile;
      }

      /*
       * Gives the N-dimension quantile of the sample
       */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::computeQuantile(const NumericalScalar prob) const
      {
        if (getDimension() == 1) return computeQuantilePerComponent(prob);
        throw NotYetImplementedException(HERE);
      }

      struct CDFPolicy
      {
        typedef NumericalScalar value_type;

        const Type::NumericalPoint & point_;
        const Bool tail_;
        const UnsignedLong dimension_;
        const NumericalScalar p_;

        CDFPolicy( const NumericalSampleImplementation & nsi, const Type::NumericalPoint & point, const Bool tail )
          : point_(point), tail_(tail), dimension_(nsi.getDimension()), p_(1.0 / nsi.getSize()) {}

        static inline value_type GetInvariant(const NumericalSampleImplementation &)
        { return value_type(0.0); }

        inline value_type & inplace_op( value_type & a, const value_type & other ) const
        { return a += other; }

        template <typename T>
        inline value_type & inplace_op( value_type & a, const T & pt ) const
        {
          UnsignedLong j = 0;
          while ( (j < dimension_) && (tail_ ^ (pt[j] <= point_[j])) ) ++j;
          if (j == dimension_) a += p_;
          return a;
        }

      }; /* end struct AddPolicy */

      /*
       * Get the empirical CDF of the sample
       */
      NumericalScalar NumericalSampleImplementation::computeEmpiricalCDF(const NumericalPoint & point,
                                                                         const Bool tail) const
      {
        ReductionFunctor<CDFPolicy> functor( *this, CDFPolicy( *this, point, tail ) );
        TBB::ParallelReduce( 0, size_, functor );
        return functor.accumulator_;
      }

      struct MaxPerComponentPolicy
      {
        typedef Type::NumericalPoint value_type;

        static inline value_type GetInvariant(const NumericalSampleImplementation & nsi)
        { return value_type(nsi.getDimension(), - std::numeric_limits<NumericalScalar>::max()); }

        template <typename T>
        static inline value_type & inplace_op( value_type & a, const T & b)
        {
          const UnsignedLong dim = a.getDimension();
          for (UnsignedLong j = 0; j < dim; ++j) a[j] = std::max( a[j], b[j] );
          return a;
        }
      }; /* end struct MaxPerComponentPolicy */

      struct MinPerComponentPolicy
      {
        typedef Type::NumericalPoint value_type;

        static inline value_type GetInvariant(const NumericalSampleImplementation & nsi)
        { return value_type(nsi.getDimension(), std::numeric_limits<NumericalScalar>::max()); }

        template <typename T>
        static inline value_type & inplace_op( value_type & a, const T & b)
        {
          const UnsignedLong dim = a.getDimension();
          for (UnsignedLong j = 0; j < dim; ++j) a[j] = std::min( a[j], b[j] );
          return a;
        }
      }; /* end struct MinPerComponentPolicy */


      /* Maximum accessor */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::getMax() const
      {
        if (size_ == 0) throw InternalException(HERE) << "Impossible to get the maximum of an empty NumericalSample";

        ReductionFunctor<MaxPerComponentPolicy> functor( *this, MaxPerComponentPolicy() );
        functor.accumulator_ = operator[](0);
        TBB::ParallelReduce( 1, size_, functor );
        return functor.accumulator_;
      }

      /* Minimum accessor */
      NumericalSampleImplementation::NumericalPoint NumericalSampleImplementation::getMin() const
      {
        if (size_ == 0) throw InternalException(HERE) << "Impossible to get the maximum of an empty NumericalSample";

        ReductionFunctor<MinPerComponentPolicy> functor( *this, MinPerComponentPolicy() );
        functor.accumulator_ = operator[](0);
        TBB::ParallelReduce( 1, size_, functor );
        return functor.accumulator_;
      }


      struct TranslationPolicy
      {
        const Type::NumericalPoint & translation_;
        const UnsignedLong dimension_;

        TranslationPolicy( const Type::NumericalPoint & translation)
          : translation_(translation), dimension_(translation_.getDimension()) {}

        inline void inplace_op( NSI_point point ) const
        { for (UnsignedLong j = 0; j < dimension_; ++j) point[j] += translation_[j]; }

      }; /* end struct TranslationPolicy */

      /*
       * Translate realizations in-place
       */
      void NumericalSampleImplementation::translate(const NumericalPoint & translation)
      {
        if (dimension_ != translation.getDimension())
          throw InvalidArgumentException(HERE) << "Translation point has incorrect dimension. Got " << translation.getDimension()
                                               << ". Expected " << dimension_;

        ParallelFunctor<TranslationPolicy> functor( *this, translation );
        TBB::ParallelFor( 0, size_, functor );
      }

      /* Get the i-th marginal distribution */
      NumericalSampleImplementation NumericalSampleImplementation::getMarginal(const UnsignedLong index) const /* throw(InvalidArgumentException) */
      {
        if (index >= dimension_) throw InvalidArgumentException(HERE) << "The index of a marginal sample must be in the range [0, dim-1]";

        // Special case for dimension 1
        if (dimension_ == 1) return *this;

        // General case
        NumericalSampleImplementation marginalSample(size_, 1);
        const Description description(getDescription());

        // If the sample has a description, extract the marginal description
        if (description.getSize() == dimension_)
          marginalSample.setDescription(Description(1, getDescription()[index]));

        for (UnsignedLong i = 0; i < size_; ++i)
          marginalSample[i][0] = operator[](i)[index];

        return marginalSample;
      }

      /* Get the distribution of the marginal distribution corresponding to indices dimensions */
      NumericalSampleImplementation NumericalSampleImplementation::getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */
      {
        if (!indices.check(dimension_ - 1)) throw InvalidArgumentException(HERE) << "The indices of a marginal sample must be in the range [0, dim-1] and  must be different";

        // Special case for dimension 1
        if (dimension_ == 1) return *this;

        // General case
        const UnsignedLong outputDimension(indices.getSize());
        NumericalSampleImplementation marginalSample(size_, outputDimension);
        const Description description(getDescription());

        // If the sample has a description, extract the marginal description
        if (description.getSize() == dimension_) {
          Description marginalDescription(outputDimension);
          for (UnsignedLong i = 0; i < outputDimension; ++i)
            marginalDescription[i] = description[indices[i]];
          marginalSample.setDescription(marginalDescription);
        }

        for (UnsignedLong i = 0; i < size_; ++i) {
          for (UnsignedLong j = 0; j < outputDimension; ++j) {
            // We access directly to the component of the NumericalPoint for performance reason
            marginalSample[i][j] = operator[](i)[indices[j]];
          }
        }

        return marginalSample;
      }


      struct ScalingPolicy
      {
        const Type::NumericalPoint & scale_;
        const UnsignedLong dimension_;

        ScalingPolicy( const Type::NumericalPoint & scale) : scale_(scale), dimension_(scale_.getDimension()) {}

        inline void inplace_op( NSI_point point ) const
        { for (UnsignedLong j = 0; j < dimension_; ++j) point[j] *= scale_[j]; }

      }; /* end struct ScalingPolicy */

      /*
       * Scale realizations componentwise in-place
       */
      void NumericalSampleImplementation::scale(const NumericalPoint & scaling)
      {
        if (dimension_ != scaling.getDimension())
          throw InvalidArgumentException(HERE) << "Scaling point has incorrect dimension. Got " << scaling.getDimension()
                                               << ". Expected " << dimension_;

        ParallelFunctor<ScalingPolicy> functor( *this, scaling );
        TBB::ParallelFor( 0, size_, functor );
      }

      /* Save to CSV file */
      void NumericalSampleImplementation::exportToCSVFile(const FileName & filename,
                                                          const Bool withDescription) const
      {
        const String csvSeparator(ResourceMap::Get( "csv-file-separator" ));

        std::ofstream csvFile(filename.c_str());
	csvFile.imbue(std::locale("C"));
        csvFile.precision(16);
        // Export the description
        if (withDescription) {
          const Description description(getDescription());
          const UnsignedLong descriptionSize(description.getSize());

          // Write the description if any
          if (descriptionSize > 0) {
            String separator;
            for (UnsignedLong i = 0; i < descriptionSize; ++i, separator = csvSeparator)
              csvFile << separator << "\"" << description[i] << "\"";
            csvFile << Os::GetEndOfLine();
          }
        }

        // Write the data
        for(UnsignedLong i = 0; i < size_; ++i, csvFile << Os::GetEndOfLine()) {
          String separator;
          for(UnsignedLong j = 0; j < dimension_; ++j, separator = csvSeparator)
            csvFile << separator << std::scientific << operator[](i)[j];
        }

        // Close the file
        csvFile.close();
      }



      /* Method save() stores the object through the StorageManager */
      void NumericalSampleImplementation::save(StorageManager::Advocate & adv) const
      {
        Common::PersistentObject::save(adv);
        adv.saveAttribute( "size_", size_);
        adv.saveAttribute( "dimension_", dimension_);
        adv.saveAttribute( "data_", data_);
        if (!p_description_.isNull())
          adv.saveAttribute( "description_", *p_description_ );
      }


      /* Method load() reloads the object from the StorageManager */
      void NumericalSampleImplementation::load(StorageManager::Advocate & adv)
      {
        Common::PersistentObject::load(adv);
        adv.loadAttribute( "size_", size_);
        adv.loadAttribute( "dimension_", dimension_);
        adv.loadAttribute( "data_", data_);
        Description description;
        adv.loadAttribute( "description_", description );
        if (description.getSize() != 0) setDescription(description);
      }


    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */
