//                                               -*- C++ -*-
/**
 * @file  SensitivityAnalysis.hxx
 * @brief SensitivityAnalysis implements the sensitivity analysis methods based on Sobol coefficients
 *
 * (C) Copyright 2005-2011 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * @author $LastChangedBy:  $
 * @date   $LastChangedDate:  $
 */

#ifndef OPENTURNS_SENSITIVITYANALYSIS_HXX
#define OPENTURNS_SENSITIVITYANALYSIS_HXX

#include "OTprivate.hxx"
#include "NumericalSample.hxx"
#include "NumericalMathFunction.hxx"
#include "NumericalPoint.hxx"
#include "Description.hxx"
#include "NumericalPointWithDescription.hxx"
#include "Graph.hxx"

namespace OpenTURNS
{
  namespace Base
  {
    namespace Stat
    {

      /**
       * @class SensitivityAnalysis
       *
       */

      class SensitivityAnalysis
      {
      public:

        typedef Func::NumericalMathFunction         NumericalMathFunction;
        typedef Type::NumericalPoint                NumericalPoint;
        typedef Type::Description                   Description;
        typedef Type::NumericalPointWithDescription NumericalPointWithDescription;
        typedef Type::SymmetricMatrix               SymmetricMatrix;
        typedef Type::SymmetricTensor               SymmetricTensor;
        typedef Graph::Graph                        Graph;
        
        /** The model is evaluated by blocks to allow for parallelization */
        static const UnsignedLong DefaultBlockSize;

        /** Constructor with input samples and model */
        SensitivityAnalysis(const NumericalSample & inputSample1,
                            const NumericalSample & inputSample2,
                            const NumericalMathFunction & model);
        
        /** First order indices accessor */
        NumericalPoint getFirstOrderIndices(const UnsignedLong marginalIndex = 0) const;

        /** Second order indices accessor */
        SymmetricMatrix getSecondOrderIndices(const UnsignedLong marginalIndex = 0) const;
        
        /** Total order indices accessor */
        NumericalPoint getTotalOrderIndices(UnsignedLong marginalIndex = 0) const;
        
        /** Block size accessor */
        void setBlockSize(const UnsignedLong blockSize);
        UnsignedLong getBlockSize() const;
        
        /** ImportanceFactors graph */
        static Graph DrawImportanceFactors(const NumericalPointWithDescription & importanceFactors,
                                           const String & title = "Importance Factors");

        /** ImportanceFactors graph */
        static Graph DrawImportanceFactors(const NumericalPoint & values,
                                           const Description & names,
                                           const String & title = "Importance Factors");

      private:
        /** Compute all the Sobol indices */
        void computeSobolIndices(const UnsignedLong order) const;

        /** Two independently generated samples */
        NumericalSample inputSample1_;
        NumericalSample inputSample2_;
    
        /** The model */
        NumericalMathFunction model_;
        
        /** First order indices */
        mutable NumericalSample firstOrderIndice_;

        /** Total order indices */
        mutable NumericalSample totalOrderIndice_;
        
        /** Second order indices */
        mutable SymmetricTensor secondOrderIndice_;
        
        /** Size of blocks */
        UnsignedLong blockSize_;
        
        /** Store the maximum order already computed */ 
        mutable UnsignedLong alreadyComputedOrder_;

      }; /* class SensitivityAnalysis */

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */
#endif /* OPENTURNS_SENSITIVITYANALYSIS_HXX */
