//                                               -*- C++ -*-
/**
 *  @file  NumericalPoint.hxx
 *  @brief NumericalPoint implements the classical mathematical point
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-05-24 19:30:41 +0200 (Tue, 24 May 2011) $
 *  Id:      $Id: NumericalPoint.hxx 1910 2011-05-24 17:30:41Z schueller $
 */
#ifndef OPENTURNS_NUMERICALPOINT_HXX
#define OPENTURNS_NUMERICALPOINT_HXX

#include <vector>
#include "PersistentCollection.hxx"
#include "Description.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      /**
       * @class NumericalPoint
       *
       * NumericalPoint implements the classical mathematical point
       */

      class NumericalPoint
        : public PersistentCollection<NumericalScalar>
      {
        CLASSNAME;

      public:

        typedef Type::PersistentCollection<NumericalScalar> InternalType;
        typedef Type::Collection<NumericalScalar>           NumericalScalarCollection;
        typedef InternalType::iterator                      iterator;
        typedef InternalType::const_iterator                const_iterator;
        typedef InternalType::reverse_iterator              reverse_iterator;
        typedef InternalType::const_reverse_iterator        const_reverse_iterator;
        typedef Common::InvalidArgumentException            InvalidArgumentException;
        typedef Common::StorageManager                      StorageManager;
        typedef Common::PersistentObject                    PersistentObject;
        typedef NumericalPoint                              ImplementationType;

        /** Default constructor */
        NumericalPoint();

        /** Constructor with size */
        NumericalPoint(const UnsignedLong size,
                       const NumericalScalar value = 0.0);

        /** Constructor from a collection */
        NumericalPoint(const Type::Collection<NumericalScalar> & coll);

        /** Virtual constructor */
        virtual NumericalPoint * clone() const;

        /** Description Accessor */
        virtual void setDescription(const Description & description) {}
        virtual Description getDescription() const { return Description( getDimension() ); }

        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;

        /** Dimension accessor */
        inline UnsignedLong getDimension() const { return PersistentCollection<NumericalScalar>::getSize(); }
        inline UnsignedLong getSize() const { return PersistentCollection<NumericalScalar>::getSize(); }

        /** Collection accessor */
        inline const NumericalScalarCollection & getCollection() const { return *this; }

#ifndef SWIG
        /** Coordinate accessor */
        NumericalScalar & operator[](const UnsignedLong index);
        const NumericalScalar & operator[](const UnsignedLong index) const;

        /** Erase the elements between first and last */
        iterator erase(iterator first, iterator last);

        /** Erase the element pointed by position */
        iterator erase(iterator position);

        /** Erase the element pointed by position */
        iterator erase(UnsignedLong position);

        /** In-place addition operator */
        NumericalPoint & operator +=(const NumericalPoint & other) /* throw (InvalidArgumentException) */;

        /** In-place substraction operator */
        NumericalPoint & operator -=(const NumericalPoint & other) /* throw (InvalidArgumentException) */;
#endif

        /**  In-place product operator */
        NumericalPoint & operator *=(const NumericalScalar scalar);

        /**  In-place division operator */
        NumericalPoint & operator /=(const NumericalScalar scalar);

        /**  Norm */
        NumericalScalar norm() const;

        /**  Norm^2 */
        NumericalScalar norm2() const;

        /** Method save() stores the object through the StorageManager */
        virtual void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        virtual void load(StorageManager::Advocate & adv);

      private:

        //      /** The description of all components */
        //      Description:: p_description_;

      }; /* class NumericalPoint */


#ifndef SWIG
      /** Comparison operator */
      Bool operator == (const NumericalPoint & lhs,
                        const NumericalPoint & rhs);

      /** Ordering operator */
      Bool operator < (const NumericalPoint & lhs,
                       const NumericalPoint & rhs);

      /** Product operator */
      NumericalPoint operator * (const NumericalScalar scalar,
                                 const NumericalPoint & point);

      /** Product operator */
      NumericalPoint operator * (const NumericalPoint & point,
                                 const NumericalScalar scalar);

      /** Division operator */
      NumericalPoint operator / (const NumericalPoint & point,
                                 const NumericalScalar scalar);

      /** Addition operator */
      NumericalPoint operator + (const NumericalPoint & lhs,
                                 const NumericalPoint & rhs);

      /** Substraction operator */
      NumericalPoint operator - (const NumericalPoint & lhs,
                                 const NumericalPoint & rhs);

#endif

      /** Dot product operator */
      NumericalScalar dot(const NumericalPoint & lhs,
                          const NumericalPoint & rhs);



    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NUMERICALPOINT_HXX */
